# "TkMC" http://wiki.tcl.tk/7772
#
##############################################################################
#
#    TkMC clone of MC file manager.
#    Tested on MS Windows OS, it should be  work on any Unix systems
#    Author: Grigoriy Abramov
#    Date: January, 2002
#    comments, questions, bug reports send to:
#    gabramov@cpe.uchicago.edu
#    Packages:
#    Entry with History and File Properties dialog by Richard Suchenwirth
#    Multiple scrollbars by Jeffrey Hobbs
#
#    TkMC - MC like file manager with basic functionality.
#
#    With TkMC you can browse, view, edit and run TCL applications.
#
#    DISCLAIMER and License Issues
#
#    TkMC - is copyrighted by Grigoriy Abramov
#
#    This program is free software; you can redistribute it and/or modify it
#    without restriction.
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

package require Tk
package require tile
package require fsdialog

wm resizable . 0 0
wm title . "TkMC v1.0chw"
wm protocol . WM_DELETE_WINDOW exit

set parms(lbh)  16
set parms(txh)  16
set parms(edh)  24
set parms(wb)   9
set parms(tb1)  "Help   F1"
set parms(tb2)  "Clear  F2"
set parms(tb3)  "View   F3"
set parms(tb4)  "Edit   F4"
set parms(tb5)  "Copy   F5"
set parms(tb6)  "RenMov F6"
set parms(tb7)  "MkDir  F7"
set parms(tb8)  "Delete F8"
set parms(tb9)  "RunTcl F9"
set parms(tb10) "Exit  F10"

ttk::style config TkMCButton {*}[ttk::style config TButton]
ttk::style layout TkMCButton [ttk::style layout TButton]

if {[info command sdltk] eq "sdltk"} {
    if {[sdltk android]} {
	wm attributes . -fullscreen 1
	catch {borg screenorientation landscape}
	set parms(lbh)  8
	set parms(txh)  8
	set parms(edh)  16
	set parms(wb)   6
	set parms(tb1)  "Help"
	set parms(tb2)  "Clear"
	set parms(tb3)  "View"
	set parms(tb4)  "Edit"
	set parms(tb5)  "Copy"
	set parms(tb6)  "RenMov"
	set parms(tb7)  "MkDir"
	set parms(tb8)  "Delete"
	set parms(tb9)  "RunTcl"
	set parms(tb10) "Exit"
	# smaller font for listboxes ...
	font create TkListboxFont {*}[font configure TkDefaultFont]
	font configure TkListboxFont -size \
	    [expr {[font configure TkListboxFont -size] - 1}]
	option add *Listbox.font TkListboxFont
	# .. and for buttons on bottom
	ttk::style config TkMCButton -font TkListboxFont
    } else {
	option add *Entry.highlightThickness 1
	option add *Listbox.highlightThickness 1
	option add *Text.highlightThickness 1
    }
    option add *Entry.highlightBackground "#d9d9d9"
    option add *Entry.highlightColor "#000000"
    option add *Entry.background "#d9d9d9"
    option add *Listbox.highlightBackground "#d9d9d9"
    option add *Listbox.highlightColor "#000000"
    option add *Listbox.background "#d9d9d9"
    option add *Text.highlightBackground "#d9d9d9"
    option add *Text.highlightColor "#000000"
}

#Disable default Error messages
proc bgerror {errmsg} {
    puts stderr $::errorInfo
    set lmsg [split $errmsg "\n"]
    if {[llength $lmsg] > 8} {
	set errmsg [join [lrange $lmsg 0 7] "\n"]
	append errmsg "\n..."
    }
    set msg [format "Error: %s" $errmsg $::errorCode]
    set parent [focus]
    if {$parent ne ""} {
	set parent [winfo toplevel $parent]
    } else {
	set parent .
    }
    tk_messageBox -parent $parent -title  "Error" -type ok \
	-icon error -message $msg
}

#Format file size
proc fmtfsize {size} {
    if {$size < 0} {
	set size [expr {0 - $size}]
    }
    if {$size > 1024*1024*1024} {
	set size [format "%2.2f GB" [expr {($size/1024.0)/1024.0/1024.0}]]
    } elseif { $size > 1024*1024} {
	set size [format "%2.2f MB" [expr {($size/1024.0)/1024.0}]]
    } elseif {$size > 1024*100} {
	set size [format "%2.2f kB" [expr {$size/1024.0}]]
    }
    return $size
}

#Format date and time
proc date_time {{when {}} {withsec 0}} {
    if {$when eq ""} {
	set when [clock seconds]
    }
    if {$withsec} {
	return [clock format $when -format {%Y-%m-%d %H:%M:%S}]
    }
    return [clock format $when -format {%Y-%m-%d %H:%M}]
}

#Conditionally focus on widget given toplevel
proc focus_if {top w} {
    set focus [focus]
    if {$focus eq ""} {
	return
    }
    if {[winfo toplevel $focus] eq $top} {
	focus $w
    }
}

#Entry history
namespace eval history {
    proc add? {w} {
	variable $w
	variable n$w
	upvar 0 $w hist
	set s [set ::[$w cget -textvariable]]
	if {$s eq ""} return
	if {$s ne [lindex $hist end]} {
	    lappend hist $s
	    set n$w [llength $hist]
	}
    }
    proc move {w where} {
	variable $w
	variable n$w
	upvar 0 $w hist
	incr n$w $where
	if {[set n$w]<0} {set n$w 0}
	if {[set n$w]>=[llength $hist]+1} {
	    set n$w [llength $hist]
	}
	set ::[$w cget -textvar] [lindex $hist [set n$w]]
	$w icursor end
    }
    proc for {type name args} {
	switch -- $type {
	    entry {
		uplevel \#0 $type $name $args
		bind $name <Up> {history::move %W -1}
		bind $name <Down> {history::move %W 1}
		bind $name <Next> {history::move %W 99999}
		bind $name <Return> {history::add? %W}
		variable $name {}
		variable n$name 0
	    }
	    default {error "usage: history::for entry <w> <args>"}
	}
    }
}

proc vol_to_widget {wbase vol} {
    regsub -all {[:/\\@.]} $vol {@} vol
    return "${wbase}${vol}"
}

proc abbrev_vol {vol} {
    set vol [string trimright [file native $vol] "\\"]
    if {[string length $vol] > 20} {
	if {$::tcl_platform(platform) eq "windows"} {
	    set drv [lindex [split $vol :] 0]
	    set vol "${drv}:\\\u2026\\[lindex [split $vol \\] end]"
	} else {
	    set vol "/\u2026/[lindex [split $vol /] end]"
	}
    }
    return $vol
}

proc fix_droid_root {} {
    set files {}
    if {([info command sdltk] eq "sdltk") && [sdltk android]} {
	if {![file readable /]} {
	    foreach dir {/assets /data /mnt /sdcard /storage} {
		if {[file isdirectory $dir]} {
		    lappend files [file tail $dir]
		}
	    }
	}
    }
    return $files
}

panedwindow .pane -relief flat -bd 0 -orient vert -opaqueresize true \
    -showhandle false
frame .pane_t -relief flat -bd 0
frame .pane_b -relief flat -bd 0
.pane add .pane_t -minsize 120 -sticky nswe -stretch always
.pane add .pane_b -minsize 120 -sticky nswe -stretch always

frame .top
frame .top.f
pack .top.f -fill x -expand true -side left
foreach x [file volume] {
    set w [vol_to_widget .top.f.f $x]
    button $w -text [abbrev_vol $x] -takefocus 0 -command [subst {
	read_dir_l [list $x]
	refresh_left
    }]
    pack $w -side left -padx 1 -pady 1
    unset w
}
if {[info command sdltk] eq "sdltk"} {
    if {[sdltk android]} {
	set x "~"
	set w [vol_to_widget .top.f.f $x]
	button $w -text $x -takefocus 0 -command [subst {
	    read_dir_l [list $x]
	    refresh_left
	}]
	pack $w -side left -padx 1 -pady 1
	unset w
    }
}
unset x

#Frame and entry areas for directory names
frame .top2
entry .top2.e_l -bg grey80 -cursor arrow -takefocus 0
entry .top2.e_r -bg grey80 -cursor arrow -takefocus 0
pack  .top2.e_l .top2.e_r -side left -fill x -expand true

#Disable focus on these entries
bind .top2.e_l <1> break
bind .top2.e_l <<PasteSelection>> break
bind .top2.e_r <1> break
bind .top2.e_r <<PasteSelection>> break

#Fake entry (for some reason it should be exists to separate
#readings from different directories
entry .fake_bot
entry .fake_ed
entry .fake_l
entry .fake_r
entry .fake_excd

#Buttons, entry, text, scrollbar for text on bottom
frame .bot
frame .bot2

pack .bot2 -in .pane_t -fill x -side bottom
entry .bot2.e_l -bg grey80 -takefocus 0 -cursor arrow
entry .bot2.e_r -bg grey80 -takefocus 0 -cursor arrow
pack .bot2.e_l .bot2.e_r -side left -fill x -expand true
text .bot.t -wrap none -height $parms(txh) -bg white -fg blue -cursor arrow \
    -yscrollcommand [list .bot.scr set]
catch {.bot.t configure -undo 1 -maxundo 500 -autoseparators 1}
ttk::scrollbar .bot.scr -orient vert -command [list .bot.t yview]
history::for entry .bot.comm -bd 1 -cursor arrow -textvariable command_line
pack .bot.comm -side bottom -fill x
pack .bot.scr -side right -fill y
pack .bot.t -side top -fill both -expand true

bind .bot.comm <Return> {+ ;
    .bot.t delete 1.0 end ;
    catch {.bot.t edit reset} ;
    .bot.t insert end [exec_cmd [.bot.comm get]] ;
    catch {.bot.t edit reset} ;
    .bot.comm delete 0 end ;
    focus_if . .bot.comm
}

bind .bot.comm <Enter> {focus_if . .bot.comm}

#Swap <Control-Tab> and <Tab> bindings in main text widget
bind .bot.t <Control-Tab> [bind Text <Tab>]
bind .bot.t <Control-Tab> {+ ; break}
bind .bot.t <Tab> [bind Text <Control-Tab>]
bind .bot.t <Tab> {+ ; break}

#Disable focus on these entries
bind .bot2.e_l <1> break
bind .bot2.e_l <<PasteSelection>> break
bind .bot2.e_r <1> break
bind .bot2.e_r <<PasteSelection>> break

frame .but
pack .but -side bottom -fill x
foreach but {.b1 .b2 .b3 .b4 .b5 .b6 .b7 .b8 .b9 .b10} {
    ttk::button .but$but -takefocus 0 -width 9 -style TkMCButton
    pack .but$but -side left -fill x -expand true -padx 1 -pady 1
}
unset but

.but.b1 config -text $parms(tb1) -command about
.but.b2 config -text $parms(tb2) -command {
    .bot.t delete 1.0 end
    catch {.bot.t edit reset}
}
.but.b3 config -text $parms(tb3) -command {}
.but.b4 config -text $parms(tb4) -command editor
.but.b5 config -text $parms(tb5) -command {}
.but.b6 config -text $parms(tb6) -command {}
.but.b7 config -text $parms(tb7) -command {}
.but.b8 config -text $parms(tb8) -command {}
.but.b9 config -text $parms(tb9) -command {
    focus_if . .bot.t
    run_tcl [.fake_bot get] .bot.t
}
.but.b10 config -text $parms(tb10) -command exit

pack .bot -in .pane_b -side bottom -fill both -expand true

#Right drive lists
frame .top.f_r
pack .top.f_r -fill x -expand true -side left

foreach x [file volume] {
    set w [vol_to_widget .top.f_r.f $x]
    button $w -text [abbrev_vol $x] -takefocus 0 -command [subst {
	read_dir_r [list $x]
	refresh_right
    }]
    pack $w -side left -padx 1 -pady 1
    unset w
}
if {[info command sdltk] eq "sdltk"} {
    if {[sdltk android]} {
	set x "~"
	set w [vol_to_widget .top.f_r.f $x]
	button $w -text $x -takefocus 0 -command [subst {
	    read_dir_r [list $x]
	    refresh_right
	}]
	pack $w -side left -padx 1 -pady 1
	unset w
    }
}
unset x

#Frame and entry areas for directory names

pack .top -side top -fill x
pack .pane -side top -fill both -expand yes
pack .top2 -in .pane_t -side top -fill x

#Left listboxes
listbox .l -width 20 -height $parms(lbh) -bd 0 -export no \
    -selectmode browse -yscrollcommand lb_scroll
listbox .sz -width 8 -height $parms(lbh) -bd 0 -export no \
    -selectmode browse -takefocus 0 -yscrollcommand lb_scroll
listbox .time -width 16 -height $parms(lbh) -bd 0 -export no \
    -selectmode browse -takefocus 0 -yscrollcommand lb_scroll
pack .l .sz .time -in .pane_t -side left -fill both -expand true

#Override some bindings on left listboxes
bind .l <Prior> {%W yview scroll -1 pages ; break}
bind .l <Next> {%W yview scroll 1 pages ; break}
bind .l <<Copy>> {handle_lb_clip .fake_l %W ; break}
bind .sz <Prior> {%W yview scroll -1 pages ; break}
bind .sz <Next> {%W yview scroll 1 pages ; break}
bind .time <Prior> {%W yview scroll -1 pages ; break}
bind .time <Next> {%W yview scroll 1 pages ; break}
bind .l <2> break
bind .l <B2-Motion> break
bind .l <B1-Leave> break
bind .l <B1-Enter> break
bind .sz <2> break
bind .sz <B2-Motion> break
bind .sz <B1-Leave> break
bind .sz <B1-Enter> break
bind .time <2> break
bind .time <B2-Motion> break
bind .time <B1-Leave> break
bind .time <B1-Enter> break

#Right side listboxes
listbox .l_r -width 20 -height $parms(lbh) -bd 0 -export no \
    -selectmode browse -yscrollcommand lb_scroll_r
listbox .sz_r -width 8 -height $parms(lbh) -bd 0 -export no \
    -selectmode browse -takefocus 0 -yscrollcommand lb_scroll_r
listbox .time_r -width 16 -height $parms(lbh) -bd 0 -export no \
    -selectmode browse -takefocus 0 -yscrollcommand lb_scroll_r
pack .l_r .sz_r .time_r -in .pane_t -side left -fill both -expand true

#Override some bindings on left listboxes
bind .l_r <Prior> {%W yview scroll -1 pages ; break}
bind .l_r <Next> {%W yview scroll 1 pages ; break}
bind .l_r <<Copy>> {handle_lb_clip .fake_r %W ; break}
bind .sz_r <Prior> {%W yview scroll -1 pages ; break}
bind .sz_r <Next> {%W yview scroll 1 pages ; break}
bind .time_r <Prior> {%W yview scroll -1 pages ; break}
bind .time_r <Next> {%W yview scroll 1 pages ; break}
bind .l_r <2> break
bind .l_r <B2-Motion> break
bind .l_r <B1-Leave> break
bind .l_r <B1-Enter> break
bind .sz_r <2> break
bind .sz_r <B2-Motion> break
bind .sz_r <B1-Leave> break
bind .sz_r <B1-Enter> break
bind .time_r <2> break
bind .time_r <B2-Motion> break
bind .time_r <B1-Leave> break
bind .time_r <B1-Enter> break

#Fill all listboxes
proc read_dir_l {dir} {
    if {$::tcl_platform(platform) eq "windows"} {
	set files {}
	catch {
	    set files [glob -directory $dir -tails -nocomplain -- *]
	}
    } else {
	set files [fix_droid_root]
	catch {
	    set files [glob -directory $dir -tails -nocomplain -- * .*]
	}
    }
    #clean all listboxes
    .l delete 0 end
    .sz delete 0 end
    .time delete 0 end
    .l insert end ".."
    .sz insert end ".."
    .time insert end ".."
    set dirs {}
    set others {}
    set dsize {}
    set dtime {}
    set fsize {}
    set ftime {}
    foreach f [lsort -dictionary $files] {
	if {$f eq "." || $f eq ".."} continue
	set ff [file join $dir $f]
	if {![file isdirectory $ff] && ![file isfile $ff]} {
	    lappend others $f
	    lappend fsize "NOT A FILE"
	    lappend ftime "------"
	} elseif {[file isdirectory $ff]} {
	    lappend dirs $f
	    lappend dsize "DIR"
	    set mt 0
	    catch {set mt [file mtime $ff]}
	    lappend dtime [date_time $mt]
	} else {
	    lappend others $f
	    lappend fsize [fmtfsize [file size $ff]]
	    set mt 0
	    catch {set mt [file mtime $ff]}
	    lappend ftime [date_time $mt]
	}
    }
    foreach f [concat $dirs $others]  {
	.l insert end $f
    }
    foreach z [concat $dsize $fsize] {
	.sz insert end $z
    }
    foreach t [concat $dtime $ftime] {
	.time insert end $t
    }
    .top2.e_l delete 0 end
    .top2.e_l insert end [file native $dir]
    .fake_l delete 0 end
    .fake_l insert end $dir
    .fake_excd delete 0 end
    .fake_excd insert end $dir
}

#Fill right listboxes
proc read_dir_r {dir} {
    if {$::tcl_platform(platform) eq "windows"} {
	set files_r {}
	catch {
	    set files_r [glob -directory $dir -tails -nocomplain -- *]
	}
    } else {
	set files_r [fix_droid_root]
	catch {
	    set files_r [glob -directory $dir -tails -nocomplain -- * .*]
	}
    }
    #clean all listboxes
    .l_r delete 0 end
    .sz_r delete 0 end
    .time_r delete 0 end
    .l_r insert end ".."
    .sz_r insert end ".."
    .time_r insert end ".."
    set dirs_r {}
    set others_r {}
    set dsize_r {}
    set dtime_r {}
    set fsize_r {}
    set ftime_r {}
    foreach f [lsort -dictionary $files_r] {
	if {$f eq "." || $f eq ".."} continue
	set ff [file join $dir $f]
	if {![file isdirectory $ff] && ![file isfile $ff]} {
	    lappend others_r $f
	    lappend fsize_r "NOT A FILE"
	    lappend ftime_r "------"
	} elseif {[file isdirectory $ff]} {
	    lappend dirs_r $f
	    lappend dsize_r "DIR"
	    set mt 0
	    catch {set mt [file mtime $ff]}
	    lappend dtime_r [date_time $mt]
	} else {
	    lappend others_r $f
	    lappend fsize_r [fmtfsize [file size $ff]]
	    set mt 0
	    catch {set mt [file mtime $ff]}
	    lappend ftime_r [date_time $mt]
	}
    }
    foreach f [concat $dirs_r $others_r] {
	.l_r insert end $f
    }
    foreach z [concat $dsize_r $fsize_r] {
	.sz_r insert end $z
    }
    foreach t [concat $dtime_r $ftime_r] {
	.time_r insert end $t
    }
    .top2.e_r delete 0 end
    .top2.e_r insert end [file native $dir]
    .fake_r delete 0 end
    .fake_r insert end $dir
    .fake_excd delete 0 end
    .fake_excd insert end $dir
}

#Listbox to clipboard handler
proc handle_lb_clip {fake w} {
    set dir [$fake get]
    set cur [$w curselection]
    if {$cur eq ""} {
	return
    }
    clipboard clear
    clipboard append -- [file join $dir [$w get $cur]]
}

read_dir_l [pwd]
read_dir_r [pwd]

#Left scroolbar and procs to scroll
ttk::scrollbar .sy -orient vert -command {lb_set yview}
pack .sy -in .pane_t -fill y -after .time -side left

proc lb_set {args} {
    foreach lb {.l .sz .time} {
	$lb {*}$args
    }
}

proc lb_scroll {args} {
    .sy set {*}$args
    lb_set yview moveto [lindex $args 0]
}

foreach lb {.l .sz .time} {
    bind $lb <ButtonPress-1> {
	lb_set select clear 0 end
	lb_set select set [%W nearest %y]
	lb_set activate [%W nearest %y]
	.bot2.e_l delete 0 end
	.bot2.e_l insert end [.l get [.l curselection]]
	break
    }
    bind $lb <B1-Motion> {
	lb_set select clear 0 end
	lb_set select set [%W nearest %y]
	lb_set activate [%W nearest %y]
	lb_set see [%W nearest %y]
	.bot2.e_l delete 0 end
	.bot2.e_l insert end [.l get [.l curselection]]
	break
    }
    bind $lb <ButtonRelease-1> {
	lb_set select clear 0 end
	lb_set select set [%W nearest %y]
	lb_set activate [%W nearest %y]
	.bot2.e_l delete 0 end
	.bot2.e_l insert end [.l get [.l curselection]]
	break
    }
}
unset lb

#Right scroolbar and proc for scroll
ttk::scrollbar .sy_r -orient vert -command {lb_set_r yview}
pack .sy_r -in .pane_t -fill y -after .time_r -side right

proc lb_set_r {args} {
    foreach lb_r {.l_r .sz_r .time_r} {
	$lb_r {*}$args
    }
}
proc lb_scroll_r {args}  {
    .sy_r set {*}$args
    lb_set_r yview moveto [lindex $args 0]
}

foreach lb_r {.l_r .sz_r .time_r} {
    bind $lb_r <ButtonPress-1> {
	lb_set_r select clear 0 end
	lb_set_r select set [%W nearest %y]
	lb_set_r activate [%W nearest %y]
	.bot2.e_r delete 0 end
	.bot2.e_r insert end [.l_r get [.l_r curselection]]
	break
    }
    bind $lb_r <B1-Motion> {
	lb_set_r select clear 0 end
	lb_set_r select set [%W nearest %y]
	lb_set_r activate [%W nearest %y]
	lb_set_r see [%W nearest %y]
	.bot2.e_r delete 0 end
	.bot2.e_r insert end [.l_r get [.l_r curselection]]
	break
    }
    bind $lb_r <ButtonRelease-1> {
	lb_set_r select clear 0 end
	lb_set_r select set [%W nearest %y]
	lb_set_r activate [%W nearest %y]
	.bot2.e_r delete 0 end
	.bot2.e_r insert end [.l_r get [.l_r curselection]]
	break
    }
}
unset lb_r

#Show file names on bottom entry right and left
#Scroll selected lines in all listboxes in left

proc show_item_down {w w2} {
    set item [$w curselection]
    set item2 [expr {$item + 1}]
    if {$item2 eq [$w index end]} {
	return
    }
    $w2 delete 0 end
    $w2 insert end [$w get $item2]
    foreach sel {.l .sz .time} {
	$sel selection clear $item
	$sel selection set $item2
	$sel selection clear $item
    }
}

proc show_item_up {w w2} {
    set item [$w curselection]
    if {$item == 0} {
	return
    }
    set item2 [expr {$item - 1}]
    $w2 delete 0 end
    $w2 insert end [$w get $item2]
    foreach sel {.l .sz .time} {
	$sel selection clear $item
	$sel selection set $item2
	$sel selection clear $item
    }
}

#Scroll all selection in right side

proc show_item_down_r {w w2} {
    set item [$w curselection]
    set item2 [expr {$item + 1}]
    if {$item2 eq [$w index end]} {
	return
    }
    $w2 delete 0 end
    $w2 insert end [$w get $item2]
    foreach sel {.l_r .sz_r .time_r} {
	$sel selection clear $item
	$sel selection set $item2
	$sel selection clear $item
    }
}

proc show_item_up_r {w w2} {
    set item [$w curselection]
    if {$item == 0} {
	return
    }
    set item2 [expr {$item - 1}]
    $w2 delete 0 end
    $w2 insert end [$w get $item2]
    foreach sel {.l_r .sz_r .time_r} {
	$sel selection clear $item
	$sel selection set $item2
	$sel selection clear $item
    }
}

proc edit_file {fake w} {
    set dir [$fake get]
    set fileex [file join $dir [$w get active]]
    if {[file isdirectory $fileex]} {
	tk_messageBox -parent . -title  "Error" -type ok \
	    -icon error -message "This is a directory"
    } else {
	editor
	if {[file size $fileex] > 16 * 1024 * 1024} {
	    error "File too large"
	}
	set f [open $fileex]
	.ed.text.t delete 1.0 end
	catch {.ed.text.t edit reset}
	.ed.text.t insert end [read $f]
	close $f
	.ed.text.t mark set insert 1.0
	.ed.text.t see insert
	catch {.ed.text.t edit reset}
	.fake_ed delete 0 end
	.fake_ed insert end $fileex
    }
}

#Change left directory list
proc read_left {} {
    set old_dir [.fake_l get]
    set dir [file norm [file join $old_dir [.l get active]]]
    if {[file isdirectory $dir]} {
	read_dir_l $dir
	.l selection set 0
	.sz selection set 0
	.time selection set 0
	.l activate 0
	.top2.e_l delete 0 end
	.top2.e_l insert end [file native $dir]
	.fake_l delete 0 end
	.fake_l insert end $dir
	.fake_excd delete 0 end
	.fake_excd insert end $dir
    } else {
	set fileex $dir
	if {[file system $dir] eq "native" && [file executable $fileex]} {
	    if {$::tcl_platform(platform) eq "windows"} {
		catch {exec $fileex < NUL: &} result
	    } else {
		catch {exec $fileex < /dev/null >@ stdout 2>@ stderr &} result
	    }
	}
	if {[id_giffile $fileex]}  {
	    img_view $fileex
	}
	if {[id_txtfile $fileex]} {
	    set openfile [open $fileex "r"]
	    .bot.t delete 1.0 end
	    catch {.bot.t edit reset}
	    .bot.t insert end [read $openfile]
	    catch {.bot.t edit reset}
	    close $openfile
	    .fake_bot delete 0 end
	    .fake_bot insert end $fileex
	}
    }
}

proc read_right {} {
    # Get selected list item
    set old_dir [.fake_r get]
    set dir_r [file norm [file join $old_dir [.l_r get active]]]
    if {[file isdirectory $dir_r]} {
	read_dir_r $dir_r
	.l_r selection set 0
	.sz_r selection set 0
	.time_r selection set 0
	.top2.e_r delete 0 end
	.top2.e_r insert end [file native $dir_r]
	.fake_r delete 0 end
	.fake_r insert end $dir_r
	.fake_excd delete 0 end
	.fake_excd insert end $dir_r
    } else {
	set fileex $dir_r
	if {[file system $dir_r] eq "native" && [file executable $fileex]} {
	    if {$::tcl_platform(platform) eq "windows"} {
		catch {exec $fileex < NUL: &} result
	    } else {
		catch {exec $fileex < /dev/null > @stdout 2> @stderr &} result
	    }
	}
	if {[id_giffile $fileex]}  {
	    img_view $fileex
	}
	if {[id_txtfile $fileex]} {
	    set openfile [open $fileex "r"]
	    .bot.t delete 1.0 end
	    catch {.bot.t edit reset}
	    .bot.t insert end [read $openfile]
	    catch {.bot.t edit reset}
	    close $openfile
	}
    }
}

proc copy_file_to_r {} {
    set file_l [.l get active]
    set left_dir [.fake_l get]
    set right_dir [.fake_r get]
    set choice [tk_messageBox -parent . -type yesno -default yes \
	    -message "Copy\n [file native [file join $left_dir $file_l]] \nto\n \
	    [file native $right_dir]" -icon question]
    if {$choice eq "yes"} {
	if {[file isdirectory [file join $left_dir $file_l]]} {
	    file copy -force [file join $left_dir $file_l] $right_dir
	} else {
	    file copy [file join $left_dir $file_l] $right_dir
	}
	refresh_right
	refresh_left
    }
}

proc copy_file_to_l {} {
    set file_r [.l_r get active]
    set left_dir [.fake_l get]
    set right_dir [.fake_r get]
    set choice [tk_messageBox -parent . -type yesno -default yes \
	    -message "Copy\n [file native [file join $right_dir $file_r]] \nto\n \
	    [file native $left_dir]" -icon question]
    if {$choice eq "yes"} {
	if {[file isdirectory [file join $right_dir $file_r]]} {
	    file copy -force [file join $right_dir $file_r] $left_dir;
	} else {
	    file copy [file join $right_dir $file_r] $left_dir
	}
	refresh_left
	refresh_right
    }
}

#Rename move files and Directories
proc ren_move {fake w} {
    catch [destroy .move]
    toplevel .move
    wm resizable .move 0 0
    wm title .move "Rename"
    grab .move
    ::tk::PlaceWindow .move widget .
    set dir [$fake get]
    set file_move [file join $dir [$w get active]]
    label .move.l -text "Current name is: \n$file_move \n\n Enter new name:  "
    entry .move.e
    .move.e insert end $file_move
    focus .move.e
    frame .move.ok -relief sunken -bd 2
    set cmd [subst {
	file rename [list $file_move] \[.move.e get\]
	refresh_right
	refresh_left
	destroy .move
    }]
    button .move.ok.ok -text OK -width 7 -command $cmd
    button .move.cancel -text Cancel -width 7 -command {destroy .move}
    pack .move.l .move.e  -side top -fill x -padx 10 -pady 5 -expand 1
    pack .move.ok .move.ok.ok .move.cancel -side left -padx 10 -pady 5
    bind .move.e <Return> $cmd
}

proc delete_file_l {} {
    set file_l [.l get active]
    set left_dir [.fake_l get]
    if {[file isdirectory [file join $left_dir $file_l]]} {
	set question [tk_messageBox -type okcancel -default cancel \
	-message "   Delete directory and files in \n   [file native $left_dir/$file_l]" \
	-icon question]
	switch -- $question {
	    ok {
		file delete -force [file join $left_dir $file_l]
		refresh_left
		refresh_right
	    }
	}
    } else {
	set question [tk_messageBox -type okcancel -default cancel \
			  -message "   Delete files \n  [file native $left_dir/$file_l]" \
			  -icon question]
	switch -- $question {
	    ok {
		file delete [file join $left_dir $file_l]
		refresh_left
		refresh_right
	    }
	}
    }
}

proc delete_file_r {} {
    set file_r [.l_r get active]
    set right_dir [.fake_r get]
    if {[file isdirectory [file join $right_dir $file_r]]} {
	set question [tk_messageBox -type okcancel -default cancel \
			  -message "   Delete directory and files in \n   [file native $right_dir/$file_r] " \
			  -icon question]
	switch -- $question {
	    ok { file delete -force [file join $right_dir $file_r]
		refresh_right
		refresh_left
	    }
	}
    } else {
	set question [tk_messageBox -type okcancel -default cancel \
			  -message "   Delete files \n [file native $right_dir/$file_r]" \
			  -icon question]
	switch -- $question {
	    ok {
		file delete [file join $right_dir $file_r]
		refresh_right
		refresh_left
	    }
	}
    }
}

proc mk_dir {w} {
    catch [destroy .mkdir]
    toplevel .mkdir
    wm resizable .mkdir 0 0
    wm title .mkdir "Create new Directory"
    grab .mkdir
    ::tk::PlaceWindow .mkdir widget .
    if {$w eq ".l"} {
	set wd [.fake_l get]
    } else {
	set wd [.fake_r get]
    }
    label .mkdir.l -text "  Current directory is: \n[file native $wd] \n\n Enter new directory name   "
    entry .mkdir.e
    focus .mkdir.e
    frame .mkdir.ok -relief sunken -bd 2 
    button .mkdir.ok.ok -text OK -width 7 \
	-command {
	    file mkdir [.mkdir.e get]
	    refresh_right
	    refresh_left
	    destroy .mkdir
	}
    button .mkdir.cancel -text Cancel -width 7 -command {destroy .mkdir}
    pack .mkdir.l .mkdir.e  -side top -fill x -padx 10 -pady 5 -expand 1
    pack .mkdir.ok .mkdir.ok.ok .mkdir.cancel -side left -padx 10 -pady 5
    bind .mkdir.e <Return> {
	file mkdir [.mkdir.e get]
	refresh_right
	refresh_left
	destroy .mkdir
    }
}

proc find_win {w} {
    catch [destroy .find]
    toplevel .find
    wm title .find "Find"
    wm resizable .find 0 0
    grab .find
    ::tk::PlaceWindow .find widget .
    if {$w eq ".l"} {
	set wd [.fake_l get]
    } else {
	set wd [.fake_r get]
    }
    label .find.l -text "\tCurrent directory is:\t\n\t[file native $wd]\t\t\n\nSearch Filename:"
    entry .find.e
    pack .find.l .find.e -fill x -padx 10 -pady 5 -expand 1
    focus .find.e
    frame .find.ok -relief sunken -bd 1
    button .find.ok.ok -text Find -width 7 -command [subst {
	.bot.t delete 1.0 end
	catch {.bot.t edit reset}
	find_file [list $wd] \[.find.e get\]
	catch {.bot.t edit reset}
	destroy .find
    }]
    button .find.cancel -text Cancel -width 7 -command {destroy .find}
    pack .find.ok .find.ok.ok .find.cancel -side left -padx 10 -pady 5
}

#Find file
proc find_file {dir name_pat} {
    catch {
	foreach match [glob -directory $dir -nocomplain -tails -- $name_pat] {
	    .bot.t insert end [file native [file join $dir $match]]
	    .bot.t insert end "\n"
	}
	update idletasks
    }
    catch {
	foreach file [glob -directory $dir -nocomplain -- * .*] {
	    set tail [file tail $file]
	    if {$tail eq "." || $tail eq ".."} continue
	    if {[file isdirectory $file]} {
		find_file [file join $dir $file] $name_pat
		update idletasks
	    }
	}
    }
}

#Refresh left listbox after copy or deleting a file
proc refresh_left {} {
    set left_dir [.fake_l get]
    set cur [.l curselection]
    if {$cur eq ""} {
	focus_if . .l
	.l selection set 0
	.sz selection set 0
	.time selection set 0
	.l see 0
	.l activate 0
    } else {
	read_dir_l $left_dir
	.l selection set $cur
	.sz selection set $cur
	.time selection set $cur
	.l see $cur
	.l activate $cur
	.top2.e_l delete 0 end
	.bot2.e_l delete 0 end
	.top2.e_l insert end [file native $left_dir]
    }
}

refresh_left

proc refresh_right {} {
    set right_dir [.fake_r get]
    set cur [.l_r curselection]
    if {$cur eq ""} {
	.l_r selection set 0
	.sz_r selection set 0
	.time_r selection set 0
	.l_r see 0
	.l_r activate 0
    } else {
	read_dir_r $right_dir
	.l_r selection set $cur
	.sz_r selection set $cur
	.time_r selection set $cur
	.l_r see $cur
	.l_r activate $cur
	.top2.e_r delete 0 end
	.bot2.e_r delete 0 end
	.top2.e_r insert end [file native $right_dir]
    }
}

refresh_right

proc view_file {fake w} {
    set dir [$fake get]
    set fileex [file join $dir [$w get active]]
    if {[file isdirectory $fileex]} {
	tk_messageBox -parent . -title  "Error" -type ok \
	    -icon error -message "This is a directory"
    } else {
	if {[file size $fileex] > 16 * 1024 * 1024} {
	    error "File too large"
	}
	set openfile [open $fileex "r"]
	.bot.t delete 1.0 end
	catch {.bot.t edit reset}
	.bot.t insert end [read $openfile]
	catch {.bot.t edit reset}
	close $openfile
	.fake_bot delete 0 end
	.fake_bot insert end $fileex
    }
}

#Bindings global
bind . <F1> about
bind . <F2> {
    .bot.t delete 1.0 end
    catch {.bot.t edit reset}
}
bind . <F3> {eval [.but.b3 cget -command]}
bind . <F4> {editor ; break}
bind . <F10> exit
bind . <F9> {
    focus_if . .bot.t
    run_tcl [.fake_bot get] .bot.t
}
bind . <Control-r> {refresh_left ; refresh_right ; break}

#Bindings keys for left side
foreach bind_left { .l .sz .time} {
    bind $bind_left <Enter> {
	focus_if . .l
	.but.b3 config -command {view_file .fake_l .l}
	.but.b4 config -command {edit_file .fake_l .l}
	.but.b5 config -command copy_file_to_r
	.but.b6 config -command {ren_move .fake_l .l}
	.but.b7 config -command {mk_dir .l}
	.but.b8 config -command delete_file_l
    }
}
unset bind_left

bind .l <Double-Button-1> read_left
bind .l <Return> read_left
bind .l <Key-r> {focus .l_r ; break}
bind .l <Up> {show_item_up .l .bot2.e_l}
bind .l <Down> {show_item_down .l .bot2.e_l}
bind .l <F3> {view_file .fake_l .l ; break}
bind .l <F4> {edit_file .fake_l .l ; break}
bind .l <F5> copy_file_to_r
bind .l <F6> {ren_move .fake_l .l}
bind .l <F7> {mk_dir .l}
bind .l <F8> delete_file_l
bind .l <Control-r> {refresh_left ; break}

#Bindings keys for right side
foreach bind_right {.l_r .sz_r .time_r} {
    bind $bind_right <Enter> {
	focus_if . .l_r
	.but.b3 config -command {view_file .fake_r .l_r}
	.but.b4 config -command {edit_file .fake_r .l_r}
	.but.b8 config -command delete_file_r
	.but.b5 config -command copy_file_to_l
	.but.b6 config -command {ren_move .fake_r .l_r}
	.but.b7 config -command {mk_dir .l_r}
    }
}
unset bind_right

bind .l_r <Double-Button-1> read_right
bind .l_r <Return> read_right
bind .l_r <Key-l> {focus .l ; break}
bind .l_r <Down> {show_item_down_r .l_r .bot2.e_r}
bind .l_r <Up> {show_item_up_r .l_r .bot2.e_r}
bind .l_r <F3> {view_file .fake_r .l_r ; break}
bind .l_r <F4> {edit_file .fake_r .l_r ; break}
bind .l_r <F5> copy_file_to_l
bind .l_r <F6> {ren_move .fake_r .l_r}
bind .l_r <F7> {mk_dir .l_r}
bind .l_r <F8> delete_file_r
bind .l_r <Control-r> {refresh_right ; break}

#Txt, ASCII and bat files viewer
proc id_allfiles {fileex} {
    set ext_txt [file extension $fileex]
    switch -nocase -- $ext_txt {
	.doc  {return 1}
	.txt  {return 1}
	.jpeg {return 1}
	.jpg  {return 1}
	.txt  {return 1}
	.htm  {return 1}
	.html {return 1}
    }
    return 0
}

proc id_txtfile {fileex} {
    set ext_txt [file extension $fileex]
    switch -nocase -- $ext_txt {
	.txt  {return 1}
	.bat  {return 1}
	.c    {return 1}
	.h    {return 1}
    }
    return 0
}

proc id_tclfile {fileex} {
    set ext_tcl [file extension $fileex]
    switch -nocase -- $ext_tcl {
	.tcl {return 1}
	.tk  {return 1}
    }
    return 0
}

proc id_giffile {fileex} {
    set ext_gif [file extension $fileex]
    switch -nocase -- $ext_gif {
	.gif  {return 1}
	.jpeg {return 1}
	.jpg  {return 1}
	.tif  {return 1}
	.png  {return 1}
    }
    return 0
}

proc id_htmlfile {fileex} {
    set ext_html [file extension $fileex]
    switch -nocase -- $ext_html {
	.htm  {return 1}
	.html  {return 1}
    }
    return 0
}

#File attributes and properties
proc fileprop {pathname} {
    set padx 6
    if {$pathname eq "."} {
	set pathname [pwd]
    }
    set pathname [file join [pwd] $pathname]
    set checkbuttons [list]
    file stat $pathname a ;# may error out if no such file
    set w .[clock clicks]
    set ::${w}(dir) [file dir $pathname]
    set ::${w}(file) [file tail $pathname]
    toplevel $w
    grab $w
    wm title $w [file nativename $pathname]
    wm resizable $w 0 0
    wm geometry $w +300+150
    set size [fmtfsize [file size $pathname]]
    set textual [list Name [file tail $pathname]  \
		     Directory [file nativename [file dir $pathname]] \
		     Type [file type $pathname]\
		     Size $size \
		     Created [date_time $a(ctime) 1]\
		     "Last modified" [date_time $a(mtime) 1]\
		     "Last accessed" [date_time $a(atime) 1]\
		    ]
    foreach {name value} [file attr $pathname] {
	if {[string match "-*compsize" $name]} {
	    continue
	}
	if {[regexp {^[01]$} $value]} {
	    lappend checkbuttons $name $value
	} else {
	    lappend textual  $name [file nativename $value]
	}
    }
    set n 0
    foreach {name value} $textual {
	grid [label $w.${n}n -text $name:] [label $w.${n}v -text $value]\
		-sticky w -padx $padx
	incr n
    }
    grid [hr $w.$n] -sticky we -columnspan 2 -padx $padx -pady 6
    set n0 [incr n]
    foreach {name value} $checkbuttons {
	incr n
	set ::${w}($name) $value
	grid [checkbutton $w.$n -text $name -var ${w}($name) -borderwidth 0] \
	    -sticky w -column 1 -padx $padx
    }
    if {[llength $checkbuttons]} {
	grid [label $w.att -text Attributes:]\
	    -row $n0 -sticky w -padx $padx
	grid [hbuttons $w.b [list OK [subst {
	    fileprop_apply $w
	    destroy $w
	    unset $w
	}] Cancel [subst {
	    destroy $w
	    unset $w
	}] Apply [subst {
	    fileprop_apply $w
	}]]] -column 1 -padx $padx
    } else {
	grid [hbuttons $w.b [list Close [subst {
	    destroy $w
	    unset $w
	}]]] -column 1 -padx $padx
    }
    wm protocol $w WM_DELETE_WINDOW [subst {destroy $w ; unset $w}]
    focus $w
}

proc fileprop_apply {w} {
    upvar #0 $w a
    set cmd [list file attributes [file join $a(dir) $a(file)]]
    foreach {name value} [array get a] {
	if {[regexp {^-} $name]} {
	    lappend cmd $name $value
	}
    }
    {*}$cmd
}

proc hbuttons {w tc} {
    frame $w
    set n 1
    foreach {t c} $tc {
	button $w.$n -text $t -command $c -width 8
	incr n
    }
    pack {*}[winfo children $w] -side left -padx 3 -pady 6 \
	-fill x -anchor e
    return $w
}

proc hr {w} {
    frame $w -height 2 -borderwidth 1 -relief sunken
}

#Popup menus

proc pop_upmen_l {w} {
    menu .popup_l -tearoff 0
    .popup_l add command -label Info -command {
	fileprop [file join [.fake_l get] [.bot2.e_l get]]
    }
    .popup_l add command -label View -command {view_file .fake_l .l}
    .popup_l add command -label Edit -command {edit_file .fake_l .l}
    .popup_l add command -label Find -command {find_win .l}
    .popup_l add command -label Copy -command copy_file_to_r
    .popup_l add command -label NewDir -command {mk_dir .l}
    .popup_l add command -label Rename -command {ren_move .fake_l .l}
    .popup_l add command -label Delete -command delete_file_l
    bind $w <ButtonPress-3>  {tk_popup .popup_l %X %Y}
}

proc pop_upmen_r {w} {
    menu .popup_r -tearoff 0
    .popup_r add command -label Info -command {
	fileprop [file join [.fake_r get] [.bot2.e_r get]]
    }
    .popup_r add command -label View -command {view_file .fake_r .l_r}
    .popup_r add command -label Edit -command {edit_file .fake_r .l_r}
    .popup_r add command -label Find -command {find_win .r}
    .popup_r add command -label Copy -command copy_file_to_l
    .popup_r add command -label NewDir -command {mk_dir .l_r}
    .popup_r add command -label Rename -command {ren_move .fake_r .l_r}
    .popup_r add command -label Delete -command delete_file_r
    bind $w <ButtonPress-3> {tk_popup .popup_r %X %Y}
}

pop_upmen_l .l
pop_upmen_r .l_r

#Image viewer
proc img_view {fileex} {
    set img [image create photo]
    if {[catch {$img read $fileex} error]} {
	image delete $img
	error $error
    }
    set w .img_$img
    toplevel $w
    wm title $w $fileex
    wm minsize $w 100 100
    set width [image width $img]
    set height [image height $img]
    canvas $w.c -relief sunken -bd 1 -highlightthickness 0 \
	-yscrollcommand [list $w.scr set] \
	-xscrollcommand [list $w.xscr set]
    ttk::scrollbar $w.scr -orient vert -command [list $w.c yview]
    ttk::scrollbar $w.xscr -orient horiz -command [list $w.c xview]
    $w.c create image 0 0 -image $img -anchor nw
    grid $w.c -column 0 -row 0 -sticky nswe
    grid $w.scr -column 1 -row 0 -sticky ns
    grid $w.xscr -column 0 -row 1 -sticky we
    grid columnconfig $w 0 -weight 1
    grid rowconfig $w 0 -weight 1
    $w.c configure -scrollregion [list 0 0 $width $height]
    wm protocol $w WM_DELETE_WINDOW [subst {
	destroy $w
	image delete $img
    }]
}

#See http://wiki.tcl.tk/21701
proc words2 {cmd} {
    if {![info complete $cmd]} {
	error [list {Not a complete command} $cmd]
    }
    set words {}
    set logical {}
    set cmd [string trimleft $cmd[set cmd {}] "\f\n\r\t\v " ]
    while {[regexp {([^\f\n\r\t\v ]*)([\f\n\r\t\v ]+)(.*)} $cmd full first delim last]} {
	append logical $first
	if {[info complete $logical\n]} {
	    lappend words $logical
	    set logical {}
	} else {
	    append logical $delim
	}
	set cmd $last[set last {}]
    }
    if {$cmd ne {}} {
	append logical $cmd
    }
    if {$logical ne {}} {
	lappend words $logical
    }
    return $words
}

#Run command from entry
proc exec_cmd {command} {
    set result " "
    if {[catch {lindex [words2 $command] 0} check_cmd]} {
	set cmd ""
    } else {
	set cmd [info commands $check_cmd]
    }
    after cancel refresh_left
    after idle refresh_left
    after cancel refresh_right
    after idle refresh_right
    if {[string length $cmd] > 0} {
	.bot.t insert end [set result [uplevel \#0 $command]]
    } else {
	set pwd [pwd]
	set dir [.fake_excd get]
	if {[file system $dir] ne "native"} {
	    error "Current directory not in native filesystem"
	}
	cd $dir
	if {$::tcl_platform(platform) eq "windows"} {
	    set err [catch {
		exec -ignorestderr -- cmd /c $command <NUL: 2>@1
	    } msg]
	} else {
	    set err [catch {
		exec -ignorestderr -- sh -c $command </dev/null 2>@1
	    } msg]
	}
	catch {cd $pwd}
	if {$err} {
	    error $msg
	}
	return $msg
    }
}

#Run tcl from text widget
proc run_tcl {scriptname w} {
    if {[interp exists eval_env]}  {
	interp delete eval_env
	tailcall run_tcl $scriptname $w
    }
    interp create eval_env
    load {} Tk eval_env
    if {([info command sdltk] eq "sdltk") && [sdltk android]} {
	load {} Borg eval_env
    }
    set all [$w get 1.0 end]
    eval_env eval [list set argv0 $scriptname]
    eval_env eval [list set argv {}]
    eval_env eval [list set argc 0]
    eval_env eval [list info script $scriptname]
    eval_env eval {proc exit args {destroy .}}
    eval_env eval $all
}

proc editor {} {
    catch {destroy .ed}
    toplevel .ed
    wm resizable .ed 0 0
    wm title .ed "TkMC Editor"
    menu .ed.menubar -relief raised -borderwidth 1
    menu .ed.menubar.file -tearoff 0
    .ed.menubar add cascade -label "File" -menu .ed.menubar.file -underline 0
    .ed.menubar.file add command -label "New" \
	-command {clear_all .ed.text.t ; .fake_ed delete 0 end}
    .ed.menubar.file add command -label "Open" -command text_open
    .ed.menubar.file add command -label "Close" \
	-command {clear_all .ed.text.t ; .fake_ed delete 0 end}
    .ed.menubar.file add command -label "Save as..." \
	-command {text_saveas}
    .ed.menubar.file add command -label "Exit" \
	-command {bell ; prompt_close .ed}
    .ed.menubar add cascade -label "Edit" -menu .ed.menubar.editt -underline 0
    menu .ed.menubar.editt -tearoff 0
    .ed.menubar.editt add command -label "Select All" \
	-command {.ed.text.t tag add sel 1.0 end}
    .ed.menubar.editt add command -label "Copy" \
	-command {tk_textCopy .ed.text.t}
    .ed.menubar.editt add command -label "Paste" \
	-command {tk_textPaste .ed.text.t}
    .ed.menubar.editt add command -label "Cut" \
	-command {tk_textCut .ed.text.t}
    .ed.menubar.editt add command -label "Clear" \
	-command {clear_all .ed.text.t ; .fake_ed delete 0 end}
    .ed.menubar add cascade -label "Test Tcl" \
	-menu .ed.menubar.run -underline 0
    menu .ed.menubar.run -tearoff 0
    .ed.menubar.run add command -label "Run Tcl" \
	-command {run_tcl [.fake_ed get] .ed.text.t}
    .ed.menubar add cascade -label "Help" -menu .ed.menubar.help -underline 0
    menu .ed.menubar.help -tearoff 0
    .ed.menubar.help add command -label "Help" -command {about .ed}
    .ed.menubar.help add command -label "About" -command {about .ed}

    .ed conf -menu .ed.menubar

    proc clear_all {w} {
	$w delete 1.0 end
	catch {$w edit reset}
    }

    proc text_open {{fn ""}} {
	set types {
	    {{All Files}    *}
	    {{Text Files}   {.txt}}
	}
	if {$fn eq "" &&
	    ([catch {ttk::getOpenFile -filetypes $types -parent .ed \
			 -title "Source File" -parent .ed} fn] || $fn eq "")} {
	    return
	}
	if {[file size $fn] > 16 * 1024 * 1024} {
	    error "File too large"
	}
	set f [open $fn]
	clear_all .ed.text.t
	.ed.text.t insert end [read $f]
	catch {.ed.text.t edit reset}
	close $f
	.ed.text.t mark set insert 1.0
	.ed.text.t see insert
	.fake_ed delete 0 end
	.fake_ed insert end $fn
    }

    # Handles window manager Close choice.

    proc prompt_close {w} {
	# If main window, prompt.
	if {$w eq "."} {
	    set result [tk_messageBox -parent .ed \
			    -title {Exit?} -type yesno\
			    -icon warning \
			    -message "Do you really want to quit?"]
	    if {$result eq "yes"} {
		destroy .ed
	    }
	} else {
	    # Not the main window, destroy it.
	    destroy $w
	}
    }

    wm protocol .ed WM_DELETE_WINDOW {prompt_close .ed}

    #save as command
    proc text_saveas {} {
	set types {
	    {{Tcl Files}        {.tcl}}
	    {{Text Files}       {.txt}}
	    {{All Files}        *}
	}
	set filename [ttk::getSaveFile -parent .ed \
			-title "Save as..." -filetypes $types]
	if {$filename ne ""} {
	    set file_open [open $filename w]
	    puts -nonewline $file_open [.ed.text.t get 1.0 end]
	    close $file_open
	    refresh_left
	    refresh_right
	}
    }

    #Image Buttons
    set imnew {
R0lGODlhEAAQAKEBAAAAAP///////////yH5BAEKAAIALAAAAAAQABAAAAIrlI+JwM1qQJj0KUmn
BDfrlmCeFo5VaQbckqqouR7iGEdtLcwe7vQWBAwCCwA7
}

    #Open button image
    set imopen {
R0lGODlhEAAQAKECAAAAAP//AP///////yH5BAEKAAIALAAAAAAQABAAAAIqlI+py40AnQGhwhtf
3RwkynVfKGobhnpgsJKqRZ4TG1sHVLNPikn+LygAADs=
}

    #Save Image
    set imsave {
R0lGODlhEAAQAMIEAAAAAAAAnAAA/87Ozv///////////////yH5BAEKAAcALAAAAAAQABAAAANB
eLrc/g4EIYC9IoAl6fhBMGSb0gnfEI4aN1WXRbpUbVdurFs54f+ESw/oE5o0xCLvCEgGlwdJM2mM
WpzVnRbCTQAAOw==
}

    #image Select all
    set imselectall {
R0lGODlhEAAQAMIDAAAAALa2qtvb/////////////////////yH5BAEKAAQALAAAAAAQABAAAANA
SLrc/g7IMSSgdoHAuw+A9o2hMgSCgKqpOiwnusquOHolsaVzC9osliwXu3Fepl6wptjQhKqc0/hr
TirXDASSAAA7
}

    #Copy Image
    set imcopy {
R0lGODlhEAAQAKEBAAAAAP///////////yH5BAEKAAIALAAAAAAQABAAAAIyhI8nC+EPA1iioSip
NTgzeWDTtzVgYjomB7GtFLnq3KanDD62mO5x39EBgzdcjFJBJQoAOw==
}

    #Image Cut
    set imcut {
R0lGODlhEAAQAKEBAAAAAP///////////yH5BAEKAAIALAAAAAAQABAAAAIolG+Ay8ovnAmyHQrT
rE/hnHxgxB3AiYpIhUKSU5qnHFvReOcgjJNLAQA7
}

    #Image Paste
    set impaste {
R0lGODlhEAAQAMIDAAAAAAAAY2NjY////////////////////yH5BAEKAAQALAAAAAAQABAAAANA
SLrcTiCCByUQwk4Fhr9R5g1bR2KoSHKfFn3sKMvlHNz4HY+BHeyenqz3M/EGOKTOGJwFIb4ckWMR
2h7SHIWSAAA7
}

    #Image Run
    set imrun {
R0lGODlhEAAQAIMAAPwCBAQCBPz+/ISChKSipMTCxLS2tLy+vMzOzMTGxNTS1AAAAAAAAAAAAAAAAAAAACH5BAEAAAAALAAAAAAQABAAAARlEMgJQqDYyiDGrR8oWJxnCcQXDMU4GEYqFN4UEHB+FEhtv7EBIYEohkjBkwJBqggEMB+ncHhaBsDUZmbAXq67EecQ02x2CMWzkAs504gCO3qcDZjkl11FMJVIN0cqHSpuGYYSfhEAIf5oQ3JlYXRlZCBieSBCTVBUb0dJRiBQcm8gdmVyc2lvbiAyLjUNCqkgRGV2ZWxDb3IgMTk5NywxOTk4LiBBbGwgcmlnaHRzIHJlc2VydmVkLg0KaHR0cDovL3d3dy5kZXZlbGNvci5jb20AOw==
}

    #Image Help
    set imhelp {
R0lGODlhEAAQAMIFAAAAAAAAqgAA/21tVba2qv///////////yH5BAEKAAcALAAAAAAQABAAAANC
eLrcOzBKN4q9dlArurBAyFQd9xUEMBYle6YrJgQW/Mgnqt4mZisVjG8HvARyuljhOFQKkw+CdEol
HkLYLNbBbSQAADs=
}

    image create photo imnew -data $imnew -gamma 1 -height 16 -width 16 -palette 5/5/4
    image create photo imopen -data $imopen -gamma 1 -height 16 -width 16 -palette 5/5/4
    image create photo imsave -data $imsave -gamma 1 -height 16 -width 16 -palette 5/5/4
    image create photo imselectall -data $imselectall
    image create photo imcopy -data $imcopy -gamma 1 -height 16 -width 16 -palette 5/5/4
    image create photo imcut -data $imcut -gamma 1 -height 16 -width 16 -palette 5/5/4
    image create photo impaste -data $impaste -gamma 1 -height 16 -width 16 -palette 5/5/4
    image create photo imrun -data $imrun
    image create photo imhelp -data $imhelp

    set ih 16
    set iw 16
    set dpi [expr int((25.4 * [winfo screenwidth .]) / [winfo screenmmwidth .])]
    if {$dpi < 140} {
	set iscale 1
    } elseif {$dpi < 240} {
	set iscale 2
    } elseif {$dpi < 320} {
	set iscale 3
    } elseif {$dpi < 400} {
	set iscale 4
    } else {
	set iscale 5
    }
    if {$iscale > 1} {
	foreach img {
	    imnew imopen imsave imselectall imcopy imcut impaste imrun imhelp
	} {
	    set tmpi [image create photo]
	    $tmpi copy $img -zoom $iscale $iscale
	    $img blank
	    $img config -width 0 -height 0
	    $img copy $tmpi
	    image delete $tmpi
	}
	set ih [expr {$ih * $iscale}]
	set iw [expr {$iw * $iscale}]
    }

    frame .ed.menubuttons
    pack .ed.menubuttons -side top -fill x
    ttk::button .ed.menubuttons.new -style Toolbutton  -image imnew \
	-takefocus 0 -command {clear_all .ed.text.t}
    ttk::button .ed.menubuttons.open -style Toolbutton -image imopen \
	-takefocus 0 -command text_open
    ttk::button .ed.menubuttons.save -style Toolbutton -image imsave \
	-takefocus 0 -command text_saveas
    ttk::button .ed.menubuttons.selectall -style Toolbutton -image imselectall \
	-takefocus 0 -command {.ed.text.t tag add sel 1.0 end}
    ttk::button .ed.menubuttons.copy -style Toolbutton -image imcopy \
	-takefocus 0 -command {tk_textCopy .ed.text.t}
    ttk::button .ed.menubuttons.cut -style Toolbutton -image imcut \
	-takefocus 0 -command {tk_textCut .ed.text.t}
    ttk::button .ed.menubuttons.paste -style Toolbutton  -image impaste \
	-takefocus 0 -command {tk_textPaste .ed.text.t}
    ttk::button .ed.menubuttons.run -style Toolbutton -image imrun \
	-takefocus 0 -command {run_tcl [.fake_ed get] .ed.text.t}
    ttk::button .ed.menubuttons.help -style Toolbutton -image imhelp \
	-takefocus 0 -command {about .ed}

    label .ed.menubuttons.l -text ""
    pack .ed.menubuttons.new .ed.menubuttons.open .ed.menubuttons.save \
	.ed.menubuttons.selectall .ed.menubuttons.copy \
	.ed.menubuttons.cut .ed.menubuttons.paste \
	.ed.menubuttons.run .ed.menubuttons.help -side left -fill x
    pack .ed.menubuttons.l -side left -fill x -padx 8

    bind .ed.menubuttons.new <Enter> \
	{.ed.menubuttons.l config -text {New}}
    bind .ed.menubuttons.new <Leave> \
	{.ed.menubuttons.l config -text {}}
    bind .ed.menubuttons.open <Enter> \
	{.ed.menubuttons.l config -text {Open File}}
    bind .ed.menubuttons.open <Leave> \
	{.ed.menubuttons.l config -text {}}
    bind .ed.menubuttons.save <Enter> \
	{.ed.menubuttons.l config -text {Save as...}}
    bind .ed.menubuttons.save <Leave> \
	{.ed.menubuttons.l config -text {}}
    bind .ed.menubuttons.selectall <Enter> \
	{.ed.menubuttons.l config -text {Select all}}
    bind .ed.menubuttons.selectall <Leave> \
	{.ed.menubuttons.l config -text {}}
    bind .ed.menubuttons.copy <Enter> \
	{.ed.menubuttons.l config -text {Copy selected text}}
    bind .ed.menubuttons.copy <Leave> \
	{.ed.menubuttons.l config -text {}}
    bind .ed.menubuttons.cut <Enter> \
	{.ed.menubuttons.l config -text {Cut selected text}}
    bind .ed.menubuttons.cut <Leave> \
	{.ed.menubuttons.l config -text {}}
    bind .ed.menubuttons.paste <Enter> \
	{.ed.menubuttons.l config -text {Paste selected text}}
    bind .ed.menubuttons.paste <Leave> \
	{.ed.menubuttons.l config -text {}}
    bind .ed.menubuttons.run <Enter> \
	{.ed.menubuttons.l config -text {Run Tcl}}
    bind .ed.menubuttons.help <Enter> \
	{.ed.menubuttons.l config -text {Help}}
    bind .ed.menubuttons.help <Leave> \
	{.ed.menubuttons.l config -text {}}

    #Text widget
    frame .ed.text
    text .ed.text.t -wrap none -bg white -cursor arrow -height $::parms(edh) \
	-yscrollcommand {.ed.text.scrl set} \
	-xscrollcommand {.ed.text.scrl2 set}
    catch {.ed.text.t configure -undo 1 -maxundo 500 -autoseparators 1}
    ttk::scrollbar .ed.text.scrl -command {.ed.text.t yview}
    ttk::scrollbar .ed.text.scrl2 -command {.ed.text.t xview} -orient horiz
    grid .ed.text.scrl -column 1 -row 0 -sticky ns
    grid .ed.text.scrl2 -column 0 -row 1 -sticky we
    grid .ed.text.t -column 0 -row 0 -sticky nswe -padx 2 -pady 2
    pack .ed.text -side left -fill both -expand true
    focus .ed.text.t
    .fake_ed delete 0 end
}

proc about {{top {}}} {
    set ttop $top
    if {$ttop eq {}} {
	set ttop .
    }
    catch {destroy $top.about}
    toplevel $top.about
    wm transient $top.about $ttop
    wm title $top.about "About TkMC"
    wm resizable $top.about 0 0
    ::tk::PlaceWindow $top.about widget $ttop
    label $top.about.l -justify c -text "\n\
 TkMC v1.0chw\n\
\n\
 Original Author: Grigoriy Abramov \n\
 Modified by: Christian Werner\n\
"
    pack $top.about.l -padx 10 -pady 5
    grab $top.about
    focus $top.about
}
