
proc Blt_ActiveLegend { graph } {
    $graph legend bind all <Enter> [list blt::ActivateLegend $graph ]
    $graph legend bind all <Leave> [list blt::DeactivateLegend $graph]
    $graph legend bind all <ButtonPress-1> [list blt::HighlightLegend $graph]
}

proc Blt_Crosshairs { graph } {
    blt::Crosshairs $graph 
}

proc Blt_ResetCrosshairs { graph state } {
    blt::Crosshairs $graph "Any-Motion" $state
}

proc Blt_ZoomStack { graph } {
    blt::ZoomStack $graph
}

proc Blt_PrintKey { graph } {
    blt::PrintKey $graph
}

proc Blt_ClosestPoint { graph } {
    blt::ClosestPoint $graph
}

#
# The following procedures that reside in the "blt" namespace are
# supposed to be private.
#

proc blt::ActivateLegend { graph } {
    set elem [$graph legend get current]
    $graph legend activate $elem
}
proc blt::DeactivateLegend { graph } {
    set elem [$graph legend get current]
    $graph legend deactivate $elem
}

proc blt::HighlightLegend { graph } {
    set elem [$graph legend get current]
    set relief [$graph element cget $elem -labelrelief]
    if { $relief eq "flat" } {
	$graph element configure $elem -labelrelief raised
	$graph element activate $elem
    } else {
	$graph element configure $elem -labelrelief flat
	$graph element deactivate $elem
    }
}

proc blt::Crosshairs { graph {event "Any-Motion"} {state "on"}} {
    $graph crosshairs $state
    bind crosshairs-$graph <$event>   {
	%W crosshairs configure -position @%x,%y 
    }
    bind crosshairs-$graph <Leave>   {
	%W crosshairs off
    }
    bind crosshairs-$graph <Enter>   {
	%W crosshairs on
    }
    $graph crosshairs configure -color red
    if { $state eq "on" } {
	blt::AddBindTag $graph crosshairs-$graph
    } elseif { $state eq "off" } {
	blt::RemoveBindTag $graph crosshairs-$graph
    }
}

proc blt::InitStack { graph } {
    set ::blt::zoomInfo($graph,interval) 100
    set ::blt::zoomInfo($graph,afterId) 0
    set ::blt::zoomInfo($graph,A,x) {}
    set ::blt::zoomInfo($graph,A,y) {}
    set ::blt::zoomInfo($graph,B,x) {}
    set ::blt::zoomInfo($graph,B,y) {}
    set ::blt::zoomInfo($graph,stack) {}
    set ::blt::zoomInfo($graph,corner) A
}

proc blt::ZoomStack { graph {start "ButtonPress-1"} {reset "ButtonPress-3"} } {
    blt::InitStack $graph
    if { [info exists ::blt::zoomMod] } {
	set modifier $::blt::zoomMod
    } else {
	set modifier ""
    }
    bind zoom-$graph <${modifier}${start}> { blt::SetZoomPoint %W %x %y }
    bind zoom-$graph <${modifier}${reset}> { 
	if { [%W inside %x %y] } { 
	    blt::ResetZoom %W 
	}
    }
    blt::AddBindTag $graph zoom-$graph
}

proc blt::PrintKey { graph {event "Shift-ButtonRelease-3"} } {
    bind print-$graph <$event>  { Blt_PostScriptDialog %W }
    blt::AddBindTag $graph print-$graph
}

proc blt::ClosestPoint { graph {event "Control-ButtonPress-2"} } {
    bind closest-point-$graph <$event>  {
	blt::FindElement %W %x %y
    }
    blt::AddBindTag $graph closest-point-$graph
}

proc blt::AddBindTag { widget tag } {
    set oldTagList [bindtags $widget]
    if { [lsearch $oldTagList $tag] < 0 } {
	bindtags $widget [linsert $oldTagList 0  $tag]
    }
}

proc blt::RemoveBindTag { widget tag } {
    set oldTagList [bindtags $widget]
    set index [lsearch $oldTagList $tag]
    if { $index >= 0 } {
	bindtags $widget [lreplace $oldTagList $index $index]
    }
}

proc blt::FindElement { graph x y } {
    if { ![$graph element closest $x $y info -interpolate yes] } {
	beep
	return
    }
    # --------------------------------------------------------------
    # find(name)		- element Id
    # find(index)		- index of closest point
    # find(x) find(y)		- coordinates of closest point
    #				  or closest point on line segment.
    # find(dist)		- distance from sample coordinate
    # --------------------------------------------------------------
    set markerName "bltClosest_$info(name)"
    catch { $graph marker delete $markerName }
    $graph marker create text -coords { $info(x) $info(y) } \
	-name $markerName \
	-text "$info(name): $info(dist)\nindex $info(index)" \
	-anchor center -justify left \
	-yoffset 0 -bg {} 

    set coords [$graph invtransform $x $y]
    set nx [lindex $coords 0]
    set ny [lindex $coords 1]

    $graph marker create line -coords "$nx $ny $info(x) $info(y)" \
	-name line.$markerName 

    blt::FlashPoint $graph $info(name) $info(index) 10
    blt::FlashPoint $graph $info(name) [expr {$info(index) + 1}] 10
}

proc blt::FlashPoint { graph name index count } {
    if { ![winfo exists $graph] } {
	return
    }
    if { $count & 1 } {
        $graph element deactivate $name 
    } else {
        $graph element activate $name $index
    }
    incr count -1
    if { $count > 0 } {
	after 200 blt::FlashPoint $graph $name $index $count
    } else {
	eval $graph marker delete [$graph marker names "bltClosest_*"]
    }
}

proc blt::GetCoords { graph x y index } {
    if { [$graph cget -invertxy] } {
	set ::blt::zoomInfo($graph,$index,x) $y
	set ::blt::zoomInfo($graph,$index,y) $x
    } else {
	set ::blt::zoomInfo($graph,$index,x) $x
	set ::blt::zoomInfo($graph,$index,y) $y
    }
}

proc blt::MarkPoint { graph index } {
    set x [$graph xaxis invtransform $::blt::zoomInfo($graph,$index,x)]
    set y [$graph yaxis invtransform $::blt::zoomInfo($graph,$index,y)]
    set marker "zoomText_$index"
    set text [format "x=%.4g\ny=%.4g" $x $y] 
    if { [$graph marker exists $marker] } {
     	$graph marker configure $marker -coords { $x $y } -text $text 
    } else {
    	$graph marker create text -coords { $x $y } -name $marker \
	    -text $text -anchor center -bg {} -justify left
    }
}

proc blt::DestroyZoomTitle { graph } {
    if { $::blt::zoomInfo($graph,corner) eq "A" } {
	catch { $graph marker delete "zoomTitle" }
    }
}

proc blt::PopZoom { graph } {
    set zoomStack $::blt::zoomInfo($graph,stack)
    if { [llength $zoomStack] > 0 } {
	set cmd [lindex $zoomStack 0]
	set ::blt::zoomInfo($graph,stack) [lrange $zoomStack 1 end]
	eval $cmd
	blt::ZoomTitleLast $graph
	if { [catch {tk busy hold $graph}] } {
	    update idletasks
	} else {
	    update
	    catch {tk busy forget $graph}
	}
	after 2000 [list blt::DestroyZoomTitle $graph]
    } else {
	catch { $graph marker delete "zoomTitle" }
    }
}

# Push the old axis limits on the stack and set the new ones

proc blt::PushZoom { graph } {
    eval $graph marker delete [$graph marker names "zoom*"]
    if { [info exists ::blt::zoomInfo($graph,afterId)] } {
	after cancel $::blt::zoomInfo($graph,afterId)
    }
    set x1 $::blt::zoomInfo($graph,A,x)
    set y1 $::blt::zoomInfo($graph,A,y)
    set x2 $::blt::zoomInfo($graph,B,x)
    set y2 $::blt::zoomInfo($graph,B,y)

    if { ($x1 == $x2) || ($y1 == $y2) } { 
	# No delta, revert to start
	return
    }
    set cmd {}
    foreach margin { xaxis yaxis x2axis y2axis } {
	foreach axis [$graph $margin use] {
	    set min [$graph axis cget $axis -min] 
	    set max [$graph axis cget $axis -max]
	    set c [list $graph axis configure $axis -min $min -max $max]
	    append cmd "$c\n"
	}
    }
    if {[catch {
	foreach margin { xaxis x2axis } {
	    foreach axis [$graph $margin use] {
		set min [$graph axis invtransform $axis $x1]
		set max [$graph axis invtransform $axis $x2]
		if { $min > $max } { 
		    $graph axis configure $axis -min $max -max $min
		} else {
		    $graph axis configure $axis -min $min -max $max
		}
	    }
	}
	foreach margin { yaxis y2axis } {
	    foreach axis [$graph $margin use] {
		set min [$graph axis invtransform $axis $y1]
		set max [$graph axis invtransform $axis $y2]
		if { $min > $max } { 
		    $graph axis configure $axis -min $max -max $min
		} else {
		    $graph axis configure $axis -min $min -max $max
		}
	    }
	}
    }]} {
	eval $cmd
    } else {
	set ::blt::zoomInfo($graph,stack) [linsert $::blt::zoomInfo($graph,stack) 0 $cmd]
    }
    if { [catch {tk busy hold $graph}] } {
	update idletasks
    } else { 
	update
	catch {tk busy forget $graph}
    }
}

#
# This routine terminates either an existing zoom, or pops back to
# the previous zoom level (if no zoom is in progress).
#

proc blt::ResetZoom { graph } {
    if { ![info exists ::blt::zoomInfo($graph,corner)] } {
	blt::InitStack $graph 
    }
    eval $graph marker delete [$graph marker names "zoom*"]
    if { $::blt::zoomInfo($graph,corner) eq "A" } {
	# Reset the whole axis
	blt::PopZoom $graph
    } else {
	if { [info exists ::blt::zoomMod] } {
	    set modifier $::blt::zoomMod
	} else {
	    set modifier "Any-"
	}
	set ::blt::zoomInfo($graph,corner) A
	blt::RemoveBindTag $graph select-region-$graph
	blt::PopZoom $graph
    }
}

option add *zoomTitle.font	  -*-helvetica-medium-R-*-*-18-*-*-*-*-*-*-* 
option add *zoomTitle.shadow	  yellow4
option add *zoomTitle.foreground  yellow1
option add *zoomTitle.coords	  [list -Inf Inf]

proc blt::ZoomTitleNext { graph } {
    set level [expr {[llength $::blt::zoomInfo($graph,stack)] + 1}]
    if { [$graph cget -invertxy] } {
	set coords [list -Inf -Inf]
    } else {
	set coords [list -Inf Inf]
    }
    $graph marker create text -name "zoomTitle" -text "Zoom #$level" \
	-coords $coords -bindtags "" -anchor nw
}

proc blt::ZoomTitleLast { graph } {
    set level [llength $::blt::zoomInfo($graph,stack)]
    if { $level > 0 } {
     	$graph marker create text -name "zoomTitle" -anchor nw \
	    -text "Zoom #$level" 
    }
}

proc blt::SetZoomPoint { graph x y } {
    if { ![info exists ::blt::zoomInfo($graph,corner)] } {
	blt::InitStack $graph
    }
    blt::GetCoords $graph $x $y $::blt::zoomInfo($graph,corner)
    if { [info exists ::blt::zoomMod] } {
	set modifier $::blt::zoomMod
    } else {
	set modifier "Any-"
    }
    bind select-region-$graph <${modifier}Motion> { 
	blt::GetCoords %W %x %y B
	#blt::MarkPoint $graph B
	blt::Box %W
    }
    if { $::blt::zoomInfo($graph,corner) eq "A" } {
	if { ![$graph inside $x $y] } {
	    return
	}
	# First corner selected, start watching motion events
	#blt::MarkPoint $graph A
	blt::ZoomTitleNext $graph 
	blt::AddBindTag $graph select-region-$graph
	set ::blt::zoomInfo($graph,corner) B
    } else {
	# Delete the modal binding
	blt::RemoveBindTag $graph select-region-$graph
	blt::PushZoom $graph 
	set ::blt::zoomInfo($graph,corner) A
    }
}

option add *zoomOutline.dashes		4	
option add *zoomTitle.anchor		nw
option add *zoomOutline.lineWidth	2
option add *zoomOutline.xor		yes

proc blt::MarchingAnts { graph offset } {
    if { ![winfo exists $graph] } {
	return
    }
    incr offset
    if { [$graph marker exists zoomOutline] } {
	$graph marker configure zoomOutline -dashoffset $offset 
	set interval $::blt::zoomInfo($graph,interval)
	set id [after $interval [list blt::MarchingAnts $graph $offset]]
	set ::blt::zoomInfo($graph,afterId) $id
    }
}

proc blt::Box { graph } {
    if { $::blt::zoomInfo($graph,A,x) > $::blt::zoomInfo($graph,B,x) } { 
	set x1 [$graph xaxis invtransform $::blt::zoomInfo($graph,B,x)]
	set y1 [$graph yaxis invtransform $::blt::zoomInfo($graph,B,y)]
	set x2 [$graph xaxis invtransform $::blt::zoomInfo($graph,A,x)]
	set y2 [$graph yaxis invtransform $::blt::zoomInfo($graph,A,y)]
    } else {
	set x1 [$graph xaxis invtransform $::blt::zoomInfo($graph,A,x)]
	set y1 [$graph yaxis invtransform $::blt::zoomInfo($graph,A,y)]
	set x2 [$graph xaxis invtransform $::blt::zoomInfo($graph,B,x)]
	set y2 [$graph yaxis invtransform $::blt::zoomInfo($graph,B,y)]
    }
    set coords { $x1 $y1 $x2 $y1 $x2 $y2 $x1 $y2 $x1 $y1 }
    if { [$graph marker exists "zoomOutline"] } {
	$graph marker configure "zoomOutline" -coords $coords 
    } else {
	set X [lindex [$graph xaxis use] 0]
	set Y [lindex [$graph yaxis use] 0]
	$graph marker create line -coords $coords -name "zoomOutline" \
	    -mapx $X -mapy $Y
	set interval $::blt::zoomInfo($graph,interval)
	set id [after $interval [list blt::MarchingAnts $graph 0]]
	set ::blt::zoomInfo($graph,afterId) $id
    }
}

proc Blt_PostScriptDialog { graph } {
    set top $graph.top
    toplevel $top
    foreach var { center landscape maxpect preview decorations padx 
	pady paperwidth paperheight width height colormode } {
	set ::blt::postscript($graph.$var) [$graph postscript cget -$var]
    }
    set row 1
    set col 0
    label $top.title -text "PostScript Options"
    grid $top $top.title -columnspan 7
    foreach bool { center landscape maxpect preview decorations } {
	set w $top.$bool-label
	label $w -text "-$bool"
	grid $w -row $row -column 0 -anchor e -pady { 2 0 } -padx { 0 4 }
	set w $top.$bool-yes
	set var ::blt::postscript($graph.$bool)
	radiobutton $w -text "yes" -variable $var -value 1
	grid $w -row $row -column 1 -anchor w
	set w $top.$bool-no
	radiobutton $w -text "no" -variable $var -value 0
	grid $w -row $row -column 2 $w -anchor w
	incr row
    }
    label $top.modes -text "-colormode"
    grid $top.modes -row $row -anchor e -pady { 2 0 } -padx { 0 4 }
    set col 1
    foreach m { color greyscale } {
	set w $top.$m
	set var ::blt::postscript($graph.colormode)
	radiobutton $w -text $m -variable $var -value $m
	grid $w -row $row -column 3 -fill y -rowspan 6
	incr col
    }
    set row 1
    frame $top.sep -width 2 -bd 1 -relief sunken
    grid $top.sep -row 1 -column 3 -fill y -rspan 6
    set col 4
    foreach value { padx pady paperwidth paperheight width height } {
	set w $top.$value-label
	label $w -text "-$value"
	grid $w -row $row -column 4 -anchor e -pady { 2 0 } -padx { 0 4 }
	set w $top.$value-entry
	set var ::blt::postscript($graph.$value)
	entry $w -textvariable $var -width 8
	grid $w -row $row -column 5 -anchor w -padx 8
	incr row
    }
    grid columnconfigure $top 3 -minsize .125i
    button $top.cancel -text "Cancel" -command [list destroy $top]
    grid $top.cancel - - -row $row -column 0 -pady 2
    button $top.reset -text "Reset" -command [list destroy $top]
    #grid $top.reset -row $row -column 1
    button $top.print -text "Print" -command [list blt::ResetPostScript $graph]
    grid $top.print - -row $row -column 4 -pady 2
}

proc blt::ResetPostScript { graph } {
    foreach var { center landscape maxpect preview decorations padx 
	pady paperwidth paperheight width height colormode } {
	set old [$graph postscript cget -$var]
	if {[catch {
	    $graph postscript configure -$var \
		[set ::blt::postscript($graph.$var)]
	}]} {
	    $graph postscript configure -$var $old
	    set $graph.$var $old
	}
    }
    set filename "out.ps"
    $graph postscript output $filename
}

