# CBOR Package for Tcl

A pure Tcl implementation of CBOR (Concise Binary Object Representation) as defined in [RFC 8949](https://www.rfc-editor.org/rfc/rfc8949.html).

## Features

- **Pure Tcl** - No compilation required, works on any Tcl installation
- **Complete Implementation** - Supports all major CBOR data types
- **Simple API** - Just two commands: `cbor::encode` and `cbor::decode`
- **RFC 8949 Compliant** - Follows the official CBOR specification

## Supported Data Types

- **Integers** - Positive and negative integers (up to 64-bit)
- **Floats** - Single and double precision floating-point numbers
- **Strings** - UTF-8 text strings
- **Byte Strings** - Raw binary data
- **Arrays** - Ordered lists of values
- **Maps** - Key-value dictionaries
- **Booleans** - `true` and `false`
- **Null** - `null` value
- **Tags** - Semantic tags (decoded but not interpreted)

## Installation

1. Copy the CBOR package directory to a location in your Tcl `auto_path`
2. Or add the package directory to your `auto_path`:

```tcl
lappend auto_path /path/to/CBOR
```

## Usage

### Basic Example

```tcl
package require cbor

# Encode a value
set data [cbor::encode "Hello, World!"]
puts "Encoded: [binary encode hex $data]"

# Decode it back
set original [cbor::decode $data]
puts "Decoded: $original"
```

### Encoding Examples

```tcl
package require cbor

# Encode integers
set encoded [cbor::encode 42]
set encoded [cbor::encode -100]

# Encode strings
set encoded [cbor::encode "Hello, CBOR!"]

# Encode booleans and null
set encoded [cbor::encode true]
set encoded [cbor::encode false]
set encoded [cbor::encode null]

# Encode arrays (lists)
set encoded [cbor::encode [list 1 2 3 4 5]]
set encoded [cbor::encode [list "a" "b" "c"]]

# Encode nested arrays
set encoded [cbor::encode [list 1 [list 2 3] [list 4 5]]]

# Encode maps (dicts)
set encoded [cbor::encode [dict create "name" "Alice" "age" 30]]

# Encode complex structures
set user [dict create \
    "name" "Bob" \
    "age" 25 \
    "emails" [list "bob@example.com" "bob@work.com"] \
    "active" true \
]
set encoded [cbor::encode $user]
```

### Decoding Examples

```tcl
package require cbor

# Decode from binary data
set data [binary decode hex "01"]  ;# Integer 1
set value [cbor::decode $data]
puts $value  ;# Output: 1

# Decode a string
set data [binary decode hex "6568656c6c6f"]  ;# "hello"
set value [cbor::decode $data]
puts $value  ;# Output: hello

# Decode an array
set data [binary decode hex "83010203"]  ;# [1, 2, 3]
set value [cbor::decode $data]
puts $value  ;# Output: 1 2 3

# Decode a map
set data [binary decode hex "a2616101616202"]  ;# {"a": 1, "b": 2}
set value [cbor::decode $data]
puts $value  ;# Output: a 1 b 2
```

### Roundtrip Example

```tcl
package require cbor

# Create a complex data structure
set original [dict create \
    "users" [list \
        [dict create "name" "Alice" "age" 30] \
        [dict create "name" "Bob" "age" 25] \
    ] \
    "count" 2 \
    "active" true \
]

# Encode to CBOR
set encoded [cbor::encode $original]
puts "Encoded size: [string length $encoded] bytes"

# Decode back
set decoded [cbor::decode $encoded]

# Verify they match
if {$decoded eq $original} {
    puts "Roundtrip successful!"
}
```

### Working with Binary Data

```tcl
package require cbor

# Encode and save to file
set data [dict create "message" "Hello" "value" 42]
set encoded [cbor::encode $data]

set fp [open "data.cbor" wb]
puts -nonewline $fp $encoded
close $fp

# Read and decode from file
set fp [open "data.cbor" rb]
set encoded [read $fp]
close $fp

set decoded [cbor::decode $encoded]
puts "Decoded: $decoded"
```

## API Reference

### `cbor::encode value`

Encodes a Tcl value into CBOR binary format.

**Parameters:**
- `value` - The Tcl value to encode (integer, string, list, dict, boolean, null, float)

**Returns:**
- Binary string containing the CBOR-encoded data

**Example:**
```tcl
set encoded [cbor::encode [list 1 2 3]]
```

### `cbor::decode data`

Decodes CBOR binary data into a Tcl value.

**Parameters:**
- `data` - Binary string containing CBOR-encoded data

**Returns:**
- The decoded Tcl value

**Example:**
```tcl
set value [cbor::decode $encoded]
```

## Type Mapping

### Tcl to CBOR

| Tcl Type | CBOR Type | Notes |
|----------|-----------|-------|
| Integer | Unsigned/Negative Integer | Automatic based on sign |
| Float | Float (64-bit) | Double precision |
| String | Text String | UTF-8 encoded |
| List | Array | Ordered sequence |
| Dict | Map | Key-value pairs |
| `true`/`false` | Boolean | Simple values |
| `null` | Null | Simple value |

### CBOR to Tcl

| CBOR Type | Tcl Type | Notes |
|-----------|----------|-------|
| Unsigned Integer | Integer | |
| Negative Integer | Integer | |
| Float | Float | All precisions converted to double |
| Text String | String | UTF-8 decoded |
| Byte String | String | Raw bytes |
| Array | List | |
| Map | Dict | |
| Boolean | `true`/`false` | String literals |
| Null | `null` | String literal |
| Undefined | `undefined` | String literal |

## Testing

Run the test suite to verify the package works correctly:

```bash
tclsh test.tcl
```

The test suite includes:
- Integer encoding/decoding (positive and negative)
- Boolean and null values
- String handling (including UTF-8)
- Array operations
- Map (dictionary) operations
- Float handling
- Complex nested structures
- Roundtrip verification

## Limitations

- **Half-precision floats** - Decoded but with reduced precision
- **Indefinite-length encoding** - Supported for decoding only
- **Tags** - Decoded but not semantically interpreted
- **Performance** - Pure Tcl is slower than C extensions for large data

## Performance Considerations

For small to medium-sized data structures, the pure Tcl implementation provides adequate performance. For high-performance requirements with large data volumes, consider:

1. Caching encoded results when possible
2. Processing data in chunks
3. Using binary channels for I/O operations
4. Implementing a C extension for critical paths

## RFC 8949 Compliance

This implementation follows RFC 8949 (CBOR) and supports:
- Major types 0-7
- Additional information encoding
- Definite and indefinite length items
- Simple values (false, true, null, undefined)
- Floating-point numbers (half, single, double precision)

## License

This package is provided as-is for use in Tcl projects.

## Contributing

Contributions are welcome! Please ensure:
- All tests pass
- New features include tests
- Code follows Tcl conventions
- Documentation is updated

## Version History

- **1.0** - Initial release
  - Complete CBOR encode/decode implementation
  - Support for all major data types
  - Pure Tcl implementation
  - Comprehensive test suite
