# Quick Start Guide

## Installation

Add the CBOR package directory to your Tcl `auto_path`:

```tcl
lappend auto_path /path/to/CBOR
package require cbor
```

## Basic Usage

### Encoding

```tcl
# Encode any Tcl value
set encoded [cbor::encode $value]
```

### Decoding

```tcl
# Decode CBOR binary data
set decoded [cbor::decode $encoded]
```

## Common Patterns

### Working with Files

```tcl
# Save to file
set data [dict create "key" "value"]
set encoded [cbor::encode $data]
set fp [open "data.cbor" wb]
puts -nonewline $fp $encoded
close $fp

# Load from file
set fp [open "data.cbor" rb]
set encoded [read $fp]
close $fp
set data [cbor::decode $encoded]
```

### Empty Collections

Due to Tcl limitations where empty string, empty list, and empty dict are all represented as `""`, use explicit functions for empty collections:

```tcl
# Empty array
set encoded [cbor::encodeArray [list]]

# Empty map
set encoded [cbor::encodeMap [dict create]]
```

### Type Detection

The package automatically detects types:
- Integers: `42`, `-100`
- Floats: `3.14`, `-2.5`
- Strings: `"hello"`, `"world"`
- Booleans: `true`, `false`
- Null: `null`
- Arrays: `[list 1 2 3]`
- Maps: `[dict create "key" "value"]`

## Running Tests

```bash
tclsh test.tcl
```

## Running Examples

```bash
tclsh example.tcl
```

## API Reference

### `cbor::encode value`
Encodes a Tcl value to CBOR binary format.

### `cbor::decode data`
Decodes CBOR binary data to a Tcl value.

### `cbor::encodeArray list`
Explicitly encodes a list as a CBOR array (useful for empty arrays).

### `cbor::encodeMap dict`
Explicitly encodes a dict as a CBOR map (useful for empty maps).

## Supported Types

✓ Integers (positive and negative, up to 64-bit)  
✓ Floating-point numbers (single and double precision)  
✓ Text strings (UTF-8)  
✓ Byte strings  
✓ Arrays (lists)  
✓ Maps (dictionaries)  
✓ Booleans  
✓ Null values  
✓ Nested structures  

## Notes

- **Pure Tcl**: No compilation required
- **RFC 8949 Compliant**: Follows official CBOR specification
- **Portable**: Works on any Tcl 8.5+ installation
- **Tested**: Comprehensive test suite included
