# cbor.tcl - Pure Tcl implementation of CBOR (RFC 8949)
# Concise Binary Object Representation
#
# Package: cbor
# Version: 1.0

package provide cbor 1.0

namespace eval ::cbor {
    namespace export encode decode encodeArray encodeMap

    # CBOR Major Types
    variable MAJOR_UINT 0      ;# Unsigned integer
    variable MAJOR_NINT 1      ;# Negative integer
    variable MAJOR_BYTES 2     ;# Byte string
    variable MAJOR_TEXT 3      ;# Text string
    variable MAJOR_ARRAY 4     ;# Array
    variable MAJOR_MAP 5       ;# Map
    variable MAJOR_TAG 6       ;# Semantic tag
    variable MAJOR_SIMPLE 7    ;# Simple values and floats

    # Additional Information values
    variable AI_UINT8 24
    variable AI_UINT16 25
    variable AI_UINT32 26
    variable AI_UINT64 27
    variable AI_INDEFINITE 31

    # Simple values
    variable SIMPLE_FALSE 20
    variable SIMPLE_TRUE 21
    variable SIMPLE_NULL 22
    variable SIMPLE_UNDEFINED 23
}

# Encode a Tcl value to CBOR binary format
proc ::cbor::encode {value} {
    variable MAJOR_UINT
    variable MAJOR_NINT
    variable MAJOR_TEXT
    variable MAJOR_ARRAY
    variable MAJOR_MAP
    variable MAJOR_SIMPLE
    variable SIMPLE_FALSE
    variable SIMPLE_TRUE
    variable SIMPLE_NULL

    # Detect type and encode accordingly
    set type [detectType $value]

    switch -exact -- $type {
        "null" {
            return [encodeSimple $SIMPLE_NULL]
        }
        "boolean" {
            if {$value} {
                return [encodeSimple $SIMPLE_TRUE]
            } else {
                return [encodeSimple $SIMPLE_FALSE]
            }
        }
        "integer" {
            return [encodeInteger $value]
        }
        "float" {
            return [encodeFloat $value]
        }
        "string" {
            return [encodeText $value]
        }
        "list" {
            return [encodeArray $value]
        }
        "dict" {
            return [encodeMap $value]
        }
        default {
            # Default to text string
            return [encodeText $value]
        }
    }
}

# Decode CBOR binary format to Tcl value
proc ::cbor::decode {data} {
    set pos 0
    set result [decodeValue data pos]
    return $result
}

# Detect the type of a Tcl value
proc ::cbor::detectType {value} {
    # Check for null (only the literal "null")
    if {$value eq "null"} {
        return "null"
    }

    # Check for boolean
    if {$value eq "true" || $value eq "false"} {
        return "boolean"
    }

    # Check for empty string - in Tcl, empty string, empty list, and empty dict
    # are all represented as "". We can't distinguish them, so we need to
    # default to one. We'll default to empty string for "" literal.
    # Note: This is a limitation of pure Tcl - users should use explicit
    # encode functions if they need to encode empty collections.
    if {$value eq ""} {
        return "string"
    }

    # Check for integer
    if {[string is integer -strict $value]} {
        return "integer"
    }

    # Check for float
    if {[string is double -strict $value]} {
        return "float"
    }

    # Check for dict (even number of elements and valid dict)
    # A dict should have string keys and be a valid dict structure
    if {[llength $value] > 0 && [llength $value] % 2 == 0} {
        if {![catch {dict size $value}]} {
            # Additional check: if all keys are strings (not pure numbers)
            # and it's a proper dict, treat as dict
            set isDict 1
            set hasStringKey 0
            foreach {key val} $value {
                if {![string is integer -strict $key] && ![string is double -strict $key]} {
                    set hasStringKey 1
                }
            }
            # Only treat as dict if it has at least one non-numeric key
            if {$hasStringKey} {
                return "dict"
            }
        }
    }

    # Check for list (more than one element or looks like a list)
    if {[llength $value] > 1} {
        return "list"
    }

    # Check if it's an empty list
    if {[llength $value] == 0 && [catch {dict size $value}]} {
        return "list"
    }

    # Default to string
    return "string"
}

# Encode an integer value
proc ::cbor::encodeInteger {value} {
    variable MAJOR_UINT
    variable MAJOR_NINT

    if {$value >= 0} {
        return [encodeUnsigned $MAJOR_UINT $value]
    } else {
        # Negative integers: encoded as -1 - value
        set encoded [expr {-1 - $value}]
        return [encodeUnsigned $MAJOR_NINT $encoded]
    }
}

# Encode an unsigned integer with major type
proc ::cbor::encodeUnsigned {majorType value} {
    variable AI_UINT8
    variable AI_UINT16
    variable AI_UINT32
    variable AI_UINT64

    set major [expr {$majorType << 5}]

    if {$value < 24} {
        # Encode in initial byte
        return [binary format c [expr {$major | $value}]]
    } elseif {$value < 256} {
        # 1-byte uint8 follows
        return [binary format cc [expr {$major | $AI_UINT8}] $value]
    } elseif {$value < 65536} {
        # 2-byte uint16 follows
        return [binary format cS [expr {$major | $AI_UINT16}] $value]
    } elseif {$value < 4294967296} {
        # 4-byte uint32 follows
        return [binary format cI [expr {$major | $AI_UINT32}] $value]
    } else {
        # 8-byte uint64 follows
        return [binary format cW [expr {$major | $AI_UINT64}] $value]
    }
}

# Encode a text string
proc ::cbor::encodeText {text} {
    variable MAJOR_TEXT

    set bytes [encoding convertto utf-8 $text]
    set length [string length $bytes]
    set header [encodeUnsigned $MAJOR_TEXT $length]

    return "${header}${bytes}"
}

# Encode a byte string
proc ::cbor::encodeBytes {bytes} {
    variable MAJOR_BYTES

    set length [string length $bytes]
    set header [encodeUnsigned $MAJOR_BYTES $length]

    return "${header}${bytes}"
}

# Encode an array (list)
proc ::cbor::encodeArray {list} {
    variable MAJOR_ARRAY

    set length [llength $list]
    set result [encodeUnsigned $MAJOR_ARRAY $length]

    foreach item $list {
        append result [encode $item]
    }

    return $result
}

# Encode a map (dict)
proc ::cbor::encodeMap {dict} {
    variable MAJOR_MAP

    set length [expr {[dict size $dict]}]
    set result [encodeUnsigned $MAJOR_MAP $length]

    dict for {key value} $dict {
        append result [encode $key]
        append result [encode $value]
    }

    return $result
}

# Encode a simple value
proc ::cbor::encodeSimple {value} {
    variable MAJOR_SIMPLE

    set major [expr {$MAJOR_SIMPLE << 5}]
    return [binary format c [expr {$major | $value}]]
}

# Encode a float (double precision)
proc ::cbor::encodeFloat {value} {
    variable MAJOR_SIMPLE
    variable AI_UINT32
    variable AI_UINT64

    set major [expr {$MAJOR_SIMPLE << 5}]

    # Use double precision (8 bytes) - additional info 27
    return [binary format cQ [expr {$major | $AI_UINT64}] $value]
}

# Decode a CBOR value from binary data
proc ::cbor::decodeValue {dataVar posVar} {
    upvar $dataVar data
    upvar $posVar pos

    variable MAJOR_UINT
    variable MAJOR_NINT
    variable MAJOR_BYTES
    variable MAJOR_TEXT
    variable MAJOR_ARRAY
    variable MAJOR_MAP
    variable MAJOR_TAG
    variable MAJOR_SIMPLE

    # Read initial byte
    binary scan [string index $data $pos] cu initialByte
    incr pos

    set majorType [expr {($initialByte >> 5) & 0x07}]
    set additionalInfo [expr {$initialByte & 0x1f}]

    # For major type 7 (simple/float), additional info is the format, not a length
    # So we don't call readArgument for it
    if {$majorType != 7} {
        set argument [readArgument data pos $additionalInfo]
    }

    switch -exact -- $majorType {
        0 {
            # Unsigned integer
            return $argument
        }
        1 {
            # Negative integer
            return [expr {-1 - $argument}]
        }
        2 {
            # Byte string
            return [readBytes data pos $argument]
        }
        3 {
            # Text string
            set bytes [readBytes data pos $argument]
            return [encoding convertfrom utf-8 $bytes]
        }
        4 {
            # Array
            return [decodeArray data pos $argument]
        }
        5 {
            # Map
            return [decodeMap data pos $argument]
        }
        6 {
            # Tag - skip tag and decode tagged value
            return [decodeValue data pos]
        }
        7 {
            # Simple value or float
            return [decodeSimple $additionalInfo data pos]
        }
        default {
            error "Unknown major type: $majorType"
        }
    }
}

# Read argument value based on additional info
proc ::cbor::readArgument {dataVar posVar additionalInfo} {
    upvar $dataVar data
    upvar $posVar pos

    variable AI_UINT8
    variable AI_UINT16
    variable AI_UINT32
    variable AI_UINT64

    if {$additionalInfo < 24} {
        return $additionalInfo
    }

    switch -exact -- $additionalInfo {
        24 {
            # 1-byte uint8
            binary scan [string index $data $pos] cu value
            incr pos
            return $value
        }
        25 {
            # 2-byte uint16
            binary scan [string range $data $pos [expr {$pos + 1}]] Su value
            incr pos 2
            return $value
        }
        26 {
            # 4-byte uint32
            binary scan [string range $data $pos [expr {$pos + 3}]] Iu value
            incr pos 4
            return $value
        }
        27 {
            # 8-byte uint64
            binary scan [string range $data $pos [expr {$pos + 7}]] Wu value
            incr pos 8
            return $value
        }
        31 {
            # Indefinite length
            return -1
        }
        default {
            error "Invalid additional info: $additionalInfo"
        }
    }
}

# Read a specified number of bytes
proc ::cbor::readBytes {dataVar posVar length} {
    upvar $dataVar data
    upvar $posVar pos

    set bytes [string range $data $pos [expr {$pos + $length - 1}]]
    incr pos $length
    return $bytes
}

# Decode an array
proc ::cbor::decodeArray {dataVar posVar length} {
    upvar $dataVar data
    upvar $posVar pos

    set result [list]

    if {$length == -1} {
        # Indefinite length - read until break
        while {1} {
            binary scan [string index $data $pos] cu byte
            if {$byte == 0xff} {
                incr pos
                break
            }
            lappend result [decodeValue data pos]
        }
    } else {
        for {set i 0} {$i < $length} {incr i} {
            lappend result [decodeValue data pos]
        }
    }

    return $result
}

# Decode a map
proc ::cbor::decodeMap {dataVar posVar length} {
    upvar $dataVar data
    upvar $posVar pos

    set result [dict create]

    if {$length == -1} {
        # Indefinite length - read until break
        while {1} {
            binary scan [string index $data $pos] cu byte
            if {$byte == 0xff} {
                incr pos
                break
            }
            set key [decodeValue data pos]
            set value [decodeValue data pos]
            dict set result $key $value
        }
    } else {
        for {set i 0} {$i < $length} {incr i} {
            set key [decodeValue data pos]
            set value [decodeValue data pos]
            dict set result $key $value
        }
    }

    return $result
}

# Decode simple values and floats
proc ::cbor::decodeSimple {additionalInfo dataVar posVar} {
    upvar $dataVar data
    upvar $posVar pos

    variable SIMPLE_FALSE
    variable SIMPLE_TRUE
    variable SIMPLE_NULL
    variable SIMPLE_UNDEFINED

    switch -exact -- $additionalInfo {
        20 {
            return false
        }
        21 {
            return true
        }
        22 {
            return null
        }
        23 {
            return undefined
        }
        25 {
            # Half-precision float (16-bit) - convert to double
            binary scan [string range $data $pos [expr {$pos + 1}]] Su bits
            incr pos 2
            return [halfFloatToDouble $bits]
        }
        26 {
            # Single-precision float (32-bit)
            binary scan [string range $data $pos [expr {$pos + 3}]] R floatValue
            incr pos 4
            return $floatValue
        }
        27 {
            # Double-precision float (64-bit)
            binary scan [string range $data $pos [expr {$pos + 7}]] Q floatValue
            incr pos 8
            return $floatValue
        }
        31 {
            # Break stop code
            return "break"
        }
        default {
            # Simple value
            return $additionalInfo
        }
    }
}

# Convert half-precision float to double (simplified)
proc ::cbor::halfFloatToDouble {bits} {
    set sign [expr {($bits >> 15) & 0x1}]
    set exponent [expr {($bits >> 10) & 0x1f}]
    set fraction [expr {$bits & 0x3ff}]

    if {$exponent == 0} {
        if {$fraction == 0} {
            return [expr {$sign ? -0.0 : 0.0}]
        }
        # Subnormal
        set value [expr {$fraction / 1024.0 / 16384.0}]
    } elseif {$exponent == 31} {
        if {$fraction == 0} {
            return [expr {$sign ? "-Inf" : "Inf"}]
        }
        return "NaN"
    } else {
        # Normal
        set value [expr {(1.0 + $fraction / 1024.0) * pow(2, $exponent - 15)}]
    }

    return [expr {$sign ? -$value : $value}]
}

# Public API
namespace eval ::cbor {
    namespace export encode decode encodeArray encodeMap
}
