#!/usr/bin/env tclsh
# Test suite for CBOR package

# Add current directory to auto_path to find the package
lappend auto_path [file dirname [info script]]

package require cbor

# Test counter
set testNum 0
set passCount 0
set failCount 0

proc test {name script expected} {
    global testNum passCount failCount
    incr testNum

    puts -nonewline "Test $testNum: $name ... "

    if {[catch {uplevel 1 $script} result]} {
        puts "ERROR: $result"
        incr failCount
        return
    }

    if {$result eq $expected} {
        puts "PASS"
        incr passCount
    } else {
        puts "FAIL"
        puts "  Expected: $expected"
        puts "  Got:      $result"
        incr failCount
    }
}

proc testRoundtrip {name value} {
    global testNum passCount failCount
    incr testNum

    puts -nonewline "Test $testNum: $name (roundtrip) ... "

    if {[catch {
        set encoded [cbor::encode $value]
        set decoded [cbor::decode $encoded]
    } result]} {
        puts "ERROR: $result"
        incr failCount
        return
    }

    if {$decoded eq $value} {
        puts "PASS"
        incr passCount
    } else {
        puts "FAIL"
        puts "  Original: $value"
        puts "  Decoded:  $decoded"
        incr failCount
    }
}

proc testBinary {name value expectedHex} {
    global testNum passCount failCount
    incr testNum

    puts -nonewline "Test $testNum: $name (binary) ... "

    if {[catch {
        set encoded [cbor::encode $value]
        set hex [binary encode hex $encoded]
    } result]} {
        puts "ERROR: $result"
        incr failCount
        return
    }

    if {$hex eq $expectedHex} {
        puts "PASS"
        incr passCount
    } else {
        puts "FAIL"
        puts "  Expected: $expectedHex"
        puts "  Got:      $hex"
        incr failCount
    }
}

puts "=== CBOR Package Test Suite ==="
puts ""

# Test integers
puts "--- Integer Tests ---"
testBinary "Integer 0" 0 "00"
testBinary "Integer 1" 1 "01"
testBinary "Integer 10" 10 "0a"
testBinary "Integer 23" 23 "17"
testBinary "Integer 24" 24 "1818"
testBinary "Integer 100" 100 "1864"
testBinary "Integer 1000" 1000 "1903e8"
testBinary "Integer -1" -1 "20"
testBinary "Integer -10" -10 "29"
testBinary "Integer -100" -100 "3863"

testRoundtrip "Roundtrip integer 0" 0
testRoundtrip "Roundtrip integer 42" 42
testRoundtrip "Roundtrip integer -42" -42
testRoundtrip "Roundtrip integer 1000000" 1000000

puts ""

# Test booleans and null
puts "--- Boolean and Null Tests ---"
testBinary "Boolean false" false "f4"
testBinary "Boolean true" true "f5"
testBinary "Null" null "f6"

testRoundtrip "Roundtrip true" true
testRoundtrip "Roundtrip false" false
testRoundtrip "Roundtrip null" null

puts ""

# Test strings
puts "--- String Tests ---"
testBinary "Empty string" "" "60"
testBinary "String 'a'" "a" "6161"
testBinary "String 'IETF'" "IETF" "6449455446"
testBinary "String with quotes" "\"test\"" "66227465737422"

testRoundtrip "Roundtrip string 'hello'" "hello"
testRoundtrip "Roundtrip string 'Hello, World!'" "Hello, World!"
testRoundtrip "Roundtrip UTF-8 string" "Hello 世界"

puts ""

# Test arrays
puts "--- Array Tests ---"
# Note: Use cbor::encodeArray for empty arrays since Tcl can't distinguish "" from [list]
test "Empty array" {binary encode hex [cbor::encodeArray [list]]} "80"
testBinary "Array \[1, 2, 3\]" [list 1 2 3] "83010203"
testBinary "Array \[1, \[2, 3\], \[4, 5\]\]" [list 1 [list 2 3] [list 4 5]] "8301820203820405"

testRoundtrip "Roundtrip array \[1, 2, 3\]" [list 1 2 3]
testRoundtrip "Roundtrip nested array" [list 1 [list 2 3] 4]
testRoundtrip "Roundtrip mixed array" [list 1 "hello" true]

puts ""

# Test maps (dicts)
puts "--- Map Tests ---"
# Note: Use cbor::encodeMap for empty maps since Tcl can't distinguish "" from [dict create]
test "Empty map" {binary encode hex [cbor::encodeMap [dict create]]} "a0"

set simpleDict [dict create "a" 1 "b" 2]
testRoundtrip "Roundtrip simple dict" $simpleDict

set nestedDict [dict create "name" "John" "age" 30 "active" true]
testRoundtrip "Roundtrip nested dict" $nestedDict

puts ""

# Test floats
puts "--- Float Tests ---"
testRoundtrip "Roundtrip float 3.14" 3.14
testRoundtrip "Roundtrip float -2.5" -2.5
testRoundtrip "Roundtrip float 0.0" 0.0

puts ""

# Test complex structures
puts "--- Complex Structure Tests ---"
set complexList [list \
    [dict create "name" "Alice" "age" 30] \
    [dict create "name" "Bob" "age" 25] \
]
testRoundtrip "Roundtrip list of dicts" $complexList

set complexDict [dict create \
    "users" [list "Alice" "Bob" "Charlie"] \
    "count" 3 \
    "active" true \
]
testRoundtrip "Roundtrip dict with list" $complexDict

puts ""

# Summary
puts "=== Test Summary ==="
puts "Total tests: $testNum"
puts "Passed: $passCount"
puts "Failed: $failCount"

if {$failCount == 0} {
    puts "\nAll tests passed! ✓"
    exit 0
} else {
    puts "\nSome tests failed."
    exit 1
}
