/*
 * Modbus-RTU Arduino example using Modbus library from
 * https://github.com/andresarmento/modbus-arduino
 */

#include <EEPROM.h>
#include <Modbus.h>
#include <ModbusSerial.h>

/*
 * Modbus address layout
 *
 *  Hreg
 *    0/1       millisecond counter 32 bits
 *    2..13     PWM outputs
 *    14/15     second counter 32 bits
 *    16/17     digital inputs as bitfield 22..53
 *    18/19     loop time milliseconds 32 bits
 *    20        input mode 2..13
 *    21        input pullup mode 2..13
 *    22        digital inputs as bitfield 2..13
 *    31        EEPROM command (0x55aa = save, 0xaa55 = clear)
 *    32..47    analog inputs
 *    48..63    analog inputs (averaged)
 *    64..79    digital inputs 2..13 (not on MEGA)
 *
 *  Ireg (only on MEGA)
 *    0..15     analog inputs
 *    16/17     digital inputs 22..53
 *    18..29    PWM outputs
 *    30/31     second counter 32 bits
 *    32        digital inputs 2..13
 *    48..63    analog inputs (averaged)
 *
 *  Coil (only on MEGA)
 *    0..31     digital inputs 22..53
 *    32..63    millisecond counter 32 bits
 *
 *  Ists (only on MEGA)
 *    0..31     digital inputs 22..53
 *    32..63    millisecond counter 32 bits
 */

/*
 * For RS-485 use pin 21 for TX control
 *
 *   ModbusSerial mb(Serial, 10, 21);
 */

ModbusSerial mb(Serial, 10, -1);
                     /* ^^--------- Slave address 10 */
                     /*     ^^----- No TX control */
long ms[3];
int ai_avg[16];

#ifdef __AVR_ATmega2560__
#define MAX_ANALOG_IN 16
#define MAX_DIGITAL_IN 32
#define MAX_DIGITAL_OUT 12
#else /* UNO, NANO, etc */
#define MAX_ANALOG_IN 6
#define MAX_DIGITAL_IN 0
#define MAX_DIGITAL_OUT 12
#endif

void
setup()
{
    int i, mode0, mode1;
    bool eok = false;

    /* 115200 baud, no parity, 8 data bits, 1 stop bit */
    mb.config(115200);
    Serial.begin(115200, SERIAL_8N1);

    /*
     * For RS-485 use pin 21 for TX control
     *
     *    pinMode(21, OUTPUT);
     */

    /* Modbus holding regs, coils, input bits/regs 0..64 */
    for (i = 0; i < 64; i++) {
        mb.addHreg(i, 0);
#ifdef __AVR_ATmega2560__
        mb.addCoil(i, false);
        mb.addIsts(i, false);
        mb.addIreg(i, 0);
#endif
    }
#ifndef __AVR_ATmega2560__
    for (; i < 80; i++) {
        mb.addHreg(i, 0);
    }
#endif
    ms[0] = millis();
    ms[1] = ms[0] + 10;
    ms[1] += 10 - ms[1] % 10;
    ms[2] = 0;
    for (i = 0; i < 16; i++) {
        ai_avg[i] = 0;
    }
    mode0 = 0x3ffc;
    mode1 = 0x0000;
    eok = (EEPROM.read(0) == 0xaa);
    if (eok) {
        mb.Hreg(15, (EEPROM.read(1) << 8L) | EEPROM.read(2));
        mb.Hreg(16, (EEPROM.read(3) << 8L) | EEPROM.read(4));
        mode0 = (EEPROM.read(5) << 8) | EEPROM.read(6);
        mode1 = (EEPROM.read(7) << 8) | EEPROM.read(8);
        mb.Hreg(31, 1);
    }
    mb.Hreg(20, mode0);
    mb.Hreg(21, mode1);
    for (i = 2; i < 2 + MAX_DIGITAL_OUT; i++) {
        if (mode1 & (1 << i)) {
            pinMode(i, INPUT_PULLUP);
            continue;
        }
        if (mode0 & (1 << i)) {
            pinMode(i, INPUT);
            continue;
        }
        if (!eok) {
            analogWrite(i, 0);
        }
    }
    if (eok) {
        for (i = 2; i < 2 + MAX_DIGITAL_OUT; i++) {
            int k;

            if ((mode0 | mode1) & (1 << i)) {
                continue;
            }
            k = (EEPROM.read(5 + i * 2) << 8) | EEPROM.read(6 + i * 2);
            mb.Hreg(i, k);
            analogWrite(i, k);
#ifdef __AVR_ATmega2560__
            mb.Ireg(i + 16, k);
#endif
        }
    }
}

/* See https://github.com/avandalen/avdweb_AnalogReadFast/blob/master/avdweb_AnalogReadFast.h */

#ifdef ARDUINO_ARCH_AVR
static int inline
analogReadFast(byte ADCpin)
{
    byte ADCSRAoriginal = ADCSRA;
    int adc;

    ADCSRA = (ADCSRA & 0xF8) | 4;
    adc = analogRead(ADCpin);
    ADCSRA = ADCSRAoriginal;
    return adc;
}
#else
#define analogReadFast(a) analogRead(a)
#endif

void
save(byte clr)
{
    int i, k;
    if (clr) {
        EEPROM.update(0, 0);
        mb.Hreg(31, 0);
        return;
    }
    k = mb.Hreg(20);
    EEPROM.update(5, k >> 8);
    EEPROM.update(6, k);
    k = mb.Hreg(21);
    EEPROM.update(7, k >> 8);
    EEPROM.update(8, k);
    for (i = 2; i < 2 + MAX_DIGITAL_OUT; i++) {
        k = mb.Hreg(i);
        EEPROM.update(5 + i * 2, k >> 8);
        EEPROM.update(6 + i * 2, k);
    }
    EEPROM.update(0, 0xaa);
    k = mb.Hreg(15);
    EEPROM.update(1, k >> 8);
    EEPROM.update(2, k);
    k = mb.Hreg(16);
    EEPROM.update(3, k >> 8);
    EEPROM.update(4, k);
    mb.Hreg(31, 1);
}

void
loop()
{
    long t, l, d, tmp;
    int i, k, mode0, mode1;
    bool b, x = false;

    t = millis();
    d = t - ms[0];
    if (d) {
        ms[0] = t;
        /* Increment millisecond counter */
        tmp = mb.Hreg(1);
        tmp |= (long) mb.Hreg(0) << 16;
        tmp += d;
        mb.Hreg(1, tmp);
        tmp = tmp >> 16;
        mb.Hreg(0, tmp);
#ifdef __AVR_ATmega2560__
        for (i = 0; i < 32; i++) {
            k = i + 32;
            mb.Coil(k, t & 1);
            mb.Ists(k, t & 1);
            t >>= 1;
        }
#endif
        ms[2] += d;
        /* Increment second counter */
        if (ms[2] >= 1000L) {
            ms[2] = 0;
            tmp = mb.Hreg(15);
            tmp |= (long) mb.Hreg(14) << 16;
            tmp++;
            mb.Hreg(15, tmp);
#ifdef __AVR_ATmega2560__
            mb.Ireg(31, tmp);
#endif
            tmp = tmp >> 16;
            mb.Hreg(14, tmp);
#ifdef __AVR_ATmega2560__
            mb.Ireg(30, tmp);
#endif
        }
    }
    d = t - ms[1];
    if (d <= 0) {
        ms[1] += 10;
        /* Read digital inputs 22..53 */
        l = 0;
#ifdef __AVR_ATmega2560__
        for (i = 22; i < 22 + MAX_DIGITAL_IN; i++) {
            k = i - 22;
            b = digitalRead(i);
            mb.Coil(k, b);
            mb.Ists(k, b);
            l |= (b ? 1L : 0L) << k;
        }
#endif
#ifdef __AVR_ATmega2560__
        mb.Ireg(17, l);
#endif
        mb.Hreg(17, l);
        l = l >> 16;
#ifdef __AVR_ATmega2560__
        mb.Ireg(16, l);
#endif
        mb.Hreg(16, l);
        /* Read digital inputs 2..13 */
        mode1 = mb.Hreg(21);
        mode0 = mb.Hreg(20);
        k = 0;
        for (i = 2; i < MAX_DIGITAL_OUT; i++) {
            b = false;
            if ((mode1 | mode0) & (1 << i)) {
                b = digitalRead(i);
            }
            k |= (b ? 1 : 0) << i;
#ifndef __AVR_ATmega2560__
            mb.Hreg(i + 64, b ? 1 : 0);
#endif
        }
        mb.Hreg(22, k);
#ifdef __AVR_ATmega2560__
        mb.Ireg(32, k);
#endif
        /* Read analog inputs */
        for (i = 0; i < MAX_ANALOG_IN; i++) {
            k = analogReadFast(A0 + i);
#ifdef __AVR_ATmega2560__
            mb.Ireg(i, k);
#endif
            mb.Hreg(i + 32, k);
            k = ai_avg[i] + ai_avg[i] + ai_avg[i] + k;
            k = k / 4;
            ai_avg[i] = k;
#ifdef __AVR_ATmega2560__
            mb.Ireg(i + 48, k);
#endif
            mb.Hreg(i + 48, k);
        }
        /* Write PWM outputs/set input modes 2..13 */
        for (i = 2; i < 2 + MAX_DIGITAL_OUT; i++) {
            if (mode1 & (1 << i)) {
                pinMode(i, INPUT_PULLUP);
                continue;
            }
            if (mode0 & (1 << i)) {
                pinMode(i, INPUT);
                continue;
            }
            k = mb.Hreg(i);
            analogWrite(i, k);
#ifdef __AVR_ATmega2560__
            mb.Ireg(i + 16, k);
#endif
        }
        /* EEPROM commands */
        k = mb.Hreg(31);
        if (k == 0x55aa) {
            save(0);
        } else if (k == 0xaa55) {
            save(1);
        }
        x = true;
    }
    if (x) {
        t = millis();
        d = t - ms[0];
        mb.Hreg(19, d);
        d = d >> 16;
        mb.Hreg(18, d);
    }
    mb.task();
}

/*
 * Local variables:
 * mode: c
 * c-basic-offset: 4
 * fill-column: 78
 * End:
 */
