/*
 * mbspy.c --
 *
 *	Modbus spy.
 */

#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <sys/time.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#undef MOUSE_MOVED
#include <curses.h>
#include <modbus/modbus.h>

static int istcp = 0;
static int hex = 0;
static modbus_t *mb = NULL;
static int maxr = 0;

#define array_size(a) (sizeof(a) / sizeof(a[0]))

static int
mbspy(void)
{
    int top = 0, i, k, nw, edit = 0, epos, count, lcount = 0, err;
    unsigned short mb_data[9999];
    char mb_flag[9999];

    if (maxr > array_size(mb_data)) {
	maxr = array_size(mb_data);
    } else if (maxr <= 0) {
	maxr = array_size(mb_data);
    }
    memset(&mb_data, 0, sizeof(mb_data));
    memset(&mb_flag, 0, sizeof(mb_flag));
    if (initscr() == (WINDOW *) ERR) {
	return 0;
    }
    if (COLS < 57 || LINES < 5) {
	endwin();
	return 0;
    }
    nw = (LINES - 1) * 8;
    if (nw > maxr) {
	nw = maxr;
    }
#ifdef NCURSES_VERSION
    ESCDELAY = 300;
#endif
    raw();
    noecho();
    idlok(stdscr, TRUE);
    scrollok(stdscr, FALSE);
    keypad(stdscr, TRUE);
    nodelay(stdscr, TRUE);
    meta(stdscr, TRUE);
    nonl();
    if (has_colors()) {
	start_color();
	init_pair(1, COLOR_WHITE, COLOR_BLUE);
	init_pair(2, COLOR_MAGENTA, COLOR_BLUE);
	init_pair(3, COLOR_BLACK, COLOR_CYAN);
	init_pair(4, COLOR_BLUE, COLOR_WHITE);
	wbkgdset(stdscr, COLOR_PAIR(1));
    }
    for (;;) {
	int ch;
#ifndef _WIN32
	fd_set rs;
	struct timeval tv;
#endif
	static const char inde[] = " *";
	static const char indp[] = "\\-/|";

	lcount++;
	k = maxr;
	i = 0;
	while (!edit && k > 0) {
	    int n = 64;

	    if (i + n > maxr) {
		n = maxr - i;
	    }
	    memset(mb_flag + i, 0, n);
	    count = modbus_read_registers(mb, i, n, mb_data + i);
	    if (count > 0) {
		memset(mb_flag + i, 1, count);
	    } else {
		memset(mb_flag + i, 3, n);
		err = errno;
	    }
	    i += n;
	    k -= n;
#ifdef _WIN32
	    ch = getch();
	    if (ch != ERR) {
		ungetch(ch);
		break;
	    }
#else
	    FD_ZERO(&rs);
	    FD_SET(0, &rs);
	    tv.tv_sec = 0;
	    tv.tv_usec = 0;
	    if (select(1, &rs, 0, 0, &tv) == 1) {
		break;
	    }
#endif
	}
	if (!edit && memchr(mb_flag, 1, sizeof(mb_flag)) == NULL) {
	    if (memchr(mb_flag, 3, sizeof(mb_flag)) != NULL) {
		memset(mb_flag, 0, sizeof(mb_flag));
	    }
	    if (has_colors()) {
		wattrset(stdscr, A_NORMAL | COLOR_PAIR(3));
		for (i = 0; i < COLS; i++) {
		    waddch(stdscr, ' ');
		}
	    } else {
		wclrtoeol(stdscr);
	    }
	    wmove(stdscr, 0, 0);
	    wprintw(stdscr, "MBSPY");
	    wprintw(stdscr, " %c  EDIT=^W   QUIT=^D", inde[edit > 1]);
	    wprintw(stdscr, hex ? "   DEC=^X" : "   HEX=^X");
	    wmove(stdscr, 1, 0);
	    wprintw(stdscr, "modbus_read_registers: %s\n",
		    modbus_strerror(err));
	    wclrtobot(stdscr);
	    wrefresh(stdscr);
	    goto again;
	}
dorefr:
	if (has_colors()) {
	    wattrset(stdscr, A_NORMAL | COLOR_PAIR(1));
	}
	for (i = top; i < top + nw; i += 8) {
	    wmove(stdscr, 1 + (i - top) / 8, 0);
	    if (hex) {
		wprintw(stdscr, "%04X:    ", i);
	    } else {
		wprintw(stdscr, "%5d:   ", i + 40001);
	    }
	    for (k = 0; k < 8; k++) {
		if (i + k >= maxr) {
		    waddstr(stdscr, "    ");
		} else if (mb_flag[i + k]) {
		    if (mb_flag[i + k] > 1) {
			wbkgdset(stdscr, COLOR_PAIR(4));
		    }
		    wprintw(stdscr, "%04X", mb_data[i + k]);
		    if (mb_flag[i + k] > 1) {
			wbkgdset(stdscr, COLOR_PAIR(1));
		    }
		} else {
		    waddstr(stdscr, "????");
		}
		waddch(stdscr, ' ');
		if (k == 3) {
		    waddch(stdscr, ' ');
		}
	    }
	    if (COLS > 72) {
		waddstr(stdscr, "   ");
		for (k = 0; k < 8; k++) {
		    union {
			unsigned short s;
			char c[2];
		    } uu;

		    if (k == 4) {
			waddch(stdscr, ' ');
		    }
		    if (!mb_flag[i + k]) {
			waddstr(stdscr, "  ");
			continue;
		    }
		    uu.s = mb_data[i + k];
		    /* this assumes little endian */
		    if (uu.c[1] >= ' ' && uu.c[1] < 0x7F) {
			waddch(stdscr, uu.c[1]);
		    } else {
			waddch(stdscr, '.');
		    }
		    if (uu.c[0] >= ' ' && uu.c[0] < 0x7F) {
			waddch(stdscr, uu.c[0]);
		    } else {
			waddch(stdscr, '.');
		    }
		}
	    }
	    wclrtoeol(stdscr);
	}
	wclrtobot(stdscr);
	wmove(stdscr, 0, 0);
	if (has_colors()) {
	    wattrset(stdscr, A_NORMAL | COLOR_PAIR(3));
	    for (i = 0; i < COLS; i++) {
		waddch(stdscr, ' ');
	    }
	} else {
	    wclrtoeol(stdscr);
	}
	wmove(stdscr, 0, 0);
	wprintw(stdscr, "MBSPY");
	if (edit) {
	    int x, x1, y;
	    static const char *epm[] = { "X...", ".X..", "..X.", "...X" };

	    wprintw(stdscr, " %c  SAVE=^W   QUIT=^D", inde[edit > 1]);
	    wprintw(stdscr, hex ? "   DEC=^X" : "   HEX=^X");
	    wmove(stdscr, 0, 58);
	    if (hex) {
		wprintw(stdscr, " %04X [%s]", epos / 4, epm[epos % 4]);
	    } else {
		wprintw(stdscr, "%5d [%s]", epos / 4 + 40001, epm[epos % 4]);
	    }
	    x = epos % 4;
	    x1 = (epos / 4) % 8;
	    y = ((epos / 4) - top) / 8;
	    wmove(stdscr, y + 1, 9 + x1 * 5 + (x1 > 3) + x);
	} else {
	    wprintw(stdscr, " %c  EDIT=^E   EXIT=^D", indp[lcount % 4]);
	    wprintw(stdscr, hex ? "   DEC=^X" : "   HEX=^X");
	    wmove(stdscr, 0, 6);
	}
	wrefresh(stdscr);
again:
#ifdef _WIN32
	if (WaitForSingleObject(GetStdHandle(STD_INPUT_HANDLE), 200)
	    != WAIT_OBJECT_0) {
	    if (edit) {
		goto again;
	    }
	    continue;
	}
#else
	FD_ZERO(&rs);
	FD_SET(0, &rs);
	tv.tv_sec = 0;
	tv.tv_usec = 200000;
	if (select(1, &rs, 0, 0, &tv) < 0) {
	    if (errno == EINTR) {
		goto again;
	    }
	    break;
	}
	if (!FD_ISSET(0, &rs)) {
	    if (edit) {
		goto again;
	    }
	    continue;
	}
#endif
	ch = getch();
#ifdef _WIN32
	if (ch == ERR) {
	    continue;
	}
#endif
	switch (ch) {
	case 0x1B:
	    if (edit) {
		edit = 0;
		continue;
	    }
	case ERR:
	case 'q':
	case 'Q':
	case 'x':
	case 'X':
	    goto done;
	case 0x04:
	    if (edit) {
		edit = 0;
		continue;
	    }
	    goto done;
	case 0x18:
	    hex = !hex;
	    goto dorefr;
	case KEY_HOME:
	    if (top != 0) {
		top = 0;
		epos = top;
		goto dorefr;
	    }
	    goto again;
	case KEY_END:
	    if (top + nw < maxr) {
		top = maxr - nw;
		epos = (maxr - 1) * 4;
		goto dorefr;
	    }
	    goto again;
	case KEY_DOWN:
	    if (edit) {
		if (epos / 4 + 8 < maxr) {
		    epos += 4 * 8;
		    if (epos / 4 >= top + nw) {
			top += 8;
		    }
		    goto dorefr;
		}
	    } else if (top + nw < maxr) {
		top += 8;
		goto dorefr;
	    }
	    goto again;
	case KEY_UP:
	    if (edit) {
		if (epos / 4 >= 8) {
		    epos -= 4 * 8;
		    if (epos / 4 < top) {
			top -= 8;
		    }
		    goto dorefr;
		}
	    } else if (top >= 8) {
		top -= 8;
		goto dorefr;
	    }
	    goto again;
	case KEY_NPAGE:
	    if (edit) {
		if (epos / 4 + nw + nw < maxr) {
		    epos += 4 * nw;
		    if (epos / 4 >= top + nw) {
			top += nw;
		    }
		    goto dorefr;
		}
		if (epos / 4 < maxr - nw) {
		    epos = 4 * (maxr - 1);
		    top = ((maxr - nw + 7) / 8) * 8;
		    goto dorefr;
		}
	    } else if (top + nw + nw < maxr) {
		top += nw;
		goto dorefr;
	    } else if (top < maxr - nw) {
		top = ((maxr - nw + 7) / 8) * 8;
		goto dorefr;
	    }
	    goto again;
	case KEY_PPAGE:
	    if (edit) {
		if (epos / 4 - nw >= 0) {
		    epos -= 4 * nw;
		    if (epos / 4 < top) {
			top -= nw;
		    }
		    goto dorefr;
		}
		if (epos != 0) {
		    epos = 0;
		    top = 0;
		    goto dorefr;
		}
	    } else if (top - nw >= 0) {
		top -= nw;
		goto dorefr;
	    } else if (top != 0) {
		top = 0;
		goto dorefr;
	    }
	    goto again;
	case KEY_RIGHT:
	    if (edit && epos < maxr * 4 - 1) {
		epos++;
		if (epos / 4 >= top + nw) {
		    top += 8;
		}
		goto dorefr;
	    }
	    goto again;
	case KEY_LEFT:
	    if (edit && epos > 0) {
		epos--;
		if (epos / 4 < top) {
		    top -= 8;
		}
		goto dorefr;
	    }
	    goto again;
	case ' ':
	case '\r':
	case '\n':
	    goto dorefr;
	case '\f':
	    wclear(stdscr);
	    goto dorefr;
	case 0x05:
	    if (!edit) {
		edit = 1;
		epos = top * 4;
		goto dorefr;
	    }
	    goto again;
	case 0x17:
	    if (edit > 1) {
		int err = 0;

		for (i = 0; i < maxr; i++) {
		    if (mb_flag[i] > 1) {
			if (modbus_write_register(mb, i, mb_data[i]) != -1) {
			    mb_flag[i] = 1;
			} else {
			    err++;
			}
		    }
		    if (!istcp) {
#ifdef _WIN32
			Sleep(10);
#else
			const struct timespec ms10 = { 0, 10000000 };

			nanosleep(&ms10, NULL);
#endif
		    }
		}
		if (!err) {
		    edit = 1;
		}
		continue;
	    }
	    goto again;
	default:
	    if (edit && mb_flag[epos / 4]) {
		int val = -1, mask;

		if (ch >= '0' && ch <= '9') {
		    val = ch - '0';
		} else if (ch >= 'A' && ch <= 'F') {
		    val = ch - 'A' + 10;
		} else if (ch >= 'a' && ch <= 'f') {
		    val = ch - 'a' + 10;
		}
		if (val >= 0) {
		    edit = 2;
		    val = val << (12 - 4 * (epos % 4));
		    mask = 0x0f << (12 - 4 * (epos % 4));
		    mb_data[epos / 4] &= ~mask;
		    mb_data[epos / 4] |= val;
		    mb_flag[epos / 4] = 2;
		    if (edit && epos < maxr * 4 - 1) {
			epos++;
			if (epos / 4 >= top + nw) {
			    top += 8;
			}
			goto dorefr;
		    }
		    goto dorefr;
		}
	    }
	    goto again;
	}
    }
done:
    endwin();
    if (err) {
	fprintf (stderr, "modbus_read_registers: %s\n", modbus_strerror(err));
    }
    return err ? 3 : 0;
}

/*
 * Command line arguments
 *
 *   Modbus/RTU    ttydevice baud,parity,bits,stop,slave,mode [nregs]
 *   Modbus/TCP    ipaddress port [nregs]
 */

int
main(int argc, char **argv)
{
    if ((argc > 1) &&
#ifdef _WIN32
	((argv[1][0] == 'C') || (argv[1][0] == 'c'))
#else
	(argv[1][0] == '/')
#endif
       ) {
	int baud = 9600, parity = 'N', bits = 8, stop = 1, slave = 1, mode = 0;

	if (argc > 2) {
	    sscanf(argv[2], "%d,%c,%d,%d,%d,%d",
		   &baud, &parity, &bits, &stop, &slave, &mode);
	}
	if ((parity == 'O') || (parity == 'E') || (parity == 'N')) {
	    /* valid parity */
	} else if (parity == 'o') {
	    parity = 'O';
	} else if (parity == 'e') {
	    parity = 'E';
	} else {
	    parity = 'N';
	}
	mb = modbus_new_rtu(argv[1], baud, parity, bits, stop);
	if (mb != NULL) {
	    modbus_set_slave(mb, slave);
	    if (mode) {
		modbus_rtu_set_serial_mode(mb, MODBUS_RTU_RS485);
	    }
	}
    } else {
	mb = modbus_new_tcp_pi((argc > 1) ? argv[1] : "localhost",
			       (argc > 2) ? argv[2] : "15002");
	istcp = 1;
    }
    if (argc > 3) {
	maxr = strtol(argv[3], NULL, 0);
    }
    if (mb == NULL) {
	fprintf(stderr, "modbus_new_%s: %s\n", istcp ? "tcp_pi" : "rtu",
		modbus_strerror(errno));
	exit(1);
    }
    if (modbus_connect(mb) == -1) {
	fprintf(stderr, "modbus_connect: %s\n", modbus_strerror(errno));
	exit(2);
    }
    if (!istcp) {
#ifdef _WIN32
	Sleep(3000);
#else
	const struct timespec sec3 = { 3, 0 };

	nanosleep(&sec3, NULL);
#endif
    }
    return mbspy();
}

/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * fill-column: 78
 * tab-width: 8
 * End:
 */
