# Simple walkie-talkie using UDP multicast and Snack audio
#
# requires:
#  - modified udp package with -sendsize option to
#    split larger write requests in smaller datagrams
#  - modified fifo (Memchan) which generates proper
#    input-blocked condition when in non-blocking mode
#  - modified snack package which deals with non-blocking
#    channels gracefully regarding input-blocked condition

package require Memchan
package require snack
package require udp

wm attributes . -fullscreen 1
. configure -bg black
sdltk touchtranslate 0

bind all <Key-Break> exit
bind . <<NetworkInfo>> network_state
bind . <<TetherInfo>> network_state
bind . <<WillEnterBackground>> talk_off

snack::sound microphone -freq 22050 -channels 1
set recfifo [fifo]
fconfigure $recfifo -translation binary -buffering none -blocking 0
microphone configure -channel $::recfifo

set talking 0
set udperrs 0
array set client {}

canvas .c -bg black -bd 0 -highlightthickness 0
pack .c -side top -fill both -expand 1 -padx 0 -pady 0

# network/tethering state changes

proc network_state {} {
    set havenet 0
    if {[string match *wifi* [borg networkinfo]]} {
	set havenet 1
	set mcast 224.1.2.3
    } else {
	array set teth [borg tetherinfo]
	if {[info exists teth(active)] && $teth(active) ne ""} {
	    set havenet 1
	    set mcast [list 224.1.2.3 [lindex $teth(active) 0]]
	}
    }
    if {$havenet} {
	if {![info exists ::udp]} {
	    set ::udp [udp_open 7777]
	    fconfigure $::udp -buffering none -translation binary \
		-blocking 0 \
	        -mcastadd $mcast -mcastloop 0 -ttl 1 \
		-sendsize 1440 -remote [list 224.1.2.3 7777]
	    fileevent $::udp readable [list udp_input $::udp 7777]
	    .c delete stop
	}
	if {$::talking} {
	    .c itemconfigure title -fill \#ff0000
	} else {
	    .c itemconfigure title -fill \#3333f0
	}
    } else {
	talk_off
	stop_sign
	.c itemconfigure title -fill black
	catch {close $::udp}
	catch {unset ::udp}
	foreach ip [array names ::client] {
	    catch {timeout $ip}
	}
    }
}

# UDP datagram input

proc udp_input {chan port} {
    set data [read $chan]
    set peer [fconfigure $chan -peer]
    lassign $peer ip rport
    if {$rport != $port || [string length $data] == 0} {
	return
    }
    if {![info exist ::client($ip)]} {
	set fifo [fifo]
	fconfigure $fifo -translation binary -buffering none -blocking 0
	set ::client($ip) [dict create fifo $fifo timer notimer play 0]
    }
    dict with ::client($ip) {}
    after cancel $timer
    puts -nonewline $fifo $data
    set len [fconfigure $fifo -length]
    if {$len >= 4000 && !$play} {
	set sound [snack::sound -freq 22050 -debug 0 -buffersize 11025 \
	    -channel $fifo]
	dict set ::client($ip) sound $sound
	$sound play -nopeeping 1 -blocking 0
	dict set ::client($ip) play 1
    } elseif {$len > 20000 && $play} {
	# overrun, consume everything so far
	read $fifo
    }
    dict set ::client($ip) timer [after 5000 [list timeout $ip]]
}

# timeout handler for incoming UDP stream

proc timeout ip {
    dict with ::client($ip) {
	after cancel $timer
	$sound stop
	$sound destroy
	close $fifo
    }
    unset ::client($ip)
}

# timer handler to send outgoing UDP datagrams

proc udp_send {} {
    set data [read $::recfifo]
    if {[catch {puts -nonewline $::udp $data}]} {
	incr ::udperrs
    } else {
	set ::udperrs 0
    }
    if {$::udperrs > 10} {
	talk_off
	return
    }
    after 60 udp_send
    update idletasks
}

# PTT function, turns sending on or off

proc talk {} {
    if {![info exists ::udp]} {
	.c itemconfigure title -fill black
	microphone stop
	after cancel udp_send
	read $::recfifo
	sdltk screensaver 1
	set ::talking 0
	return
    }
    if {!$::talking} {
	.c itemconfigure title -fill \#ff0000
	snack::audio play_gain 25
	microphone record
	after 80 udp_send
	after 200 {borg vibrate 90}
	sdltk screensaver 0
	set ::talking 1
    } else {
	.c itemconfigure title -fill \#3333f0
	snack::audio play_gain 100
	microphone stop
	after cancel udp_send
	read $::recfifo
	sdltk screensaver 1
	set ::talking 0
    }
    update idletasks
    update idletasks
    update idletasks
}

proc talk_on {} {
    if {!$::talking} {
	talk
    }
}

proc talk_off {} {
    if {$::talking} {
	talk
    }
}

proc stop_sign {} {
    lassign [.c bbox image] x1 y1 x2 y2
    set w [expr {$x2 - $x1}]
    set h [expr {$y2 - $y1}]
    set w2 [expr {$w / 2}]
    set t [expr {$y1 + $h / 2 - $w2}]
    .c create oval $x1 $t $x2 [expr {$t + $w}] -fill \#ff0000 -tags stop
    set w3 [expr {0.1 * $w}]
    set t2 [expr {$y1 + $h / 2 - $w3}]
    set t3 [expr {$y1 + $h / 2 + $w3}]
    .c create rectangle [expr {$x1 + $w3}] $t2 [expr {$x2 - $w3}] $t3 \
	-fill \#ffffff -tags stop
}

proc config_canvas {} {
    set w [winfo screenwidth .]
    set h [winfo screenheight .]
    set mw [winfo screenmmwidth .]
    set mh [winfo screenmmheight .]
    if {$mw < $mh} {
	set mh $mw
    }
    set fs [expr round($mh / 4.0)]
    .c delete image
    .c create image [expr {$w / 2}] [expr {$h / 2}] -image bc611 -tags image
    lassign [.c bbox image] x1 y1 x2 y2
    .c delete stop
    .c delete title
    .c create text [expr {$w / 2}] [expr {$y2 + 50}] -text "TiCkLe TalKie" \
	-font [list {DejaVu Sans} $fs] -fill \#3333f0 -anchor center \
	-tags title -justify center
    if {![info exists ::udp]} {
	.c itemconfigure title -fill black
	stop_sign
    } elseif {$::talking} {
	.c itemconfigure title -fill \#ff0000
    }
}

proc mkphoto {} {
    image create photo bc611 -data {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    }
    set w [winfo screenwidth .]
    set h [winfo screenheight .]
    if {$w < $h} {
	set h $w
    }
    set scale [expr {$h / 1000.0}]
    bc611 copy bc611 -scale $scale

}

mkphoto
bind .c <Configure> config_canvas
.c bind image <ButtonPress-1> talk
after 1000 event generate . <<NetworkInfo>>
