##############################################################################
#
# Proof-of-concept Tcl interface to libsnap7 (http://snap7.sourceforge.net)
# using Ffidl and TclOO. Client-side only.
#
# chw May 2018
#
##############################################################################

package require Ffidl
package require Ffidlrt
package require TclOO

package provide snap7 0.1

namespace eval ::snap7 {
    ::ffidl::typedef S7Object pointer

    variable lib
    variable lib_1
    switch -exact $::tcl_platform(platform) {
	unix {
	    set lib libsnap7[info sharedlibextension]
	    set lib_1 ${lib}.1
	}
	windows {
	    set lib snap7[info sharedlibextension]
	}
	default {
	    error "unsupported platform"
	}
    }
    # Prefer library in script install directory,
    # also prefer versioned library name on unix platforms
    set lib [file join [file dirname [info script]] $lib]
    if {[info exists lib_1]} {
	set lib_1 [file join [file dirname [info script]] $lib_1]
	if {[file readable $lib_1]} {
	    set lib $lib_1
	}
    }
    # Use system path as fall back
    if {![file readable $lib]} {
	if {[info exists lib_1]} {
	    set lib [file tail $lib_1]
	} else {
	    set lib [file tail $lib]
	}
    }

    # _GetParam/_SetParam names, types, and numbers
    # NOTE: not all are supported for "Cli_" functions
    variable params {
	LocalPort uint16 1
	RemotePort uint16 2
	PingTimeout int 3
	SendTimeout int 4
	RecvTimeout int 5
	WorkInterval int 6
	SrcRef uint16 7
	DstRef uint16 8
	SrcTSap uint16 9
	PDURequest int 10
	MaxClients int 11
	BSendTimeout int 12
	BRecvTimeout int 13
	RecoveryTime int 14
	KeepAliveTime int 15
    }

    # Create the Ffidl callouts, Tcl procs with "_" prefix,
    # library function names with "Cli_" prefix.
    apply [list lib {
	foreach {ret name arglist} {
	    S7Object Create {}
	    void Destroy {pointer-var}
	    int ConnectTo {S7Object pointer-utf8 int int}
	    int SetConnectionType {S7Object uint16}
	    int GetParam {S7Object int pointer-var}
	    int SetParam {S7Object int pointer-var}
	    int Disconnect {S7Object}
	    int GetConnected {S7Object pointer-var}
	    int GetPduLength {S7Object pointer-var pointer-var}
	    int DBRead {S7Object int int int pointer-var}
	    int DBWrite {S7Object int int int pointer-var}
	    int ErrorText {int pointer-var int}
	} {
	    if {$::tcl_platform(platform) eq "windows"} {
		::ffidl::callout ::snap7::_$name $arglist $ret \
		    [::ffidl::symbol $lib Cli_$name] stdcall
	    } else {
		::ffidl::callout ::snap7::_$name $arglist $ret \
		    [::ffidl::symbol $lib Cli_$name]
	    }
	}
    } [namespace current]] $lib

    proc error {code} {
	set v [binary format x1024]
	::snap7::_ErrorText $code v 1024
	if {$v eq ""} {
	    set v "unknown error code $code"
	}
	return -code error $v
    }
}

oo::class create ::snap7::snap7

oo::define ::snap7::snap7 {
    constructor {args} {
	my variable s7obj
	set s7obj [::snap7::_Create]
	if {$s7obj == 0} {
	    error "unable to get snap7 client object"
	}
    }

    destructor {
	my variable s7obj
	if {[info exists s7obj]} {
	    set v [binary format [::ffidl::info format pointer] $s7obj]
	    ::snap7::_Disconnect $s7obj
	    ::snap7::_Destroy v
	}
	return ""
    }

    method connect {addr port rack slot} {
	my variable s7obj
	my param RemotePort $port
	set code [::snap7::_ConnectTo $s7obj $addr $rack $slot]
	if {$code != 0} {
	    tailcall ::snap7::error $code
	}
	return ""
    }

    method disconnect {} {
	my variable s7obj
	set code [::snap7::_Disconnect $s7obj]
	if {$code != 0} {
	    tailcall ::snap7::error $code
	}
	return ""
    }

    method conntype {type} {
	my variable s7obj
	# 1=PG, 2=OP, 3=basic
	set code [::snap7::_SetConnectionType $s7obj $type]
	if {$code != 0} {
	    tailcall ::snap7::error $code
	}
	return ""
    }

    method param {args} {
	my variable s7obj
	if {[llength $args] == 0} {
	    foreach {name type id} $::snap7::params {
		lappend list $name
	    }
	    return $list
	}
	if {[llength $args] > 2} {
	    error "wrong number arguments, need name and optional value"
	}
	set want [lindex $args 0]
	set found 0
	foreach {name type id} $::snap7::params {
	    if {$want eq $name} {
		set found 1
		break
	    }
	}
	if {!$found} {
	    error "parameter \"$want\" not found"
	}
	if {[llength $args] > 1} {
	    set v [binary format [::ffidl::info format $type] [lindex $args 1]]
	    set code [::snap7::_SetParam $s7obj $id v]
	    if {$code != 0} {
		tailcall ::snap7::error $code
	    }
	    return ""
	}
	set v [binary format x[::ffidl::info sizeof $type]]
	set code [::snap7::_GetParam $s7obj $id v]
	if {$code != 0} {
	    tailcall ::snap7::error $code
	}
	binary scan $v [::ffidl::info format $type] v
	return $v
    }

    method isconnected {} {
	my variable s7obj
	set v [binary format x[::ffidl::info sizeof int]]
	set code [::snap7::_GetConnected $s7obj v]
	if {$code != 0} {
	    tailcall ::snap7::error $code
	}
	binary scan $v [::ffidl::info format int] v
	return $v
    }

    method pdulength {} {
	my variable s7obj
	set req [binary format x[::ffidl::info sizeof int]]
	set neg [binary format x[::ffidl::info sizeof int]]
	set code [::snap7::_GetPduLength $s7obj req neg]
	if {$code != 0} {
	    tailcall ::snap7::error $code
	}
	binary scan $req [::ffidl::info format int] req
	binary scan $neg [::ffidl::info format int] neg
	return [list $req $neg]
    }

    method dbreada {db start n} {
	my variable s7obj
	if {$n <= 0} {
	    return -code error "at least one item required"
	}
	if {$start < 0} {
	    return -code error "start offset must be positive"
	}
	set v [binary format x[expr {$n * [::ffidl::info sizeof uint8]}]]
	set code [::snap7::_DBRead $s7obj $db $start $n v]
	if {$code != 0} {
	    tailcall ::snap7::error $code
	}
	return $v
    }

    method dbread {db start {n 1}} {
	my variable s7obj
	if {$n <= 0} {
	    return -code error "at least one item required"
	}
	if {$start < 0} {
	    return -code error "start offset must be positive"
	}
	set v [my dbreada $db $start $n]
	binary scan $v [::ffidl::info format uint8]${n} v
	return $v
    }

    method dbwritea {db start bytes} {
	my variable s7obj
	set v ""
	binary scan $bytes a* v
	set n [string length $v]
	if {$n <= 0} {
	    return -code error "data missing"
	}
	if {$start < 0} {
	    return -code error "start offset must be positive"
	}
	set code [::snap7::_DBWrite $s7obj $db $start $n v]
	if {$code != 0} {
	    tailcall ::snap7::error $code
	}
	return ""
    }

    method dbwrite {db start args} {
	my variable s7obj
	set n [llength $args]
	if {$n <= 0} {
	    return -code error "at least one value required"
	}
	if {$start < 0} {
	    return -code error "start offset must be positive"
	}
	set v [binary format [::ffidl::info format uint8]${n} $args]
	set code [::snap7::_DBWrite $s7obj $db $start $n v]
	if {$code != 0} {
	    tailcall ::snap7::error $code
	}
	return ""
    }
}

namespace eval ::snap7 {
    proc new {args} {
	uplevel 1 ::snap7::snap7 create $args
    }
}
