#
# ow_ds18b20.tcl
#
# Enables 1-Wire on pin 4, and searches and reads DS18B20
# temperature sensor(s) periodically.
#

package require tfirmata

proc search_proc {pin alarm addr} {
    if {[string match "28.*" $addr]} {
        # 28 is first address byte of DS18B20
        set ::ds18b20($addr) 1
        puts "found DS18B20 $addr"
    }
}

proc read_data {pin addr data} {
    lassign $data lo hi
    set temp [expr {$lo + ($hi << 8)}]
    if {$temp > 0x3ff} {
        set temp [expr {0 - $temp}]
    }
    set temp [expr {$temp * 0.0625}]
    puts [format "DS18B20 %s: temp %3.3f degC" $addr $temp]
}

set bd [tfirmata::open /dev/ttyUSB0]

$bd owconfig 4 0
$bd owsearch 4 -command {search_proc %C %S %A}
tfirmata::sleep 1000

if {![array exists ::ds18b20]} {
    puts stderr "no DS18B20 found, exiting"
    exit
}

foreach addr [array names ::ds18b20] {
    # write "Write Scratchpad" command and config to 11 bit resolution
    $bd owtrans 4 -reset -address $addr -write {0x4e 0 0 0x40}
}

while {1} {
    foreach addr [array names ::ds18b20] {
        # write "Convert T" command
        $bd owtrans 4 -reset -address $addr -write 0x44
    }
    # wait for conversion(s) to complete
    tfirmata::sleep 500
    foreach addr [array names ::ds18b20] {
        # write "Read Scratchpad" command and fetch 9 bytes
        $bd owtrans 4 -reset -address $addr -write 0xbe -read 9 -command \
           [list read_data %C $addr %V]
    }
    tfirmata::sleep 500
}

