#
# ow_ds18b20_sched.tcl
#
# Enables 1-Wire on pin 4, and searches and reads DS18B20
# temperature sensor(s) periodically using scheduler.
#

package require tfirmata

proc search_proc {pin alarm addr} {
    if {[string match "28.*" $addr]} {
        # 28 is first address byte of DS18B20
        set ::ds18b20($addr) -1
        puts "found DS18B20 $addr"
    }
}

proc read_data {pin tid data} {
    lassign $data lo hi
    set temp [expr {$lo + ($hi << 8)}]
    if {$temp > 0x3ff} {
        set temp [expr {0 - $temp}]
    }
    set temp [expr {$temp * 0.0625}]
    puts [format "DS18B20 %s: temp %3.3f degC" $::ds18b20($tid) $temp]
}

set bd [tfirmata::open /dev/ttyUSB0]

$bd owconfig 4 0
$bd owsearch 4 -command {search_proc %C %S %A}
tfirmata::sleep 1000

if {![array exists ::ds18b20]} {
    puts stderr "no DS18B20 found, exiting"
    exit
}

# one wire catch-everything callback
$bd owcallback [list read_data %C %T %V]

foreach addr [array names ::ds18b20] {
    # write "Write Scratchpad" command and config to 11 bit resolution
    $bd owtrans 4 -reset -address $addr -write {0x4e 0 0 0x40}
}

# query pin 4 state to sync
$bd state 4

# clean task state
$bd resettasks

# make task data buffer
set td [$bd taskdata]

# all found one wire addresses
set all_ds18b20 [array names ::ds18b20]

foreach addr $::all_ds18b20 {
    # add "Convert T" command to task
    $td owtrans 4 -reset -address $addr -write 0x44
}

# 500 ms for conversion
$td delaytask 500

# transaction id for mapping to one wire address
set tid 1

foreach addr $::all_ds18b20 {
    # remember tid to one wire address mapping
    set ::ds18b20($tid) $addr
    set ::ds18b20($addr) $tid
    # add "Read Scratchpad" command to task
    $td owtrans 4 -reset -address $addr -write 0xbe -read 9 -tid $tid
    incr tid
}

# 500 ms until next conversion
$td delaytask 500

# create and run task
$bd createtask 1 $td
$bd scheduletask 1 0

vwait forever
