#
# temp_sys.tcl
#
# Closed control script to control hardware interfaced to an Arduino Uno.
#
# The hardware consists of a TWI temperature sensor, three LEDs for a simple 
# visual indication of temperature, an LDR for sensing ambient light, a lamp 
# output which is turned on when ambient light is below a threshold, an input 
# switch to enable lamp output, and a heartbeat LED. 
# 
# The temperature sensor is a DS1621 TWI device connected to Arduino pins 19 
# and 20. Three temperature indicator LEDs, green, yellow and red, are on pins 
# 3, 5 and 6. Temperature LEDs are driven with a PWM so that brightness 
# depends on temperature. The LDR is part of a potential divider that is 
# measured on analog channel 0. Lamp output is on pin 12. Lamp enable is on 
# pin 2. Heartbeat LED is on pin 13.
#

package require tfirmata

set bd [tfirmata::open /dev/ttyUSB0]

# for lamp control
set onThreshold 550
set offThreshold 450
set lampOn 0

# configure board
$bd mode 2 in 3 5 6 pwm 12 13 out
$bd dset 2 1 12 0 13 0
$bd period 1000
$bd dstream 0 on
$bd astream 0 on
$bd dcommand updateLamp
$bd acommand updateLamp
$bd twiconfig
$bd twiset 0x48 0xee
$bd twiget -repeat 0x48 0xaa 2 -command {updateTempLeds [lindex [list %D] 0]}

# update temperature leds based on TWI temperature measurement
proc updateTempLeds {temp} {
    set green [expr {255 - ($temp - 17) * 255 / 4}]
    set yellow [expr {255 - abs($temp - 21) * 255 / 4}]
    set red [expr {255 - (25 - $temp) * 255 / 4}]
    foreach {var pin} {green 3 yellow 5 red 6} {
        if {[set $var] < 0} {
            set $var 0
        }   
        if {[set $var] > 255} {
            set $var 255
        }
        $::bd aset $pin [set $var]
    }
}

# update lamp output, depending on enable switch and LDR
proc updateLamp {} {
    global bd onThreshold offThreshold lampOn
    set val [$bd aget 0]
    if {!$lampOn} {
        if {[$bd dget 2] && $val > $onThreshold} {
            set lampOn 1
            $bd dset 12 $lampOn
        }
    } else {
        if {![$bd dget 2] || $val < $offThreshold} {
            set lampOn 0
            $bd dset 12 $lampOn
        }
    }
}

# toggle heartbeat LED
while {1} {
    $bd dset 13 1
    tfirmata::sleep 250
    $bd dset 13 0
    tfirmata::sleep 250
}

