# filesystem.tcl --
#
# A simple native filesystem mapping for topcua
#
# Copyright (c) 2023 Christian Werner <chw at ch minus werner dot de>
#
# See the file "license.terms" for information on usage and redistribution of
# this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# Usage: opcua::filesystem handle foldername rootdir
#
#	handle		server handle
#	foldername	name for folder below /Root/Objects
#	rootdir		native root directory for filesystem
#
# Up to 256 open native files below rootdir are managed. By using the
# session identifiers in various method calls the number of open native
# files per session is limited to 32. When files are opened or closed
# a cleanup function for orphaned session identifiers removes left over
# open files. Since open file handles are tied to sessions open files
# are naturally isolated between sessions in terms of OPC/UA communication.
#
# Implemented FileType methods:
#   Open Close Read Write SetPosition GetPosition
#
# Implemented FileType properties:
#   OpenCount Size Writable UserWritable
#
# Implemented FileDirectoryType methods:
#   CreateDirectory CreateFile Delete MoveOrCopy
#
# Non-standard FileDirectory methods (available only on top node):
#   CloseAll Rescan

package provide topcua::filesystem 0.1

namespace eval ::opcua {}

# Check for valid file/directory name in OPC/UA space.

proc opcua::fs_namevalid {name} {
    if {$name eq "." || $name eq ".." || [string match {*[:/\|~]*} $name]} {
	return 0
    } elseif {$name eq {}} {
	return 0
    }
    return 1
}

# Setup methods given ObjectType, namespace and method name list.

proc opcua::_fs_setmeths {handle type ns meths} {
    foreach meth $meths {
	set mid [translate $handle $type HasComponent $meth]
	set mid [lindex $mid 0]
	set oargs {}
	if {![catch {
	    translate $handle $mid HasProperty OutputArguments
	} oid]} {
	    set oid [lindex $oid 0]
	    set oargs [read $handle $oid]
	}
	methods $handle $mid $oargs [list ${ns}::$meth $handle]
    }
}

# Setup data source callbacks given Object, namespace and Variable names.

proc opcua::fs_setdsrc {handle obj ns vars} {
    foreach var $vars {
	set vid [translate $handle $obj HasProperty $var]
	set vid [lindex $vid 0]
	datasources $handle $vid [list ${ns}::$var $handle]
    }
}

# Fill address space from native filesystem with corresponding
# FileType and FileDirectoryType objects.

proc opcua::fs_populate {handle ns top dir} {
    if {[catch {glob -directory $dir -tails -nocomplain * .*} list]} {
	return
    }
    set dirs {}
    foreach file $list {
	if {$file eq "." || $file eq ".."} {
	    continue
	}
	if {[string match {*[:/\|~]*} $file]} {
	    continue
	}
	set fullname [file join $dir $file]
	if {[::file isdirectory $fullname]} {
	    if {![catch {translate $handle $top Organizes $file} node]} {
		set node [lindex $node 0]
	    } else {
		# automatic numeric node identifer
		set node [add $handle Object "i=0" $top Organizes \
			      $file [set ${ns}::type_dir]]
	    }
	    set ${ns}::nd($node) $fullname
	    lappend dirs $node $fullname
	} else {
	    if {![catch {translate $handle $top Organizes $file} node]} {
		set node [lindex $node 0]
	    } else {
		# automatic numeric node identifer
		set node [add $handle Object "i=0" $top Organizes \
			      $file [set ${ns}::type_file]]
	    }
	    set ${ns}::nf($node) $fullname
	    # setup data sources for Variables in FileType object
	    fs_setdsrc $handle $node $ns {
		OpenCount Size UserWritable Writable
	    }
	}
    }
    foreach {top dir} $dirs {
	fs_populate $handle $ns $top $dir
    }
}

# Load XML fragment when open62541 was built with minimum nodeset
# adding the FileType and FileDirectoryType Object types.

proc opcua::_fs_loadxml {handle} {
    set xml {
	H4sIAMio8mMCA+1dbXPiOBL+Pr9Cx4etvasEG4zfdpNsESAT6gjkgMzs3Jcr
	B0TiO2NztkmG+/XXkoHw4hiMTEJIT2Vmgt1qSd3qR612mz774+fQIU/UD2zP
	Pc8V8nKOULfn9W334Tw3DgenRo78cfHl7C+np1/I30jFG018++ExJL/2/kqK
	sqyeFuVigXQfKWndVsiVN3b7VgjMTkjd7eVJ2XEIbxAQnwbUf6L9PDBivJbp
	yU29Sxp2j7oBJTAOmZGwv7fUH9oBGx+xA/JIfXo/IQ++5Ya0f0IGPqXEG5De
	o+U/0BMSesRyJ2QEM/Jc1ty7Dy3bhekQi/Rg9Iw4fAROgTcIny2fAn2fWEHg
	9WwLWJK+1xsPqRvyUTEOA9uhAfk1hCnmOtNGub/yrvrUcojtEnZvdos82+Gj
	Nw5ZU5hy6Nu9SB6223PGTLAzCuLYQzvqh3OYygn4jgN6wtqzAZ+QIahjwP6n
	fIqj8b1jB48npG8z7vfjEC4G7CIX3gmbkOT5JKCOM2ViwwT4vF+GyclYXyMm
	33AqMd7786M3ZLSs8XxWILHB2HehY8qb9T2QIO/337QXsiuM+8BzHO8Z5hh1
	7PZtNrvgt5ky2Tqx7r0nyqcWrSTXC2Hk0XCYYkYvCp/eCh4tWEb3fECRFGEM
	IHZ2dTY7nw0lCGFZ2KCTkefzjldnneeDuK6RTuuq+73crpF6h9y2W9/q1VqV
	5Mod+Jw7Id/r3evWXZcARbvc7P4grStSbv4gf683q1wxtT9v27VOh7TapH5z
	26jXqrDem5XGXbXe/EouoWmzBcu5Dosa+HZbvM8pt3qtw5f/FbmptSvXcKV8
	WW/Uuz9OyFW922Rsr4BvmdyW29165a5RbpPbu/Ztq1ODQVRZ22arWW9etaGv
	2k2t2c1D33CN1L7BB9K5LjcavMPyHUyjzUdZad3+aNe/XndZ8+tWo1qD65c1
	GGL5slGLOoQJVhrl+s0JqZZvyl9rvGELGLU5WTRG1v77dY1fhV7L8FPp1ltN
	Np9Kq9ltw8cTmHG7O2/9vd6pnZByu96BAbP2V+0WdMIEDI1anA80bdYiRkz4
	yzoCEvb5rsN+5bLj3Vdr5QZw7LD2i/QzgGGLrecNRw4NKZlaB7EeADGYfZOe
	5cKagiULEMRxha/SxzAc/SZJ3qg3mGNT3vMfpCk2SaBSieGTNO3lPJs/0YBh
	+cOPRcCuh0MORs/uDJ6ZPc2wjGESCSZBSIcAMg59gnUP1tKnQT7LMZ2eXnz5
	cnZXbgLnDg0JbBZgzD8D+zw3ldPz83P+WeECgt2gIP150+j0HunQOrVdZo09
	mpu36m9ulSMNKwhvGODZFOhhcymeysXTYqkry7/xn39OGc6ZravqrgxsCwVJ
	VqT52PPQf+7iCyFnwJ06Aft19oHwf+98O5lnjnxb2ChLOXLLQZfTVGHvYMMt
	mKewJ8qFxeFKvFvppd+zsmNbAZ2NgX8i/N/z3KXnOdRycxf2eeFM4hfjyDqX
	k5AyomIC0YxGSaCpu2FBY0SlBKK7OZWazEopMiJtA6uISk9mpZUYkbGBVURl
	JlBdOZ4VcnHKCVRVD3TJhVVIEjtTc9ceRoRJYu2ABbsPnGyThhZIk6T7dWz3
	OVGSov4cOjWHwxsnTVIEs4t6xDFJE7WfI9iZaX+BPEkn/xhbDrfdphVJqZgk
	9YbXA/L/0X6X/uQjLiauefCVxkEFBsKHYSaLf9wLx340hCQNNMfDe+pzMi15
	QdKHKZ2+YU3OCZMEdW3BTIYjz53qqqQnE9/6HjhG4YTTJo205T9YLsg04Jav
	JnOtPUH3HW/s9yKg0JLJm+CPgXL57EobZtcZ34eTEWdb2jCKLtBV6QC2Ngal
	vIWc3IJjqQNm0x5HVqtsEF9teqjhtBtGXqVBz7dHs7Eo5gY1WuCuc2xMAoSu
	x5ZvRJe0Hq98b/hCWdgwq8EAnG9OWdogYn9MQSNzzqYsb2hxZTnBapMNeqxG
	Rx3Ln9Tc0J9E2KKa+iYzmB4Tpp1sWINfx5Yf4ZBaSMRWoC33+/VoK9W1TfNl
	husPrMgQGP2iNqe/Rls2OEQtfuph65ZEwHieYzuHqqs5cul7zwFl+AfaBD+t
	y60g6rRqByPHmrCbF7N7Z9Li1YiuAjJ/8PzJxSV0CifpgUdm5IRdOpPmFFPG
	i6fWC+bFBODG+HQAfi04Yfl1h+apIKtSxfOpBCfeQLq1/LAoS6V8MQ+Lbpld
	1EV7xmwqwcVLZP4bG+IaYDHJgFrnROkZFDVDmIEuxACmYKZlsITwwMGQhTko
	whxUYQ6GKAdTWA5majnM9iNSD648/xlA5Dw3YBjH4XN9SgsXphCwaPZTHPhm
	+bYFjuMKCmjLKNCB7RhOCxYwC9fgAlxKKxrkzJ9dhwrGIA4mUprk+j6rpVbl
	2s6rGzubVawquFi20sZM+Am6YLCxrIvvvh0yqi30MT+KrStkxgWVsptS9GWl
	3AXUz0Yxi5xQOTsoh21zy8ppjahbAd8h3BLCWKBgXTFzLqiVDVq5oeGj11/W
	iSGv6+Q1dbwi/AzkvuIPGQVBh8oovoPWXjwJQbVFeko0JaOwrLa6OxqHZf+B
	+9ZBrAKZpl/syT4vmrCBfbOcMW1b7n/Oc8Cx7PvWpGoDDxaMDNi1GJ0v9/WJ
	rC7GyV5VH1zhIp132LCDsDWo/QwjmUZ+3nZxZtmQ5KLEhhfMY8xTriv8Xu7A
	Pdag3l+8BBfrfdAVj7Cw8BE7OC9cWWwtrTc/u/T6kxV+M/UvX4YbfBUwZZxJ
	Lwti4f5sBa7eWB2j8toIo1G+xuZsvqAvTgFEXj6t0q0s9SiUvthD3AzPpGVR
	nEmvKOJMitX7bM3Ml8jWW6dRXIHpcfh2Br/SGVo8WvyqxbMY0rXl9h1Ru9c/
	q93HOmfKstVXHC+gKbwzTr8H96x07N5VKb13paB3JYi1CmItYu37Ye3KA5Y2
	tfopoJaR7wFpNdGDsH7sUK2lh2o1PVQXEaoXoVpFqEao3h6qD19RDeo+hI+C
	StIwZrGEs/oOMQsVYxYIzm9j82ydCVo8y6tEF3puvMb6E/E04QpOvwcn2jx2
	H9hM7wMb6AMLwqyBMIs+8DH5wLgfZrwfmiu5FV9peOsFEa5tvysutMp+bzRF
	My3MY8+0MNNnWpiYaSG4uZr43BU31/eMXZg75FuYmG+Bdv9Gdv/iEQhZvYnu
	2oL9rmRbdHZy1zp7ddeOPfPCTJ95YSoYyhBEXcy8QG/rqEIZuD2KbY/xbyEr
	iqqsv4VctX2g9fxJ0uvIS0Qp3ktearfvF5SVLF5QXvqWBCaydFv2anNNE3uh
	VVHSpwCtchB9qRY4KMIc1Exfy9UKu72WO93nlo1ixWf5xQl/X1q77PIvD+Hv
	OdKZDO89x+7FWA+79K9bx+rRR8/pUz/G1+H2F2Ngr3aYgQ+zqgdVFdWkKviS
	t64XDNExaKnX47ovx9Uh7M6B+yUbOyJEvD8XO6zXF/erByKm65X0c58C9M5X
	WfwKBWOIeFVpz7F8DqPzu4Yem6a+zDfrYxMMSiyNEhgcYhpl7N6y07mLSSjl
	uStaHRjl3v3cFYuleO6KfTK5uLGJPqIsYqR7yYxTZ+llaPqfNdKNtr+D7fMV
	KGr9+M7Loh0bcQ4eO0rs4NuZ8uu+HWO5B7fOFHTrNPnY3TozvVuHmYGi0I6Z
	gduH09GjO3w9tel/xzQImbqi7/sRUxf634t4q8k7+N9ZYfQn9r8RpFOANLre
	+HT6U56R2MOGZXyuUlb1Y/VRTnSVaaDDa2dEfe5yiop9whOxz/oEBZMzj/oA
	xCaY8gAUKRzj2rvvrbHP53BvjUPs6eNlb2beGNrKLLSlrWSq3HhPtOWzKnO7
	gHJspucLy+xDW1pJNLSlHnloSyulDm1pO2SKlhDZF9SnYaZoOmRfxAg8PR2y
	xrqszGe4kISC6jpkdUXPlbIwLFTUXhXVpM8Y488+aqymjxprCmZtoP/zdlaP
	QWPxVzBWrF6LzzRPSjDPIK88w3Ty1VOqLnrMNUQZiOaQ6MIvR+hFYQ7CSfm6
	8EsihrAcDGE5pK9AF5PWH/s660dN63/Fe9C3KiMXAQ6WkVt3A7SsimIxBNu6
	jNyqPrCM3N6UYqYqI5dCMZ+yjFy2ytHl7cvIxUEYlpET00rc4xa9uFUZuUgd
	rz1o4SWL3qjIHAxZEfX8Sgf5iGVLpW7xiEVXUj9iYesAH56L2GTMaQBDDHEh
	Biwyl3VQUS+lDypmZvCfN6iIFo8Jju+YKaOr2xWZ2+i7KW9Wgg4GrR2775X+
	hXwdX8gXRWJ8KReR+B2RWN+qBN1GIH6zAnUwZNHnL7p57EBupAdyHV/BFQRy
	HYEcgRwL1GGBukRoNneId+gY70BwxoI8H9LBNuTtCtRt9LCNNytfB4MuHLmH
	bBRSe8iGjB6yGAgbWGkDPWQsX4e7ZcJuWUxTvm7Tnhn/9XB7LG4HExDN8DCO
	PcPDSJ/hYWCGh+jWi897cet9z7iHsUOeh4F5Hmj3WL3n4zpzapridhudOeWN
	S9/BBI4948NIn/FhqBgGEcRkzPhAXwxL3+Hmmbx56jsUQlLepdSRIZqQYxzo
	V0Iqme2z6RNyDB1jHoL7LD7z3TKKjaWO9hf32CHfw8B8D7R9LHX04V04U05X
	6kh562JGZkHQcTOLR+64menzREzMExEEbxPzRFLUycBiRljM6DN72GYxvYed
	GUZ/Xg8bQToNSKNzjc8bPuspSMm6mNEbliuC4ZeO/YiTvqiFqWBsWnD3xC91
	3g6TsVzR/oBZTVeuSHnrgkSmJhqe0o8du9Pn75gqFiQSxG7M30mH3ViQ6KNo
	DAsSYUEiVBQWJPogkV99h8ivirkV6P9gQaKPYffrBYk0pbASr3qJGcaZ6cvd
	DCx0ufaLppQMseoxmqIWhDmUhDloGVSw2a1UzHpYggk1beJztCZeL6Vs6Dum
	RVeA4AHuXVxaASV1d+ARtp7mbZgcCLt3Js1Jpx15PW4wfCAXDOACQDh/Jo38
	OtY9FWRVqng+lfpeL5BgkmFRlkp5JQ9yXWYnHlsBKYvVoIJVI2cdGGFqzCYw
	wqaXMjASrbvjDWpv2NVjsQx3dcyWfkuPnoFKWo8+Q7s9TI8eDRdTnQ/9WRLz
	Y1OkOm/y1+K/tic5EfowXLXSPlw1tSjqqimH7KqpxdSuGlttx5tivRHxYw6N
	iPiYH4350ZgfHYevSnqfOjOA/aA+NSIsJjdjcjMmN28+95QyTW7eeDJKmfp8
	GKcidS+nIvWgDzVq+kNN6VPHn9USbrmYEf3OaK6lyIjeiNVq+nzpw8BrbS94
	rYtGsYyDBnw9PeBrx5yJvRnwNQR8TKPGNGpUF6ZRo6Iwjfr9AsTGDgFi7XMn
	XaDzgjnQ+zda9olJr0OB8v/ZhmXo3hIBAA==
    }
    set xml [binary decode base64 $xml]
    set xml [zlib gunzip $xml]
    loader $handle $xml
}

# Constructor, see comment on top.

proc opcua::filesystem {handle foldername rootdir} {
    if {![fs_namevalid $foldername]} {
	return -code error "invalid folder '$foldername'"
    }
    if {![::file isdirectory $rootdir]} {
	return -code error "invalid root directory '$rootdir'"
    }
    set ns ::opcua::${handle}::_fs
    if {[::info exists ${ns}::root]} {
	return -code error "filesystem already setup"
    }
    # make opcua stuff visible in filesystem's namespace
    ::namespace eval $ns { ::namespace import ::opcua::* }
    if {[info $handle] ne "server"} {
	::namespace delete $ns
	return -code error "not a server handle"
    }
    set rootdir [file normalize $rootdir]

    # node identifier of the FileType ObjectType
    if {[catch {
	set ${ns}::type_file \
	    [lindex [translate $handle [root] / Types / ObjectTypes \
			 / BaseObjectType / FileType] 0]
    }]} {
	_fs_loadxml $handle
	set ${ns}::type_file \
	    [lindex [translate $handle [root] / Types / ObjectTypes \
			 / BaseObjectType / FileType] 0]
    }

    # node identifier of the FileDirectoryType ObjectType
    set ${ns}::type_dir \
	[lindex [translate $handle [root] / Types / ObjectTypes \
		     / BaseObjectType / FolderType / FileDirectoryType] 0]

    # remember native root directory
    set ${ns}::root $rootdir

    # open file handles indexed by UInt32 number, tuples of open file and name
    array set ${ns}::fh {}

    # open counters for files indexed by file names, integers
    array set ${ns}::fo {}

    # session to file handles list indexed by session identifier
    array set ${ns}::sf {}

    # node to filename mapping for files indexed by node identifier
    array set ${ns}::nf {}

    # node to filename mapping for directories indexed by node identifier
    array set ${ns}::nd {}

    # UInt32 number for generating open file handles
    set ${ns}::id 0

    # return new UInt32 identifier for open file handles
    proc ${ns}::_getid {} {
	variable id	;# the identifier counter
	incr id
	set id [expr {$id & 0xFFFFFFFF}]
	if {$id == 0} {
	    incr id
	}
	return $id
    }

    # FileType.Open method
    # in:  Byte mode (bits: 0=read 1=write 2=trunc 4=append)
    # out: UInt32 open file handle
    # exc: BADNOTREADABLE	0x803a0000
    #      BADNOTWRITABLE	0x803b0000
    #      BADINVALIDSTATE	0x80af0000
    #      BADINVALIDARGUMENT	0x80ab0000
    #      BADNOTFOUND		0x803e0000
    #      BADUNEXPECTEDERROR	0x80010000
    proc ${ns}::Open {handle node meth types mode} {
	variable nf	;# node to native file mapping
	variable fh	;# array of open file handles
	variable fo	;# array of open counters
	variable sf	;# array of session to file handle lists
	if {![::info exists nf($node)]} {
	    set msg "node does not exist"
	    set ::errorCode [list opcua FileType.Open 0x803e0000 $msg]
	    return -code break $msg
	}
	_cleanup $handle
	# limit total number of open files to 256
	if {[llength [array names fh]] > 255} {
	    set msg "too many open files"
	    set ::errorCode [list opcua FileType.Open 0x80af0000 $msg]
	    return -code break $msg
	}
	set s [session $handle current]
	if {[::info exists sf($s)]} {
	    # limit number of open files per session to 32
	    if {[llength $sf($s)] > 31} {
		set msg "too many open files in session"
		set ::errorCode [list opcua FileType.Open 0x80af0000 $msg]
		return -code break $msg
	    }
	}
	set m [list BINARY]
	if {($mode & 3) == 1} {
	    lappend m RDONLY
	} elseif {($mode & 3) == 2} {
	    lappend m WRONLY
	} elseif {($mode & 3) == 3} {
	    lappend m RDWR
	}
	if {$mode & 4} {
	    lappend m TRUNC
	    if {$mode & 2} {
		lappend m CREAT
	    }
	}
	if {$mode & 8} {
	    lappend m APPEND
	}
	set id [_getid]
	set name $nf($node)
	if {[catch {::open $name $m} f]} {
	    set ::errorCode [list opcua FileType.Open 0x80ab0000 $f]
	    return -code break $f
	}
	set fh($id) [list $f $name]
	lappend sf($s) $id
	incr fo($name)
	return $id
    }

    # FileType.GetPosition method
    # in:  UInt32 open file handle
    # out: UInt64 position
    # exc: BADINVALIDARGUMENT	0x80ab0000
    proc ${ns}::GetPosition {handle node meth types id} {
	variable fh	;# array of open file handles
	variable sf	;# array of session to file handle lists
	if {![::info exists fh($id)]} {
	    set msg "invalid file handle"
	    set ::errorCode [list opcua FileType.GetPosition 0x80ab0000 $msg]
	    return -code break $msg
	}
	set s [session $handle current]
	if {![::info exists sf($s)] || $id ni $sf($s)} {
	    set msg "invalid file handle"
	    set ::errorCode [list opcua FileType.GetPosition 0x80ab0000 $msg]
	    return -code break $msg
	}
	lassign $fh($id) f name
	if {[catch {::tell $f $pos} n]} {
	    set ::errorCode [list opcua FileType.GetPosition 0x80ab0000 $n]
	    return -code break $n
	}
	return $n
    }

    # FileType.SetPosition method
    # in:  UInt32 open file handle
    #      UInt64 new position
    # out: void
    # exc: BADINVALIDARGUMENT	0x80ab0000
    proc ${ns}::SetPosition {handle node meth types id pos} {
	variable fh	;# array of open file handles
	variable sf	;# array of session to file handle lists
	if {![::info exists fh($id)]} {
	    set msg "invalid file handle"
	    set ::errorCode [list opcua FileType.SetPosition 0x80ab0000 $msg]
	    return -code break $msg
	}
	set s [session $handle current]
	if {![::info exists sf($s)] || $id ni $sf($s)} {
	    set msg "invalid file handle"
	    set ::errorCode [list opcua FileType.SetPosition 0x80ab0000 $msg]
	    return -code break $msg
	}
	lassign $fh($id) f name
	if {[catch {::seek $f $pos start} err]} {
	    set ::errorCode [list opcua FileType.SetPosition 0x80ab0000 $err]
	    return -code break $err
	}
	return {}
    }

    # FileType.Close method
    # in:  UInt32 open file handle
    # out: void
    # exc: BADINVALIDARGUMENT	0x80ab0000
    proc ${ns}::Close {handle node meth types id} {
	variable nf	;# node to native file mapping
	variable fh	;# array of open file handles
	variable fo	;# array of open counters
	variable sf	;# array of session to file handle lists
	if {[::info exists fh($id)]} {
	    set s [session $handle current]
	    if {![::info exists sf($s)] || $id ni $sf($s)} {
		set msg "invalid file handle"
		set ::errorCode \
		    [list opcua FileType.Close 0x80ab0000 $msg]
		return -code break $msg
	    }
	    lassign $fh($id) f name
	    catch {::close $f}
	    unset fh($id)
	    incr fo($name) -1
	    set sf($s) [lsearch -inline -not -all -exact $sf($s) $id]
	    _cleanup $handle
	} else {
	    set msg "invalid file handle"
	    set ::errorCode [list opcua FileType.Close 0x80ab0000 $msg]
	    return -code break $msg
	}
	return {}
    }

    # FileType.Read method
    # in:  UInt32 open file handle
    #      Int32 number bytes to read
    # out: ByteString bytes read
    # exc: BADINVALIDSTATE	0x80af0000
    #      BADINVALIDARGUMENT	0x80ab0000
    #      BADUNEXPECTEDERROR	0x80010000
    proc ${ns}::Read {handle node meth types id count} {
	variable fh	;# array of open file handles
	variable sf	;# array of session to file handle lists
	if {![::info exists fh($id)]} {
	    set msg "invalid file handle"
	    set ::errorCode [list opcua FileType.Read 0x80ab0000 $msg]
	    return -code break $msg
	}
	set s [session $handle current]
	if {![::info exists sf($s)] || $id ni $sf($s)} {
	    set msg "invalid file handle"
	    set ::errorCode [list opcua FileType.Read 0x80ab0000 $msg]
	    return -code break $msg
	}
	lassign $fh($id) f name
	if {[catch {::read $f $count} n]} {
	    set ::errorCode [list opcua FileType.Read 0x80af0000 $n]
	    return -code break $n
	}
	return $n
    }

    # FileType.Write method
    # in:  UInt32 open file handle
    #      ByteString bytes to write
    # out: void
    # exc: BADINVALIDSTATE	0x80af0000
    #      BADNOTWRITABLE	0x803b0000
    #      BADINVALIDARGUMENT	0x80ab0000
    proc ${ns}::Write {handle node meth types id bytes} {
	variable fh	;# array of open file handles
	variable sf	;# array of session to file handle lists
	if {![::info exists fh($id)]} {
	    set msg "invalid file handle"
	    set ::errorCode [list opcua FileType.Write 0x80ab0000 $msg]
	    return -code break $msg
	}
	set s [session $handle current]
	if {![::info exists sf($s)] || $id ni $sf($s)} {
	    set msg "invalid file handle"
	    set ::errorCode [list opcua FileType.Write 0x80ab0000 $msg]
	    return -code break $msg
	}
	lassign $fh($id) f name
	if {[catch {::puts -nonewline $f $bytes} err]} {
	    set ::errorCode [list opcua FileType.Write 0x80af0000 $msg]
	    return -code break $n
	}
	return {}
    }

    # Data source for FileType.Size variable
    proc ${ns}::Size {handle node op args} {
	if {$op eq "read"} {
	    variable nf		;# node to native file mapping
	    set value 0
	    if {![catch {parent $handle $node} obj]} {
		if {[::info exists nf($obj)]} {
		    catch {set value [::file size $nf($obj)]}
		}
	    }
	    return [list UInt64 $value]
	}
	return {}
    }

    # Data source for FileType.OpenCount variable
    proc ${ns}::OpenCount {handle node op args} {
	if {$op eq "read"} {
	    variable nf		;# node to native file mapping
	    variable fo		;# array of open counters
	    set count 0
	    if {![catch {parent $handle $node} obj]} {
		if {[::info exists fo($nf($obj))]} {
		    set count $fo($nf($obj))
		}
	    }
	    return [list UInt16 $count]
	}
	return {}
    }

    # Data source for FileType.Writable variable
    proc ${ns}::Writable {handle node op args} {
	if {$op eq "read"} {
	    variable nf		;# node to native file mapping
	    set value 0
	    if {![catch {parent $handle $node} obj]} {
		if {[::info exists nf($obj)]} {
		    catch {set value [::file writable $nf($obj)]}
		}
	    }
	    return [list Boolean $value]
	}
	return {}
    }

    # Data source for FileType.UserWritable variable
    proc ${ns}::UserWritable {handle node op args} {
	if {$op eq "read"} {
	    variable nf		;# node to native file mapping
	    set value 0
	    if {![catch {parent $handle $node} obj]} {
		if {[::info exists nf($obj)]} {
		    catch {set value [::file writable $nf($obj)]}
		}
	    }
	    return [list Boolean $value]
	}
	return {}
    }

    # FileDirectoryType.CreateDirectory method
    # in:  String name of directory to create
    # out: NodeId of directory
    # exc: BADBROWSENAMEDUPLICATED	0x80610000
    #      BADUSERACCESSDENIED		0x801f0000
    proc ${ns}::CreateDirectory {handle node meth types name} {
	variable nd		;# node to native dir mapping
	variable type_dir	;# ObjectType of FileDirectoryType
	if {![fs_namevalid $name]} {
	    set msg "invalid directory name"
	    set ::errorCode \
		[list opcua FileDirectoryType.CreateDirectory 0x801f0000 $msg]
	    return -code break $msg
	}
	set dir [file join $nd($node) $name]
	if {[catch {::file mkdir $dir} err]} {
	    set ::errorCode \
		[list opcua FileDirectoryType.CreateDirectory 0x801f0000 $err]
	    return -code break $err
	}
	# automatic numeric node identifer
	set id [add $handle Object "i=0" $node Organizes $name $type_dir]
	return [list NodeId $id]
    }

    # FileDirectoryType.CreateFile method
    # in:  String name of file to create
    #      Boolean when true keep file opened
    # out: NodeId of file
    #      UInt32 open file handle or 0
    # exc: BADBROWSENAMEDUPLICATED	0x80610000
    #      BADUSERACCESSDENIED		0x801f0000
    proc ${ns}::CreateFile {handle node meth types name mode} {
	variable nd		;# node to native dir mapping
	variable nf		;# node to native file mapping
	variable fh		;# array of open file handles
	variable fo		;# array of open counters
	variable sf		;# array of session to file handle lists
	variable type_file	;# ObjectType of FileType
	if {![fs_namevalid $name]} {
	    set msg "invalid file name"
	    set ::errorCode \
		[list opcua FileDirectoryType.CreateFile 0x801f0000 $msg]
	    return -code break $msg
	}
	set s [session $handle current]
	if {$mode} {
	    _cleanup $handle
	    # limit total number of open files to 256
	    if {[llength [array names fh]] > 255} {
		set msg "too many open files"
		set ::errorCode [list opcua FileType.Open 0x80af0000 $msg]
		return -code break $msg
	    }
	    if {[::info exists sf($s)]} {
		# limit number of open files per session to 32
		if {[llength $sf($s)] > 31} {
		    set msg "too many open files in session"
		    set ::errorCode [list opcua FileType.Open 0x80af0000 $msg]
		    return -code break $msg
		}
	    }
	}
	set file [file join $nf($node) $name]
	set found 0
	foreach n [array names nf] {
	    if {$nf($n) eq $file} {
		incr found
		break
	    }
	}
	if {$found && $mode} {
	    set id [_getid]
	    set name $nf($n)
	    if {[catch {::open $name {RDWR CREAT BINARY}} f]} {
		set ::errorCode \
		    [list opcua FileDirectoryType.CreateFile 0x801f0000 $f]
		return -code break $f
	    }
	    set fh($id) [list $f $name]
	    incr fo($name)
	    lappend sf($s) $id
	    return [list $n $id]
	}
	if {$found} {
	    return [list $n 0]
	}
	if {[catch {::open $file {RDWR CREAT BINARY}} f]} {
	    set ::errorCode \
		[list opcua FileDirectoryType.CreateFile 0x801f0000 $f]
	    return -code break $f
	}
	# create new FileType object with automatic numeric node identifer
	set n [add $handle Object "i=0" $node Organizes $name $type_file]
	set nf($n) $name
	# setup data sources for Variables in FileType object
	fs_setdsrc $handle $n [::namespace current] {
	    OpenCount Size UserWritable Writable
	}
	if {$mode} {
	    set fh($id) [list $f $file]
	    incr fo($file)
	    lappend sf($s) $id
	    return [list $n $id]
	}
	catch {::close $f}
	return [list $n 0]
    }

    # FileDirectoryType.Delete method
    # in:  NodeId
    # out: void
    # exc: BADNOTFOUND		0x803e0000
    #      BADUSERACCESSDENIED	0x801f0000
    #      BADINVALIDSTATE	0x80af0000
    proc ${ns}::Delete {handle node meth types todel} {
	variable nf	;# node to native file mapping
	variable nd	;# node to native dir mapping
	if {[::info exists nf($todel)]} {
	    if {[catch {::file delete -- $nf($todel)} err]} {
		set ::errorCode \
		    [list opcua FileDirectoryType.Delete 0x80af0000 $err]
		return -code break $err
	    }
	} elseif {[::info exists nd($todel)]} {
	    foreach n [array names nf] {
		if {[string match $nd($todel)/* $nf($n)]} {
		    delete $handle Node $n 1
		    unset nf($n)
		}
	    }
	    if {[catch {::file delete -force -- $nd($todel)} err]} {
		set ::errorCode \
		    [list opcua FileDirectoryType.Delete 0x80af0000 $err]
		return -code break $err
	    }
	} else {
	    set msg "node not found"
	    set ::errorCode \
		[list opcua FileDirectoryType.Delete 0x803e0000 $msg]
	    return -code break $msg
	}
	delete $handle Node $todel 1
	return {}
    }

    # FileDirectoryType.MoveOrCopy method
    # in:  NodeId source item
    #      NodeId destination item
    #      Boolean make a copy
    #      String name of destination or copy
    # out: NodeId new item
    # exc: BADNOTFOUND		0x803e0000
    #      BADUSERACCESSDENIED	0x801f0000
    #      BADINVALIDSTATE	0x80af0000
    proc ${ns}::MoveOrCopy {handle node meth types src dst copy name} {
	variable nf		;# node to native file mapping
	variable nd		;# node to native dir mapping
	variable type_file	;# ObjectType of FileType
	variable type_dir	;# ObjectType of FileDirectoryType
	if {![::info exists nd($dst)]} {
	    set msg "destination node not found"
	    set ::errorCode \
		[list opcua FileDirectoryType.MoveOrCopy 0x803e0000 $msg]
	    return -code break $msg
	}
	set isfile 0
	if {[::info exists nf($src)]} {
	    set from $nf($src)
	    incr isfile
	} elseif {[::info exists nd($src)]} {
	    set from $nd($src)
	} else {
	    set msg "source node not found"
	    set ::errorCode \
		[list opcua FileDirectoryType.MoveOrCopy 0x803e0000 $msg]
	    return -code break $msg
	}
	if {$name eq {}} {
	    set name [::file tail $from]
	} elseif {![fs_namevalid $name]} {
	    set msg "invalid destination name"
	    set ::errorCode \
		[list opcua FileDirectoryType.MoveOrCopy 0x80af0000 $msg]
	    return -code break $msg
	}
	if {$copy} {
	    set cmd copy
	} else {
	    set cmd rename
	}
	set to [::file join $nd($dst) $name]
	if {[catch {::file $cmd -- $from $to} err]} {
	    set ::errorCode \
		[list opcua FileDirectoryType.MoveOrCopy 0x801f0000 $err]
	    return -code break $err
	}
	if {$copy} {
	    if {[::file isdirectory $to]} {
		# automatic numeric node identifer
		set n [add $handle Object "i=0" $dst Organizes $name $type_dir]
		set nd($n) $to
	    } else {
		# automatic numeric node identifer
		set n [add $handle Object "i=0" $dst Organizes $name $type_file]
		set nf($n) $to
		fs_setdsrc $handle $n [::namespace current] \
		    { OpenCount Size UserWritable Writable }
	    }
	} elseif {$isfile} {
	    delete $handle Node $src 1
	    unset nf($src)
	    # automatic numeric node identifer
	    set n [add $handle Object "i=0" $dst Organizes $name $type_file]
	    set nf($n) $to
	    fs_setdsrc $handle $n [::namespace current] \
		{ OpenCount Size UserWritable Writable }
	} else {
	    delete $handle Node $src 1
	    unset nd($src)
	    # automatic numeric node identifer
	    set n [add $handle Object "i=0" $dst Organizes $name $type_dir]
	    set nd($n) $to
	}
	return $n
    }

    # FileDirectoryType.CloseAll method (non-standard, only on top node)
    # in:  void
    # out: void
    # exc: void
    proc ${ns}::CloseAll {handle node meth types} {
	variable fh	;# array of open file handles
	variable fo	;# array of open counters
	variable sf	;# array of session to file handle lists
	foreach id [array names fh] {
	    lassign $fh($id) f name
	    catch {::close $f}
	    unset fh($id)
	}
	unset -nocomplain fo
	array set fo {}
	unset -nocomplain sf
	array set sf {}
	return {}
    }

    # FileDirectoryType.Rescan method (non-standard, only on top node)
    # in:  void
    # out: void
    # exc: void
    proc ${ns}::Rescan {handle node meth types} {
	variable top	;# top level node identifier
	variable root	;# native root directory
	variable nd	;# node to native dir mapping
	variable nf	;# node to native file mapping
	_cleanup $handle
	fs_populate $handle [::namespace current] $top $root
	# verify that files still exists, for which we have nodes
	foreach n [array names nf] {
	    if {![::file exists $nf($n)]} {
		unset nf($n)
		delete $handle Node $n 1
	    }
	}
	# verify that directories still exists, for which we have nodes
	foreach n [array names nd] {
	    if {![::file isdirectory $nd($n)]} {
		unset nd($n)
		delete $handle Node $n 1
	    }
	}
    }

    # cleanup orphaned sessions
    proc ${ns}::_cleanup {handle} {
	variable fh	;# array of open file handles
	variable fo	;# array of open counters
	variable sf	;# array of session to file handle lists
	set all [session $handle list]
	foreach s [array names sf] {
	    if {$s ni $all} {
		foreach id $sf($s) {
		    lassign $fh($id) f name
		    catch {::close $f}
		    unset fh($id)
		    incr fo($name) -1
		}
		unset sf($s)
	    }
	}
    }

    # wire methods to ObjectTypes
    _fs_setmeths $handle [set ${ns}::type_file] \
	$ns { Open GetPosition SetPosition Close Read Write }
    _fs_setmeths $handle [set ${ns}::type_dir] \
	$ns { CreateDirectory CreateFile Delete MoveOrCopy }

    # set MinimumSamplingInterval to a reasonable value
    # for all Variables of FileType
    foreach var { OpenCount Size UserWritable Writable } {
	set vid [translate $handle [set ${ns}::type_file] HasProperty $var]
	set vid [lindex $vid 0]
	write $handle $vid MinimumSamplingInterval Double 10000.0
    }

    # create toplevel node
    set of [lindex [translate $handle [root] / Objects] 0]
    # automatic numeric node identifer
    set ${ns}::top [add $handle Object "i=0" $of Organizes \
			$foldername [set ${ns}::type_dir]]
    set ${ns}::nd([set ${ns}::top]) $rootdir

    # non-standard helper methods with automatic numeric node identifiers
    add $handle SimpleMethod "i=0" [set ${ns}::top] \
	HasComponent {} "CloseAll" {} [list ${ns}::CloseAll $handle]
    add $handle SimpleMethod "i=0" [set ${ns}::top] \
	HasComponent {} "Rescan" {} [list ${ns}::Rescan $handle]

    # fill with files/directories which are already online
    fs_populate $handle $ns [set ${ns}::top] $rootdir
}

# Destructor, experimental, not exposed in opcua ensemble.

proc opcua::fsdestroy {handle} {
    set ns ::opcua::${handle}::_fs
    if {![::namespace exists $ns]} {
	return
    }
    if {[array exists ${ns}::fh]} {
	foreach id [array names ${ns}::fh] {
	    catch {::close [set ${ns}::fh($id)]}
	}
    }
    if {[array exists ${ns}::nf]} {
	foreach n [array names ${ns}::nf] {
	    catch {delete $handle Node $n 1}
	}
    }
    if {[array exists ${ns}::nd]} {
	foreach n [array names ${ns}::nd] {
	    catch {delete $handle Node $n 1}
	}
    }
    if {[::info exists ${ns}::top]} {
	catch {delete $handle Node [set ${ns}::top] 1}
    }
    ::namespace delete $ns
}

# Rescan, not exposed in opcua ensemble.

proc opcua::fsrescan {handle} {
    set ns ::opcua::${handle}::_fs
    if {![::namespace exists $ns]} {
	return
    }
    tailcall ${ns}::Rescan $handle _ _ _
}

# Make opcua::filesystem visible in opcua ensemble.

namespace eval ::opcua {
    apply [list ns {
	if {[::info command $ns] eq {}} return
	set cmds [::namespace ensemble configure $ns -subcommands]
	lappend cmds filesystem
	::namespace ensemble configure $ns -subcommands $cmds
	::namespace export {[a-z]*}
    } [::namespace current]] [::namespace current]
}
