# sqlmodel.tcl --
#
# SQLite based model of an open62541 address space.
#
# Copyright (c) 2023-2024 Christian Werner <chw at ch minus werner dot de>
#
# See the file "license.terms" for information on usage and redistribution of
# this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# Alternate data model describing OPC/UA address space using SQLite
# instead of XML. In contrast to a Companion Spec in XML the model
# consists of the entire address space including namespace zero.
# Thus, the import function first detects exising nodes and references
# and creates only missing items. The following commands are exported
# from the "opcua" namespace.
#
# opcua::sqlmodel::export handle -file name
#
#   Exports the OPC/UA address space of "handle" to the SQLite
#   database "name". The file is created, temporarily opened,
#   written, and finally closed.
#
# opcua::sqlmodel::export handle -db db ?-schema schema?
#
#   Exports the OPC/UA address space of "handle" to the SQLite
#   database handle "db" which must refer to an open, writable
#   SQLite database. Optional "schema" is the table name prefix
#   in case an attached database below "db" shall be written.
#
# opcua::sqlmodel::export handle -data varname
#
#   Exports the OPC/UA address space of "handle" to variable
#   "varname" as a serializaton of an SQLite database.
#
# opcua::sqlmodel::export handle -chan chan
#
#   Exports the OPC/UA address space of "handle" to an open
#   and writable channel "chan" which is written with a
#   a serializaton of an SQLite database.
#
# opcua::sqlmodel::import handle -file name
#
#   Imports into the OPC/UA address space of "handle" from the
#   SQLite database "name". Only non-existing nodes, references,
#   and data type information is imported. The database file
#   is temporarily opened, read, and finally closed.
#
# opcua::sqlmodel::import handle -db db ?-schema schema?
#
#   Imports into the OPC/UA address space of "handle" from the
#   SQLite database handle "db" which must refer to an opened,
#   readable SQLite database. Optional "schema" is the table name
#   prefix in case an attached database below "db" shall be used
#   for the import. Only non-existing nodes, references, and data
#   type information is imported.
#
# opcua::sqlmodel::import handle -data value ?-schema schema?
#
#   Imports into the OPC/UA address space of "handle" from the
#   serialized SQLite database in "value".
#
# opcua::sqlmodel::import handle -chan chan ?-schema schema?
#
#   Imports into the OPC/UA address space of "handle" from a
#   serialized SQLite database which is read from the open and
#   readable channel "chan".
#
# opcua::sqlmodel::ns0full handle
#
#   Loads full namespace zero into "handle" from a local database
#   file "ns0.db.gz" which must be located in the directory where
#   this package has been installed.
#
# opcua::sqlmodel::mkspecsdb dbname ?tag?
#
#   Tries to create a database of OPC/UA companion specs into "dbname".
#   The XML nodeset data is obtained from a ZIP which is downloaded
#   from github. "tag" is the git tag, "latest" is the default tag.
#   The database is space optimized and has the tables "Models",
#   "Requires", and "UNECE" which provide meta information of the
#   companion specs and the corresponding XML nodeset data in gzip
#   format.
#
# opcua::sqlmodel::loadspec handle name ?dbname?
#
#   Looks up the companion spec "name" in the database "dbname" or
#   if "dbname" is omitted in "$::env(HOME)/uaspecs.db" and loads the
#   corresponding XML into "handle" which must refer to a server object.
#   "name" may be given as URL or as short name, e.g. "DI". The
#   loading process inspects both the current server address space
#   and the database and tries to resolve required companion specs
#   automatically.
#
# opcua::sqlmodel::listspecs ?dbname?
#
#   List companion specs in the database "dbname" or if "dbname" is
#   omitted in "$::env(HOME)/uaspecs.db". The result is a list of
#   alternating short names and URLs.
#
# opcua::sqlmodel::getunece ?dbname?
#
#   Return UNECE information from database "dbname" or if "dbname" is
#   omitted from "$::env(HOME)/uaspecs.db" as list of alternating
#   UNECE code, unit identifier, display name, and description.

package provide topcua::sqlmodel 0.1

package require topcua
package require sqlite3

namespace eval ::opcua::sqlmodel {

    ::namespace import ::opcua::*

    ######################################################################
    # Database schema template, "@" must be replaced by schema,
    # "main." or "" for main SQLite database, "other." for an
    # attached database in case of "ATTACH 'file' AS other".
    # The VIEWs are only for convenience when browsing with
    # an SQLite database browser like tksqlite.

    variable _db_schema {
	CREATE TEMP TABLE TempNodes(
	    NodeId VARCHAR PRIMARY KEY NOT NULL,

	    -- Reference to NodeClasses
	    NodeClass INTEGER NOT NULL DEFAULT 0,
	    BrowseName VARCHAR NOT NULL,

	    -- Reference to LocalizedTexts
	    DisplayName INTEGER NOT NULL DEFAULT 0,

	    -- Reference to LocalizedTexts
	    Description INTEGER,
	    WriteMask INTEGER NOT NULL,
	    UserWriteMask INTEGER NOT NULL,
	    IsAbstract INTEGER,
	    Symmetric INTEGER,

	    -- Reference to LocalizedTexts
	    InverseName INTEGER,

	    ContainsNoLoops INTEGER,
	    EventNotifier INTEGER,
	    Value TEXT,

	    -- Node identifier
	    ParentId VARCHAR,

	    -- Node identifier
	    ReferenceId VARCHAR,

	    -- Node identifier
	    ReferenceTypeId VARCHAR,

	    -- Node identifier
	    DataType VARCHAR,

	    ValueRank INTEGER,
	    ArrayDimensions VARCHAR,
	    AccessLevel INTEGER,
	    UserAccessLevel INTEGER,
	    MinimumSamplingInterval DOUBLE,
	    Historizing INTEGER,
	    Executable INTEGER,
	    UserExecutable INTEGER,

	    -- Reference to DataTypeDescriptions
	    DataTypeDefinition INTEGER,

	    RolePermissions INTEGER,
	    UserRolePermissions INTEGER,
	    AccessRestrictions INTEGER,
	    AccessLevelEx INTEGER
	);

	CREATE TABLE @Nodes(
	    NodeId VARCHAR PRIMARY KEY NOT NULL,

	    -- Reference to NodeClasses
	    NodeClass INTEGER NOT NULL DEFAULT 0,
	    BrowseName VARCHAR NOT NULL,

	    -- Reference to LocalizedTexts
	    DisplayName INTEGER NOT NULL DEFAULT 0,

	    -- Reference to LocalizedTexts
	    Description INTEGER,
	    WriteMask INTEGER NOT NULL,
	    UserWriteMask INTEGER NOT NULL,
	    IsAbstract INTEGER,
	    Symmetric INTEGER,

	    -- Reference to LocalizedTexts
	    InverseName INTEGER,

	    ContainsNoLoops INTEGER,
	    EventNotifier INTEGER,
	    Value TEXT,

	    -- ROWID of node identifier
	    ParentId INTEGER,

	    -- ROWID of node identifier
	    ReferenceId INTEGER,

	    -- ROWID of node identifier
	    ReferenceTypeId INTEGER,

	    -- ROWID of node identifier
	    DataType INTEGER,

	    ValueRank INTEGER,
	    ArrayDimensions VARCHAR,
	    AccessLevel INTEGER,
	    UserAccessLevel INTEGER,
	    MinimumSamplingInterval DOUBLE,
	    Historizing INTEGER,
	    Executable INTEGER,
	    UserExecutable INTEGER,

	    -- Reference to DataTypeDescriptions
	    DataTypeDefinition INTEGER,

	    RolePermissions INTEGER,
	    UserRolePermissions INTEGER,
	    AccessRestrictions INTEGER,
	    AccessLevelEx INTEGER
	);

	CREATE TABLE @LocalizedTexts(
	    "Key" INTEGER NOT NULL,
	    Locale VARCHAR NOT NULL DEFAULT "",
	    Text VARCHAR NOT NULL,
	    PRIMARY KEY("Key", Locale)
	);

	CREATE TABLE @NodeClasses(
	    "Key" INTEGER PRIMARY KEY NOT NULL,
	    Name VARCHAR NOT NULL
	);

	CREATE TEMP TABLE TempDataTypeDescriptions(
	    "Key" INTEGER PRIMARY KEY NOT NULL,

	    -- Node identifier
	    DefaultEncodingId VARCHAR NOT NULL,

	    -- Node identifier
	    BaseDataType VARCHAR NOT NULL,
	    StructureType INTEGER NOT NULL
	);

	CREATE TABLE @DataTypeDescriptions(
	    "Key" INTEGER PRIMARY KEY NOT NULL,

	    -- ROWID of node identifier
	    DefaultEncodingId INTEGER,

	    -- ROWID of node identifier
	    BaseDataType INTEGER,
	    StructureType INTEGER NOT NULL
	);

	CREATE TEMP TABLE TempStructureFields(
	    "Key" INTEGER PRIMARY KEY NOT NULL,

	    -- Reference to DataTypeDescriptions
	    DataTypeDescription INTEGER NOT NULL,

	    Name VARCHAR NOT NULL,

	    -- Reference to LocalizedTexts
	    Description INTEGER,

	    -- Node identifier
	    DataType VARCHAR NOT NULL,
	    ValueRank INTEGER NOT NULL,
	    ArrayDimensions VARCHAR,
	    MaxStringLength INTEGER NOT NULL DEFAULT 0,
	    IsOptional INTEGER NOT NULL DEFAULT 0
	);

	CREATE UNIQUE INDEX TempStructureFieldsIndex1
	    ON TempStructureFields(DataTypeDescription, Name);

	CREATE TABLE @StructureFields(
	    "Key" INTEGER PRIMARY KEY NOT NULL,

	    -- Reference to DataTypeDescriptions
	    DataTypeDescription INTEGER NOT NULL,

	    Name VARCHAR NOT NULL,

	    -- Reference to LocalizedTexts
	    Description INTEGER,

	    -- ROWID of node identifier
	    DataType INTEGER,
	    ValueRank INTEGER NOT NULL,
	    ArrayDimensions VARCHAR,
	    MaxStringLength INTEGER NOT NULL DEFAULT 0,
	    IsOptional INTEGER NOT NULL DEFAULT 0
	);

	CREATE UNIQUE INDEX @StructureFieldsIndex1
	    ON StructureFields(DataTypeDescription, Name);

	CREATE TEMP TABLE TempReferences(
	    NodeId VARCHAR NOT NULL,

	    -- Node identifier
	    Source VARCHAR NOT NULL,

	    -- Node identifier
	    Target VARCHAR NOT NULL,

	    IsForward INTEGER NOT NULL DEFAULT 1,
	    PRIMARY KEY(NodeId, Source, Target, IsForward)
	);

	CREATE TABLE @"References"(
	    -- ROWID of node identifier
	    NodeId INTEGER,

	    -- ROWID of node identifier
	    Source INTEGER,

	    -- ROWID of node identifier
	    Target INTEGER,

	    IsForward INTEGER NOT NULL DEFAULT 1,
	    PRIMARY KEY(NodeId, Source, Target, IsForward)
	);

	CREATE TABLE @Namespaces(
	    "Index" INTEGER PRIMARY KEY NOT NULL,
	    URL VARCHAR NOT NULL
	);

	CREATE VIEW @NodesView AS SELECT
	    n.NodeId AS NodeId,
	    (SELECT nc.Name FROM NodeClasses AS nc
	     WHERE nc."Key" = n.NodeClass) AS NodeClass,
	    n.BrowseName AS BrowseName,
	    (SELECT lt.Text FROM LocalizedTexts AS lt
	     WHERE lt."Key" = n.DisplayName
	     ORDER BY lt.Locale LIMIT 1) AS DisplayName,
	    (SELECT lt.Text FROM LocalizedTexts AS lt
	     WHERE lt."Key" = n.Description
	     ORDER BY lt.Locale LIMIT 1) AS Description,
	    n.WriteMask AS WriteMask,
	    n.UserWriteMask AS UserWriteMask,
	    n.IsAbstract AS IsAbstract,
	    n.Symmetric AS Symmetric,
	    (SELECT lt.Text FROM LocalizedTexts AS lt
	     WHERE lt."Key" = n.InverseName
	     ORDER BY lt.Locale LIMIT 1) AS InverseName,
	    n.ContainsNoLoops AS ContainsNoLoops,
	    n.EventNotifier AS EventNotifier,
	    n.Value AS Value,
	    (SELECT m.NodeId FROM Nodes AS m
	     WHERE n.ParentId = m.ROWID) AS ParentId,
	    (SELECT m.NodeId FROM Nodes AS m
	     WHERE n.ReferenceId = m.ROWID) AS ReferenceId,
	    (SELECT m.NodeId FROM Nodes AS m
	     WHERE n.ReferenceTypeId = m.ROWID) AS ReferenceTypeId,
	    (SELECT m.NodeId FROM Nodes AS m
	     WHERE n.DataType = m.ROWID) AS DataType,
	    n.ValueRank AS ValueRank,
	    n.ArrayDimensions AS ArrayDimensions,
	    n.AccessLevel AS AccessLevel,
	    n.UserAccessLevel AS UserAccessLevel,
	    n.MinimumSamplingInterval AS MinimumSamplingInterval,
	    n.Historizing AS Historizing,
	    n.Executable AS Executable,
	    n.UserExecutable AS UserExecutable,
	    (SELECT m.NodeId FROM Nodes AS m, DataTypeDescriptions AS dt
	     WHERE dt.DefaultEncodingId = m.ROWID AND
	     n.DataTypeDefinition = dt."Key") AS DataTypeDefinition,
	    n.RolePermissions AS RolePermissions,
	    n.UserRolePermissions AS UserRolePermissions,
	    n.AccessRestrictions AS AccessRestrictions,
	    n.AccessLevelEx AS AccessLevelEx
	    FROM Nodes AS n;

	CREATE VIEW @ReferencesView AS SELECT
	    (SELECT m.NodeId FROM Nodes AS m
	     WHERE r.NodeId = m.ROWID) AS NodeId,
	    (SELECT m.NodeId FROM Nodes AS m
	     WHERE r.Source = m.ROWID) AS Source,
	    (SELECT m.NodeId FROM Nodes AS m
	     WHERE r.Target = m.ROWID) AS Target,
	    r.IsForward AS IsForward
	    FROM "References" AS r;

	CREATE VIEW @DataTypesView AS SELECT
	    n.NodeId AS NodeId,
	    n.BrowseName AS BrowseName,
	    (SELECT lt.Text FROM LocalizedTexts AS lt
	     WHERE lt."Key" = n.DisplayName
	     ORDER BY lt.Locale LIMIT 1) AS DisplayName,
	    (SELECT lt.Text FROM LocalizedTexts AS lt
	     WHERE lt."Key" = n.Description
	     ORDER BY lt.Locale LIMIT 1) AS Description,
	    (SELECT m.NodeId FROM Nodes AS m
	     WHERE d.DefaultEncodingId = m.ROWID) AS DefaultEncodingId,
	    (SELECT m.NodeId FROM Nodes AS m
	     WHERE d.BaseDataType = m.ROWID) AS BaseDataType,
	    d.StructureType AS StructureType,
	    f.Name AS FieldName,
	    (SELECT lt.Text FROM LocalizedTexts AS lt
	     WHERE lt."Key" = f.Description
	     ORDER BY lt.Locale LIMIT 1) AS FieldDescription,
	    (SELECT m.NodeId FROM Nodes AS m
	     WHERE f.DataType = m.ROWID) AS DataType,
	    f.ValueRank AS ValueRank,
	    f.ArrayDimensions AS ArrayDimensions,
	    f.MaxStringLength AS MaxStringLength,
	    f.IsOptional AS IsOptional
	    FROM Nodes AS n, DataTypeDescriptions AS d, StructureFields AS f
	    WHERE n.DataTypeDefinition = d."Key" AND
	    d."Key" = f.DataTypeDescription;
    }

    # Delete all VIEWs and TABLEs from database. For handling the
    # "@" marker, see comment above.

    variable _db_clrschema {
	DROP VIEW IF EXISTS @NodesView;
	DROP VIEW IF EXISTS @ReferencesView;
	DROP VIEW IF EXISTS @DataTypesView;
	DROP TABLE IF EXISTS @Nodes;
	DROP TABLE IF EXISTS @LocalizedTexts;
	DROP TABLE IF EXISTS @NodeClasses;
	DROP TABLE IF EXISTS @DataTypeDescriptions;
	DROP TABLE IF EXISTS @StructureFields;
	DROP TABLE IF EXISTS @"References";
	DROP TABLE IF EXISTS @Namespaces;
	DROP TABLE IF EXISTS TempNodes;
	DROP TABLE IF EXISTS TempDataTypeDescriptions;
	DROP TABLE IF EXISTS TempReferences;
    }

    # Counter for "Key" in LocalizedTexts TABLE.

    variable _lcltxt_key

    # Add entry to "LocalizedTexts" TABLE and return its key.

    proc _db_addlcltxt {db map lcltxt} {
	variable _lcltxt_key
	set Locale [dict get $lcltxt locale]
	set Text [dict get $lcltxt text]
	$db eval [string map $map {
	    SELECT "Key" FROM @LocalizedTexts
	    WHERE Locale = $Locale AND Text = $Text
	}] data {
	    return $data(Key)
	}
	incr _lcltxt_key
	set key $_lcltxt_key
	$db eval [string map $map {
	    INSERT INTO @LocalizedTexts("Key", Locale, Text)
	    VALUES($key, $Locale, $Text)
	}]
	return $_lcltxt_key
    }

    # Add entry to "StructureFields" TABLE.

    proc _db_addstrfield {db map dtkey field} {
	set DataTypeDescription $dtkey
	dict with field {
	    if {![::info exists Description]} {
		set Description 0	;# use empty string
	    } elseif {[dict get $Description text] eq {}} {
		set Description 0	;# use empty string
	    } else {
		set Description [_db_addlcltxt $db $map $Description]
	    }
	    $db eval {
		INSERT INTO TempStructureFields(DataTypeDescription,
		    Name, Description, DataType, ValueRank,
		    ArrayDimensions, MaxStringLength, IsOptional)
		VALUES($DataTypeDescription,
		    $Name, $Description, $DataType, $ValueRank,
		    $ArrayDimensions, $MaxStringLength, $IsOptional)
	    }
	}
    }

    # Add entries to "DataTypeDescriptions" and "StructureFields" TABLEs
    # and return key of data type definition.

    proc _db_adddtdef {db map dtdef} {
	if {![dict exists $dtdef DefaultEncodingId] ||
	    ![dict exists $dtdef BaseDataType] ||
	    ![dict exists $dtdef StructureType]} {
	    return {}
	}
	set DefaultEncodingId [dict get $dtdef DefaultEncodingId]
	set BaseDataType [dict get $dtdef BaseDataType]
	set StructureType [dict get $dtdef StructureType]
	$db eval {
	    INSERT INTO TempDataTypeDescriptions
		(DefaultEncodingId, BaseDataType, StructureType)
	    VALUES($DefaultEncodingId, $BaseDataType, $StructureType)
	}
	set key [$db last_insert_rowid]
	foreach field [dict get $dtdef Fields] {
	    _db_addstrfield $db $map $key $field
	}
	return $key
    }

    # Add row to "Nodes" TABLE.

    proc _db_addnode {handle db map ncv NodeId NodeClass ParentId
		ReferenceId ReferenceTypeId} {
	upvar $ncv nc
	# numeric NodeClass for INSERT
	set ncls $nc($NodeClass)
	set sql1 {INSERT OR REPLACE INTO TempNodes(}
	append sql1 {NodeId, NodeClass, ParentId,}
	append sql1 { ReferenceId, ReferenceTypeId,}
	set sql2 {VALUES($NodeId, $ncls, $ParentId,}
	append sql2 { $ReferenceId, $ReferenceTypeId,}
	lappend atlist \
	    BrowseName DisplayName Description WriteMask UserWriteMask
	switch -exact -- $NodeClass {
	    Object {
		lappend atlist EventNotifier DataType
	    }
	    Method {
		lappend atlist Executable UserExecutable
	    }
	    ReferenceType {
		lappend atlist IsAbstract Symmetric InverseName
	    }
	    ObjectType - DataType {
		lappend atlist IsAbstract
	    }
	    VariableType {
		lappend atlist IsAbstract DataType ValueRank ArrayDimensions
	    }
	    Variable {
		lappend atlist AccessLevel UserAccessLevel \
		    MinimumSamplingInterval Historizing DataType \
		    ValueRank ArrayDimensions AccessLevelEx
	    }
	    View {
		lappend atlist EventNotifier ContainsNoLoops
	    }
	}
	lappend atlist RolePermissions UserRolePermissions AccessRestrictions
	foreach attr $atlist {
	    lappend rdlist $NodeId $attr {}
	}
	set nalist {}
	foreach {sc v} [mreadx $handle {} {*}$rdlist] attr $atlist {
	    if {$sc == 0} {
		if {$attr in {DisplayName Description InverseName}} {
		    if {![dict exists $v text] || ([dict get $v text] eq {})} {
			continue
		    }
		    set $attr [_db_addlcltxt $db $map $v]
		} else {
		    set $attr $v
		}
		append sql1 " " $attr ","
		append sql2 " \$" $attr ","
		lappend nalist $attr
	    }
	}
	if {("DataType" in $atlist) && ("DataType" ni $nalist)} {
	    # look for HasTypeDefinition reference instead
	    lassign [browse $handle $NodeId Forward HasTypeDefinition] \
		DataType _ _ _ _ _
	    if {$DataType ne {}} {
		append sql1 " DataType,"
		append sql2 " \$DataType,"
		lappend nalist DataType
	    }
	}
	if {$NodeClass eq "DataType"} {
	    if {![catch {read $handle $NodeId DataTypeDefinition} dtdef] &&
		($dtdef ne {})} {
		set DataTypeDefinition [_db_adddtdef $db $map $dtdef]
		if {$DataTypeDefinition ne {}} {
		    append sql1 " DataTypeDefinition,"
		    append sql2 " \$DataTypeDefinition,"
		    lappend nalist DataTypeDefinition
		}
	    }
	}
	if {$NodeClass in {Variable VariableType}} {
	    if {![catch {readjson $handle $NodeId Value} Value] &&
		($Value ne {}) && ($Value ne "null")} {
		# if it looks like a string, remove double quotes
		if {([string index $Value 0] eq "\"") &&
		    ([string index $Value end] eq "\"")} {
		    set Value [string range $Value 1 end-1]
		}
		append sql1 " Value,"
		append sql2 " \$Value,"
		lappend nalist Value
	    }
	}
	set sql1 [string trimright $sql1 ","]
	set sql2 [string trimright $sql2 ","]
	append sql1 ")"
	append sql2 ")"
	$db eval ${sql1}${sql2}
    }

    # Add row to "References" TABLE.

    proc _db_addref {db NodeId Source Target IsForward} {
	$db eval {
	    INSERT OR REPLACE INTO TempReferences
		(NodeId, Source, Target, IsForward)
	    VALUES($NodeId, $Source, $Target, $IsForward)
	}
    }

    # Write namespace info into database.

    proc _db_writens {handle db map} {
	set sql [string map $map {
	    INSERT INTO @Namespaces("Index", URL) VALUES($index, $url)
	}]
	foreach {index url} [namespace $handle] {
	    $db eval $sql
	}
    }

    # Find parent node identifier and reference type, prefer
    # matching or lowest namespace, search inverse hierarchical
    # references only.

    proc _db_getparent {handle nodeid} {
	set brlist {}
	foreach {parent brname dname ncls refid typeid} \
	    [browse $handle $nodeid Inverse /] {
	    if {[string match "ns=*;*" $parent]} {
		lappend brlist $parent $refid
	    } else {
		lappend brlist "ns=0;$parent" $refid
	    }
	}
	# prefer matching namespace if not zero
	if {[string match "ns=*;*" $nodeid]} {
	    scan $nodeid "ns=%d;%s" ns _
	    set ns [format "ns=%d;*" $ns]
	    foreach {parent invref} $brlist {
		if {[string match $ns $parent]} {
		    return [list $parent $invref]
		}
	    }
	}
	# now we may sort since all node identifiers have a "ns=*" pattern
	lassign [lsort -dictionary -index 0 -stride 2 $brlist] parent invref
	# and strip "ns=0;*" off, if any
	if {[string match "ns=0;*" $parent]} {
	    set parent [string range $parent 5 end]
	}
	return [list $parent $invref]
    }

    # Tree walker inserting OPC/UA info into database.

    proc _db_treewalk {handle db map exv ncv rootid} {
	upvar $exv exists
	upvar $ncv nc
	set refsfor {}
	if {![catch {mbrowse $handle \
		[list $rootid Forward] [list $rootid Inverse]} brlist]} {
	    foreach {nodeid brname dname ncls refid typeid} [join $brlist] {
		if {![::info exists exists($nodeid)]} {
		    lassign [_db_getparent $handle $nodeid] parent invref
		    if {($ncls eq "Method") && ($parent eq {})} {
			# Method without parent Object is bogus
			set exists($nodeid) 1
			continue
		    }
		    _db_addnode $handle $db $map nc $nodeid $ncls \
			$parent $invref $typeid
		    set exists($nodeid) 1
		    lappend refsfor $nodeid
		    _db_treewalk $handle $db $map exists nc $nodeid
		}
	    }
	}
	foreach rootid $refsfor {
	    if {![catch {mbrowse $handle \
		    [list $rootid Forward] [list $rootid Inverse]} brlist]} {
		if {$exists($rootid) < 0} {
		    # excluded
		    continue
		}
		lassign $brlist fwd inv
		foreach {nodeid brname dname ncls refid typeid} $fwd {
		    if {[::info exists exists($nodeid)] &&
			$exists($nodeid) < 0} {
			# excluded
			continue
		    }
		    _db_addref $db $refid $rootid $nodeid 1
		}
		foreach {nodeid brname dname ncls refid typeid} $inv {
		    if {[::info exists exists($nodeid)] &&
			$exists($nodeid) < 0} {
			# excluded
			continue
		    }
		    _db_addref $db $refid $rootid $nodeid 0
		}
	    }
	}
    }

    # Compact database by transferring temporary TABLEs to final
    # TABLEs with certain NodeId fields replaced by ROWIDs.

    proc _db_compact {db map} {
	set isql [string map $map {
	    INSERT INTO @Nodes
		(ROWID, NodeId, NodeClass, BrowseName,
		 DisplayName, Description, WriteMask, UserWriteMask,
		 IsAbstract, Symmetric, InverseName, ContainsNoLoops,
		 EventNotifier, Value, ParentId, ReferenceId,
		 ReferenceTypeId, DataType, ValueRank, ArrayDimensions,
		 AccessLevel, UserAccessLevel, MinimumSamplingInterval,
		 Historizing, Executable, UserExecutable,
		 DataTypeDefinition, RolePermissions, UserRolePermissions,
		 AccessRestrictions, AccessLevelEx)
	    SELECT ROWID, NodeId, NodeClass, BrowseName,
		DisplayName, Description, WriteMask, UserWriteMask,
		IsAbstract, Symmetric, InverseName, ContainsNoLoops,
		EventNotifier, Value, NULL, NULL,
		NULL, NULL, ValueRank, ArrayDimensions,
		AccessLevel, UserAccessLevel, MinimumSamplingInterval,
		Historizing, Executable, $d(UserExecutable),
		DataTypeDefinition, RolePermissions, UserRolePermissions,
		AccessRestrictions, AccessLevelEx
	     FROM TempNodes
	}]
	set rsql [string map $map {
	    SELECT ROWID AS ROWID, NodeId FROM TempNodes
	}]
	$db eval $isql
	array set nm {}
	$db eval $rsql d {
	    set nm($d(NodeId)) $d(ROWID)
	}
	set rsql [string map $map {
	    SELECT ROWID AS ROWID, ParentId, ReferenceId, ReferenceTypeId,
		DataType FROM TempNodes
	}]
	$db eval $rsql d {
	    set sql {UPDATE @Nodes SET}
	    set up 0
	    if {[::info exists nm($d(ParentId))]} {
		set key_1 $nm($d(ParentId))
		append sql { ParentId = $key_1,}
		incr up
	    }
	    if {[::info exists nm($d(ReferenceId))]} {
		set key_2 $nm($d(ReferenceId))
		append sql { ReferenceId = $key_2,}
		incr up
	    }
	    if {[::info exists nm($d(ReferenceTypeId))]} {
		set key_3 $nm($d(ReferenceTypeId))
		append sql { ReferenceTypeId = $key_3,}
		incr up
	    }
	    if {[::info exists nm($d(DataType))]} {
		set key_4 $nm($d(DataType))
		append sql { DataType = $key_4,}
		incr up
	    }
	    if {$up} {
		set sql [string trimright $sql ","]
		append sql { WHERE ROWID = $d(ROWID)}
		$db eval [string map $map $sql]
	    }
	}
	set isql [string map $map {
	    INSERT INTO @DataTypeDescriptions
		("Key", DefaultEncodingId, BaseDataType, StructureType)
	    SELECT "Key", NULL, NULL, StructureType
	    FROM TempDataTypeDescriptions
	}]
	set rsql [string map $map {
	    SELECT "Key", DefaultEncodingId, BaseDataType
	    FROM TempDataTypeDescriptions
	}]
	$db eval $isql
	$db eval $rsql d {
	    set sql {UPDATE @DataTypeDescriptions SET}
	    set up 0
	    if {[::info exists nm($d(DefaultEncodingId))]} {
		set key_1 $nm($d(DefaultEncodingId))
		append sql { DefaultEncodingId = $key_1,}
		incr up
	    }
	    if {[::info exists nm($d(BaseDataType))]} {
		set key_2 $nm($d(BaseDataType))
		append sql { BaseDataType = $key_2,}
		incr up
	    }
	    if {$up} {
		set sql [string trimright $sql ","]
		append sql { WHERE Key = $d(Key)}
		$db eval [string map $map $sql]
	    }
	}
	set isql [string map $map {
	    INSERT INTO @StructureFields
		("Key", DataTypeDescription, Name, Description,
		 DataType, ValueRank, ArrayDimensions,
		 MaxStringLength, IsOptional)
	    SELECT "Key", DataTypeDescription, Name, Description,
		NULL, ValueRank, ArrayDimensions,
		MaxStringLength, IsOptional
	    FROM TempStructureFields
	}]
	set rsql [string map $map {
	    SELECT "Key", DataType FROM TempStructureFields
	}]
	$db eval $isql
	$db eval $rsql d {
	    if {[::info exists nm($d(DataType))]} {
		set key $nm($d(DataType))
		$db eval [string map $map {
		    UPDATE @StructureFields
		    SET DataType = $key
		    WHERE Key = $d(Key)
		}]
	    }
	}
	set isql [string map $map {
	    INSERT INTO @"References"
		(ROWID, NodeId, Source, Target, IsForward)
	    SELECT ROWID, NULL, NULL, NULL, IsForward
	    FROM TempReferences
	}]
	set rsql [string map $map {
	    SELECT ROWID AS ROWID, * FROM TempReferences
	}]
	$db eval $isql
	$db eval $rsql d {
	    set sql {UPDATE @"References" SET}
	    set up 0
	    if {[::info exists nm($d(NodeId))]} {
		set key_1 $nm($d(NodeId))
		append sql { NodeId = $key_1,}
		incr up
	    }
	    if {[::info exists nm($d(Source))]} {
		set key_2 $nm($d(Source))
		append sql { Source = $key_2,}
		incr up
	    }
	    if {[::info exists nm($d(Target))]} {
		set key_3 $nm($d(Target))
		append sql { Target = $key_3,}
		incr up
	    }
	    if {$up} {
		set sql [string trimright $sql ","]
		append sql { WHERE ROWID = $d(ROWID)}
		$db eval [string map $map $sql]
	    }
	}
	$db eval [string map $map {
	    DROP TABLE TempNodes;
	    DROP TABLE TempDataTypeDescriptions;
	    DROP TABLE TempStructureFields;
	    DROP TABLE TempReferences;
	    CREATE INDEX @NodesIndex1 ON Nodes(ParentId);
	    CREATE INDEX @NodesIndex2 ON Nodes(ReferenceId);
	    CREATE INDEX @NodesIndex3 ON Nodes(DataType);
	}]
    }

    ######################################################################
    # Export OPC/UA address space given OPC/UA handle into sqlite
    # database given handle. Provide some variants of the function
    # to deal with files or an open database handle.

    proc _db_export {handle db schema} {
	variable _db_schema
	variable _db_clrschema
	variable _lcltxt_key
	set _lcltxt_key 0
	array set exists {}
	# exclude some dynamically numbered internal nodes,
	# which might have internal server methods bound
	# plus server specific stuff like diagnostics
	foreach path {
	    { Objects / Server / PublishSubscribe /
		1:LoadPubSubConfigurationFile }
	    { Objects / Server / PublishSubscribe /
		1:DeletePubSubConfiguration }
	    { Objects / Server / ServerDiagnostics }
	    { Objects / Server / VendorServerInfo }
	} {
	    if {![catch {translate $handle [root] / {*}$path} n]} {
		set n [lindex $n 0]
		set exists($n) -1
		foreach {_ n _ _ _ _ _ _} [tree $handle $n] {
		    set exists($n) -1
		}
	    }
	}
	if {$schema ne {}} {
	    append schema "."
	}
	set map [list @ $schema]
	set dbschema [string map $map $_db_schema]
	set dbclrschema [string map $map $_db_clrschema]
	# prime empty string into "LocalizedTexts"
	set string0 {INSERT INTO @LocalizedTexts VALUES(0, '', '')}
	set string0 [string map $map $string0]
	set string1 {}
	set n 0
	foreach cls {
	    {} Object Variable Method ObjectType
	    VariableType ReferenceType DataType View
	} {
	    append string1 {INSERT INTO @NodeClasses VALUES(}
	    append string1 $n ", '" $cls "');\n"
	    set nc($cls) $n
	    incr n
	}
	set string1 [string map $map $string1]
	try {
	    $db eval {BEGIN TRANSACTION}
	    $db eval $dbclrschema
	    $db eval $dbschema
	    $db eval $string0
	    $db eval $string1
	    _db_writens $handle $db $map
	    _db_treewalk $handle $db $map exists nc [root]
	    _db_compact $db $map
	} on error {result opts} {
	    $db eval ROLLBACK
	    return -code error -options $opts $result
	} finally {
	    catch {$db eval COMMIT}
	    catch {$db eval VACUUM}
	}
	return {}
    }

    proc export {handle args} {
	set schema {}
	if {[llength $args] % 2} {
	    return -code error -errorcode [list opcua::sqlmodel export -1] \
		"odd number of arguments"
	}
	foreach {option value} $args {
	    if {[catch {::tcl::prefix match -message "option" {
		-file -db -schema -data -channel
	    } $option} opt]} {
		return -code error \
		    -errorcode [list opcua::sqlmodel export -1] $opt
	    }
	    switch -exact -- $opt {
		-file {
		    set mode file
		    set name $value
		}
		-db {
		    set mode db
		    set name $value
		}
		-schema {
		    set schema $value
		}
		-data {
		    set mode data
		    set name $value
		}
		-channel {
		    set mode chan
		    set name $value
		}
	    }
	}
	if {![::info exists mode]} {
	    return -code error \
		-errorcode [list opcua::sqlmodel export -1] "mode missing"
	}
	if {$mode eq "db"} {
	    tailcall _db_export $handle $name $schema
	}
	if {$mode eq "data"} {
	    set db [::namespace current]::_dbtemp
	    sqlite3 $db :memory:
	    try {
		_db_export $handle $db {}
		upvar $name out
		set out [$db serialize]
	    } on error {result opts} {
		return -code error -options $opts $result
	    } finally {
		$db close
	    }
	    return {}
	}
	if {$mode eq "chan"} {
	    set db [::namespace current]::_dbtemp
	    sqlite3 $db :memory:
	    try {
		_db_export $handle $db {}
		::puts -nonewline $name [$db serialize]
	    } on error {result opts} {
		return -code error -options $opts $result
	    } finally {
		$db close
	    }
	    return {}
	}
	# mode is "file"
	set db [::namespace current]::_dbtemp
	sqlite3 $db file:[file normalize $name] -uri 1
	try {
	    _db_export $handle $db {}
	} on error {result opts} {
	    return -code error -options $opts $result
	} finally {
	    $db close
	}
	return {}
    }

    # Create a node from database.

    proc _db_makenode {handle db map nmvar bmvar fnvar ncvar nodeid} {
	upvar $nmvar nm
	upvar $bmvar bm
	upvar $ncvar nc
	upvar $fnvar fn
	set rowid $nm($nodeid)
	set data(NodeClass) {}
	$db eval [string map $map {
	    SELECT ROWID AS ROWID, * FROM @Nodes WHERE ROWID = $rowid
	}] data {
	    set data(NodeId) $nm($data(ROWID))
	    set data(BrowseName) $bm($data(ROWID))
	    if {[::info exists nm($data(ParentId))]} {
		set data(ParentId) $nm($data(ParentId))
	    }
	    if {[::info exists nm($data(ReferenceId))]} {
		set data(ReferenceId) $nm($data(ReferenceId))
	    }
	    if {[::info exists nm($data(ReferenceTypeId))]} {
		set data(ReferenceTypeId) $nm($data(ReferenceTypeId))
	    }
	    if {[::info exists nm($data(DataType))]} {
		set data(DataType) $nm($data(DataType))
	    }
	}
	if {![::info exists nc($data(NodeClass))]} {
	    return 0
	}
	set data(NodeClass) $nc($data(NodeClass))
	if {$data(NodeClass) ni {
	    Object Variable Method ObjectType VariableType
	    ReferenceType DataType View
	}} {
	    return 0
	}
	variable _lnodes
	if {[::info exists _lnodes($data(NodeId))]} {
	    if {$_lnodes($data(NodeId)) eq $data(NodeClass)} {
		return 0
	    }
	    # if automatically created node, delete it and go on
	    catch {
		delete $handle Node $data(NodeId) 1
		unset _lnodes($data(NodeId))
	    }
	}
	set att [attrs default $data(NodeClass)Attributes]
	set tsql [string map $map {
	    SELECT * FROM LocalizedTexts
	    WHERE "Key" = $key
	    ORDER BY Locale LIMIT 1
	}]
	set key $data(DisplayName)
	set lcltxt(Locale) {}
	set lcltxt(Text) {}
	$db eval $tsql lcltxt {}
	dict set att DisplayName locale $lcltxt(Locale)
	dict set att DisplayName text $lcltxt(Text)
	set key $data(Description)
	set lcltxt(Locale) {}
	set lcltxt(Text) {}
	$db eval $tsql lcltxt {}
	dict set att Description locale $lcltxt(Locale)
	dict set att Description text $lcltxt(Text)
	dict set att WriteMask $data(WriteMask)
	dict set att UserWriteMask $data(UserWriteMask)
	switch -exact -- $data(NodeClass) {
	    Object {
		dict set att EventNotifier $data(EventNotifier)
		add $handle Object $nodeid $data(ParentId) \
		    $data(ReferenceId) $data(BrowseName) $data(DataType) $att
	    }
	    Method {
		if {$data(ParentId) eq {}} {
		    # no parent, do nothing
		    return 0
		}
		# find old references to existing method(s)
		if {[catch {
		    translate $handle $data(ParentId) \
			$data(ReferenceId) $data(BrowseName)
		} mno]} {
		    unset mno
		}
		dict set att Executable $data(Executable)
		dict set att UserExecutable $data(UserExecutable)
		# add mapping to implementation like "opcua::loader"
		set mns 0
		scan $nodeid "ns=%d;" mns
		set mname [dict get $att DisplayName text]
		set mname ::opcua::${handle}::impl${mns}_$mname
		add $handle Method $nodeid $data(ParentId) \
		    $data(ReferenceId) {} $data(BrowseName) {} $mname $att
		# now delete old references
		if {[::info exists mno]} {
		    foreach {n _ _} $mno {
			delete $handle Reference $data(ParentId) \
			    $data(ReferenceId) $n 1 1
		    }
		}
	    }
	    ReferenceType {
		dict set att IsAbstract $data(IsAbstract)
		dict set att Symmetric $data(Symmetric)
		set key $data(InverseName)
		set lcltxt(Locale) {}
		set lcltxt(Text) {}
		$db eval $tsql lcltxt {}
		dict set att InverseName locale $lcltxt(Locale)
		dict set att InverseName text $lcltxt(Text)
		add $handle ReferenceType $nodeid $data(ParentId) \
		    $data(ReferenceId) $data(BrowseName) $att
	    }
	    ObjectType {
		dict set att IsAbstract $data(IsAbstract)
		add $handle ObjectType $nodeid $data(ParentId) \
		    $data(ReferenceId) $data(BrowseName) $att
	    }
	    DataType {
		dict set att IsAbstract $data(IsAbstract)
		add $handle DataType $nodeid $data(ParentId) \
		    $data(ReferenceId) $data(BrowseName) $att
	    }
	    VariableType {
		dict set att IsAbstract $data(IsAbstract)
		dict set att DataType $data(DataType)
		dict set att ValueRank $data(ValueRank)
		dict set att ArrayDimensions $data(ArrayDimensions)
		if {($data(ValueRank) >= [const VALUERANK_ONE_DIMENSION]) &&
		    ($data(ArrayDimensions) eq {})} {
		    dict set att ArrayDimensions [lrepeat $data(ValueRank) 0]
		}
		if {[catch {
			add $handle VariableType $nodeid $data(ParentId) \
			    $data(ReferenceId) $data(BrowseName) \
			    $data(ReferenceTypeId) $att
		}]} {
		    sef fn($nodeid) $data(ReferenceTypeId)
		    add $handle VariableType $nodeid $data(ParentId) \
			$data(ReferenceId) $data(BrowseName) {} $att
		}
	    }
	    Variable {
		dict set att AccessLevel $data(AccessLevel)
		dict set att UserAccessLevel $data(UserAccessLevel)
		dict set att MinimumSamplingInterval \
		    $data(MinimumSamplingInterval)
		dict set att Historizing $data(Historizing)
		dict set att DataType $data(DataType)
		dict set att ValueRank $data(ValueRank)
		dict set att ArrayDimensions $data(ArrayDimensions)
		if {($data(ValueRank) >= [const VALUERANK_ONE_DIMENSION]) &&
		    ($data(ArrayDimensions) eq {})} {
		    dict set att ArrayDimensions [lrepeat $data(ValueRank) 0]
		}
		if {[catch {
			add $handle Variable $nodeid $data(ParentId) \
			    $data(ReferenceId) $data(BrowseName) \
			    $data(ReferenceTypeId) $att
		}]} {
		    set fn($nodeid) $data(ReferenceTypeId)
		    add $handle Variable $nodeid $data(ParentId) \
			$data(ReferenceId) $data(BrowseName) {} $att
		}
	    }
	    View {
		dict set att EventNotifier $data(EventNotifier)
		dict set att ContainsNoLoops $data(ContainsNoLoops)
		add $handle View $nodeid $data(ParentId) \
		    $data(ReferenceId) $data(BrowseName) $att
	    }
	}
	return 1
    }

    # Tree walker setting up map of existing nodes etc.

    proc _db_twexisting {handle exv rootid} {
	upvar $exv exists
	if {![catch {mbrowse $handle \
		[list $rootid Forward] [list $rootid Inverse]} brlist]} {
	    foreach {nodeid brname dname ncls refid typeid} [join $brlist] {
		if {![::info exists exists($nodeid)]} {
		    set exists($nodeid) 1
		    _db_twexisting $handle exists $nodeid
		}
	    }
	}
    }

    # Generate types: enums and subtypes are made from inspecting
    # the tree below /Root/Types/DataTypes/BaseDataType. Structure
    # types must be made from the DataTypeDescriptions and
    # StructureFields information from the database.

    proc _db_gentypes {handle db map nmvar} {
	upvar $nmvar nm
	set defs {}	;# collects typedef commands for later
	array set currmap [types map $handle]
	set currtypes [array names currmap]
	unset currmap
	# enums and subtypes (nodeids of Variant and BaseDataType are equal)
	set tree [tree $handle [types nodeid Variant]]
	foreach {level nodeid brname dname cls refid typeid parent} $tree {
	    # only data types to be considered
	    if {$cls ne "DataType"} {
		continue
	    }
	    if {$nodeid in $currtypes} {
		continue
	    }
	    set kind {}
	    if {[catch {read $handle $nodeid DataTypeDefinition} def]} {
		# check for enum
		if {![catch {
		    translate $handle $nodeid HasProperty EnumStrings
		} eid]} {
		    set kind enum
		} elseif {![catch {
		    translate $handle $nodeid HasProperty EnumValues
		} eid]} {
		    set kind enum
		} elseif {($parent ne [types nodeid ExtensionObject]) &&
			  ($parent ne [types nodeid Variant])} {
		    # both, parent type and current type must not be abstract
		    if {![catch {read $handle $parent IsAbstract} isabstr] &&
			!$isabstr &&
			![catch {read $handle $nodeid IsAbstract} isabstr] &&
			!$isabstr} {
			# not an enum, could be a simple subtype
			set kind subtype
		    }
		}
	    } elseif {![dict exists $def StructureType]} {
		# not a struct, check for enum
		if {[dict exists $def Fields]} {
		    set kind enum
		} else {
		    # not an enum, ignored
		    continue
		}
	    } else {
		# all else ignored
		continue
	    }
	    switch -exact -- $kind {
		enum {
		    lappend defs \
			[list typedef $handle enum $brname $nodeid \
			     [types nodeid UInt32] UInt32]
		}
		subtype {
		    lappend defs \
			[list typedef $handle subtype $brname $nodeid $parent]
		}
	    }
	}
	# structures
	set fsql [string map $map {
	    SELECT * FROM @StructureFields
	    WHERE DataTypeDescription = $key
	}]
	$db eval [string map $map {
	    SELECT dt."Key" AS "Key",
		dt.DefaultEncodingId AS DefaultEncodingId,
		dt.StructureType AS StructureType,
		n.NodeId AS DataType, n.BrowseName AS BrowseName
	    FROM @DataTypeDescriptions AS dt, @Nodes AS n
	    WHERE dt."Key" = n.DataTypeDefinition AND
		dt.DefaultEncodingId IS NOT NULL
	}] dt {
	    set isopt 0
	    if {$dt(StructureType) == [const STRUCTURETYPE_STRUCTURE]} {
		set cmd [list typedef $handle struct]
	    } elseif {$dt(StructureType) == [const STRUCTURETYPE_UNION]} {
		set cmd [list typedef $handle union]
	    } elseif {$dt(StructureType) ==
		      [const STRUCTURETYPE_STRUCTUREWITHOPTIONALFIELDS]} {
		set cmd [list typedef $handle optstruct]
		set isopt 1
	    } else {
		continue
	    }
	    if {![catch {read $handle $dt(DataType) DataTypeDefinition}]} {
		# already defined
		continue
	    }
	    set dt(DefaultEncodingId) $nm($dt(DefaultEncodingId))
	    lappend cmd $dt(BrowseName) $dt(DataType) $dt(DefaultEncodingId)
	    set key $dt(Key)
	    set fcount 0
	    $db eval $fsql sf {
		if {$isopt} {
		    if {$sf(IsOptional)} {
			lappend cmd "optional"
		    } else {
			lappend cmd "mandatory"
		    }
		}
		lappend cmd $nm($sf(DataType))
		if {$sf(ValueRank) > [const VALUERANK_SCALAR]} {
		    lappend cmd *$sf(Name)
		} else {
		    lappend cmd $sf(Name)
		}
		incr fcount
	    }
	    if {$fcount == 0} {
		# abstract struct w/o fields, hopefully
		continue
	    }
	    lappend defs $cmd
	}
	if {[llength $defs]} {
	    typedef $handle begin
	    set max [llength $defs]
	    # multiple rounds since order of defs is undefined
	    while {1} {
		set ndefs {}
		foreach cmd $defs {
		    if {[catch {{*}$cmd}]} {
			lappend ndefs $cmd
		    }
		}
		set nmax [llength $ndefs]
		if {($nmax == 0) || ($nmax == $max)} {
		    break
		}
		set defs $ndefs
		set max $nmax
	    }
	    typedef $handle commit
	}

	# finally, let generic gentypes pick up XML definitions
	gentypes $handle
    }

    # Capture new node identifiers while iterating over the table
    # of nodes to be created.

    proc _db_oninit {nodeid cls} {
	variable _lnodes
	set _lnodes($nodeid) $cls
    }

    ######################################################################
    # Import from sqlite database into OPC/UA address space given
    # OPC/UA and database handles. Provide some variants of the
    # function to deal with files or an open database handle.

    proc _db_import {handle db schema} {
	if {[info $handle] ne "server"} {
	    return -code error -errorcode {opcua Internal 0 Good} \
		"not a server handle"
	}
	if {[lindex [state $handle] 0] ne "stopped"} {
	    return -code error -errorcode {opcua Internal 0 Good} \
		"server must be stopped"
	}
	set map [list @ $schema]

	# make namespace info
	array set nsm {}
	array set nsinfo [namespace $handle]
	foreach {key url} [lsort -integer [array names nsinfo]] {
	    lappend nsnames $url
	}
	set addns {}
	$db eval [string map $map {
	    SELECT "Index", URL FROM @Namespaces
	    WHERE "Index" > 1 ORDER By "Index"
	}] data {
	    if {[::info exists nsinfo($data(Index))] &&
		($data(URL) ne $nsinfo($data(Index)))} {
		set index [lsearch $nsnames $data(URL)]
		if {$index < 0} {
		    lappend addns $data(URL) $data(Index)
		} else {
		    set nsm($data(Index)) $index
		}
	    }
	    if {![::info exists nsinfo($data(Index))]} {
		lappend addns $data(URL) $data(Index)
	    }
	}
	foreach {url srcindex} $addns {
	    set index [add $handle Namespace $url]
	    set nsm($srcindex) $index
	    set nsinfo($index) $url
	}
	foreach index [array names nsm] {
	    set srcpat "ns=$index;*"
	    set nsm($srcpat) "ns=$nsm($index);"
	    set srcpat "$index:*"
	    set nsm($srcpat) "$nsm($index):"
	}
	unset -nocomplain data url srcindex index addns srcpat
	array set exists {}

	# nothing/NULL exists, just in case
	set exists() {}

	# tree walk for existing nodes
	_db_twexisting $handle exists [root]
	array set requires {}
	array set nc {}
	$db eval [string map $map {
	    SELECT "Key", Name FROM @NodeClasses
	}] data {
	    set nc($data(Key)) $data(Name)
	}

	# make table of nodes whose references need be fixed
	array set fn {}

	# make mapping tables for node identifiers and browse names
	array set nm {}
	array set bm {}
	array set requires {}
	$db eval [string map $map {
	    SELECT ROWID AS ROWID, NodeId, BrowseName FROM @Nodes
	}] data {
	    set nm($data(ROWID)) $data(NodeId)
	    # no syntactic overlap between ROWID and NodeId,
	    # thus reverse mapping later can be in "nm" array, too
	    set nm0($data(NodeId)) $data(ROWID)
	    set bm0($data(ROWID)) $data(BrowseName)
	}

	# rewrite Namespace indices, make "nm" array to have
	#  - ROWID to new NodeId mapping
	#  - new NodeId to ROWID mapping
	foreach pat [array names nsm ns=*] {
	    set srclen [string length $pat]
	    incr srclen -1	;# omit "*"
	    foreach id [array names nm0 $pat] {
		set newid $nsm($pat)[string range $id $srclen end]
		set nm($newid) $nm0($id)
		set nm($nm0($id)) $newid
		set requires($newid) {}
		unset nm0($id)
	    }
	}

	# take over remaining items from "nm0"
	foreach id [array names nm0] {
	    set nm($id) $nm0($id)
	    set requires($id) {}
	    unset nm0($id)
	}

	# rewrite BrowseNames, make "bm" array to have
	#  - ROWID to new BrowseName mapping
	foreach pat [array names nsm *:*] {
	    set srclen [string length $pat]
	    incr srclen -1	;# omit "*"
	    foreach {id name} [array get bm0] {
		if {[string match $pat $name]} {
		    set name $nsm($pat)[string range $name $srclen end]
		    set bm($id) $name
		    unset bm0($id)
		}
	    }
	}

	# take over remaining items from "bm0"
	foreach id [array names bm0] {
	    set bm($id) $bm0($id)
	    unset bm0($id)
	}
	unset -nocomplain data pat id newid name srclen nm0 bm0

	# fill "requires" array with dependent node identifiers
	$db eval [string map $map {
	    SELECT ROWID AS ROWID, ParentId, ReferenceId FROM @Nodes
	}] data {
	    set data(NodeId) $nm($data(ROWID))
	    if {[::info exists nm($data(ParentId))]} {
		lappend requires($data(NodeId)) $nm($data(ParentId))
	    }
	    if {[::info exists nm($data(ReferenceId))]} {
		lappend requires($data(NodeId)) $nm($data(ReferenceId))
	    }
	}
	$db eval [string map $map {
	    SELECT ROWID AS ROWID, DataType FROM @Nodes
	    WHERE DataTypeDefinition IS NULL
	}] data {
	    set data(NodeId) $nm($data(ROWID))
	    if {[::info exists nm($data(DataType))]} {
		lappend requires($data(NodeId)) $nm($data(DataType))
	    }
	}
	$db eval [string map $map {
	    SELECT ROWID AS ROWID, ReferenceTypeId FROM @Nodes
	    WHERE ReferenceTypeId IS NOT NULL
	}] data {
	    set data(NodeId) $nm($data(ROWID))
	    if {[::info exists nm($data(ReferenceTypeId))]} {
		lappend requires($data(NodeId)) $nm($data(ReferenceTypeId))
	    }
	}
	$db eval [string map $map {
	    SELECT n.ROWID AS ROWID, n.DataType AS DataType,
		dt.DefaultEncodingId AS DefaultEncodingId,
		dt.BaseDataType AS BaseDataType,
		sf.DataType AS FieldDataType
	    FROM @Nodes AS n, @DataTypeDescriptions AS dt,
		@StructureFields AS sf
	    WHERE n.DataTypeDefinition = dt."Key" AND
		sf.DataTypeDescription = dt."Key" AND
		dt.DefaultEncodingId IS NOT NULL
	}] data {
	    set data(NodeId) $nm($data(ROWID))
	    if {[::info exists nm($data(DataType))]} {
		lappend requires($data(NodeId)) $nm($data(DataType))
	    }
	    lappend requires($data(NodeId)) \
		$nm($data(DefaultEncodingId)) \
		$nm($data(BaseDataType)) \
		$nm($data(FieldDataType))
	}
	$db eval [string map $map {
	    SELECT Target FROM @"References" WHERE IsForward <> 0
	}] data {
	    set data(Target) $nm($data(Target))
	    lappend requires($data(Target))
	}
	$db eval [string map $map {
	    SELECT Source FROM @"References" WHERE IsForward = 0
	}] data {
	    set data(Source) $nm($data(Source))
	    lappend requires($data(Source))
	}

	# all exisiting nodes can be eliminated from requires
	foreach nodeid [array names requires] {
	    if {[::info exists exists($nodeid)]} {
		unset requires($nodeid)
	    }
	}

	# all "requires" lists must be unique
	foreach nodeid [array names requires] {
	    set requires($nodeid) [lsort -unique $requires($nodeid)]
	}

	# all existing nodes need be thrown out from require lists
	set all [array names exists]
	foreach nodeid [array names requires] {
	    set rest {}
	    foreach n $requires($nodeid) {
		if {$n eq $nodeid} {
		    continue
		}
		if {$n ni $all} {
		    lappend rest $n
		}
	    }
	    set requires($nodeid) $rest
	}

	# remember current state of affairs regarding methods
	set old_meths [list]
	foreach {nodeid _ _} [methods $handle] {
	    lappend old_meths $nodeid
	}

	# setup capture for newly created nodes, this info is used
	# later to delete superfluous automatically created nodes
	variable _lnodes
	array set _lnodes {}
	set oninit [oninitialize $handle]
	oninitialize $handle [::namespace current]::_db_oninit

	# the try block ensures cleanup of nodes
	try {
	    set counts [array size requires]
	    while {1} {
		# for each empty "requires" item create node and remove
		# it from "requires"
		foreach nodeid [array names requires] {
		    if {[llength $requires($nodeid)] == 0} {
			if {[_db_makenode $handle $db $map \
				nm bm fn nc $nodeid]} {
			    set exists($nodeid) 2	;# flag created node
			    unset -nocomplain _lnodes($nodeid)
			}
			unset requires($nodeid)
		    }
		}
		# remove new made nodes from "requires"
		set all [array names exists]
		foreach nodeid [array names requires] {
		    set rest {}
		    foreach n $requires($nodeid) {
			if {$n ni $all} {
			    lappend rest $n
			}
		    }
		    set requires($nodeid) $rest
		}
		lappend counts [array size requires]
		# break if everything done or no new nodes were made
		if {([lindex $counts end] == 0) ||
		    ([lindex $counts end-1] == [lindex $counts end])} {
		    break
		}
	    }
	} on error {result opts} {
	    return -code error options $opts $result
	} finally {
	    oninitialize $handle $oninit
	    # cleanup automatically created nodes
	    foreach nodeid [array names _lnodes] {
		catch {delete $handle Node $nodeid 1}
	    }
	    unset _lnodes
	}

	# create references
	$db eval [string map $map {
	    SELECT NodeId, Source, Target, IsForward FROM @"References"
	}] ref {
	    if {![::info exists nm($ref(Source))] ||
		![::info exists nm($ref(Target))] ||
		![::info exists nm($ref(NodeId))]} {
		continue
	    }
	    set ref(Source) $nm($ref(Source))
	    set ref(Target) $nm($ref(Target))
	    set ref(NodeId) $nm($ref(NodeId))
	    if {![::info exists exists($ref(Source))] ||
		![::info exists exists($ref(Target))]} {
		continue
	    }
	    if {$exists($ref(Source)) + $exists($ref(Target)) <= 2} {
		# at least one newly created node required,
		# otherwise assumed that reference exists already
		continue
	    }
	    if {[catch {
		add $handle Reference $ref(Source) $ref(NodeId) \
		    $ref(Target) $ref(IsForward)
	    } err opts]} {
		if {![string match "*already defined" $err]} {
		    return -code error -options $opts $err
		}
	    }
	}

	# pass 1: write values to Variable and VariableType
	array set wretry {}
	$db eval [string map $map {
	    SELECT ROWID AS ROWID, DataType, Value, ValueRank,
		ArrayDimensions FROM @Nodes WHERE Value IS NOT NULL
	}] wr {
	    set wr(NodeId) $nm($wr(ROWID))
	    if {![::info exists exists($wr(NodeId))] ||
		($exists($wr(NodeId)) < 2)} {
		continue
	    }
	    if {$wr(ValueRank) >= [const VALUERANK_ONE_OR_MORE_DIMENSIONS]} {
		if {[::info exists wr(ArrayDimensions)] &&
		    ($wr(ArrayDimensions) ne {})} {
		    set vr ([join $wr(ArrayDimensions) ,])
		} else {
		    set vr *
		}
	    } elseif {$wr(ValueRank) == [const VALUERANK_SCALAR]} {
		set vr !
	    } else {
		set vr {}
	    }
	    set wr(DataType) $nm($wr(DataType))
	    set jval $wr(Value)
	    if {$wr(DataType) eq [types nodeid Variant]} {
		# a Variant, guess real type
		if {[string is double -strict $jval]} {
		    set wr(DataType) Double
		} elseif {[string is boolean -strict $jval]} {
		    set wr(DataType) Boolean
		} elseif {[string is entier -strict $jval]} {
		    set wr(DataType) Int64
		} elseif {$jval eq "null"} {
		    set wr(Value) {}
		} elseif {[string match "\"*\"" $jval]} {
		    set wr(DataType) String
		    set wr(Value) [string trim $jval "\""]
		} elseif {![string match "{*}" $jval]} {
		    set wr(DataType) String
		}
	    }
	    if {($vr eq "!") && ($wr(DataType) eq [types nodeid ByteString])} {
		# try base64 decode
		if {[catch {binary decode base64 $jval} bval]} {
		    unset bval
		} elseif {$bval eq ""} {
		    unset bval
		}
	    }
	    if {![string match "{*}" $jval]} {
		set jval " $jval "
	    }
	    set wrok 0
	    if {[::info exists bval] && ![catch {
		write $handle $wr(NodeId) Value $wr(DataType) $bval
	    }]} {
		incr wrok
	    }
	    unset -nocomplain bval
	    if {!$wrok && ![catch {
		writejson $handle $wr(NodeId) Value ${vr}$wr(DataType) $jval
		# TODO: fixup for ExtensionObjects required?
		set dv [read $handle $wr(NodeId) Value {}]
		set dims {}
		if {[dict exists $dv arrayDimensions]} {
		    set dims [dict get $dv arrayDimensions]
		}
		if {[llength $dims]} {
		    set vr ([join $dims ,])
		    if {$vr eq "(0)"} {
			set vr ([llength [dict get $dv value]])
		    }
		}
		write $handle $wr(NodeId) Value ${vr}$wr(DataType) \
		    [dict get $dv value]
	    }]} {
		incr wrok
	    }
	    if {!$wrok && ![catch {
		if {$vr eq "(0)"} {
		    set vr ([llength $wr(Value)])
		}
		write $handle $wr(NodeId) Value ${vr}$wr(DataType) $wr(Value)
	    }]} {
		incr wrok
	    }
	    if {!$wrok} {
		# only retry later when not an abstract type
		if {![read $handle $wr(DataType) IsAbstract]} {
		    set wretry($wr(NodeId)) \
			[list $wr(DataType) $vr $wr(Value) $jval]
		}
	    }
	}

	# (re)generate type info to allow failed writes to complete
	variable _lnodes
	array set _lnodes {}
	set oninit [oninitialize $handle]
	oninitialize $handle [::namespace current]::_db_oninit
	try {
	    _db_gentypes $handle $db $map nm
	} on error {result opts} {
	    return -code error options $opts $result
	} finally {
	    oninitialize $handle $oninit
	    # see cleanup at end
	}

	# pass 2: again write values to Variable and VariableType,
	# failed write operations are left in the "wretry" array
	foreach nodeid [array names wretry] {
	    set wr(NodeId) $nodeid
	    lassign $wretry($nodeid) wr(DataType) vr wr(Value) jval
	    # TODO: should the base64 decoded value also be retried?
	    set wrok 0
	    if {!$wrok && ![catch {
		writejson $handle $wr(NodeId) Value ${vr}$wr(DataType) $jval
		# TODO: fixup for ExtensionObjects required?
		set dv [read $handle $wr(NodeId) Value {}]
		set dims {}
		if {[dict exists $dv arrayDimensions]} {
		    set dims [dict get $dv arrayDimensions]
		}
		if {[llength $dims]} {
		    set vr ([join $dims ,])
		    if {$vr eq "(0)"} {
			set vr ([llength [dict get $dv value]])
		    }
		}
		write $handle $wr(NodeId) Value ${vr}$wr(DataType) \
		    [dict get $dv value]
	    } err]} {
		incr wrok
	    }
	    if {!$wrok && ![catch {
		if {$vr eq "(0)"} {
		    set vr ([llength $wr(Value)])
		}
		write $handle $wr(NodeId) Value ${vr}$wr(DataType) $wr(Value)
	    } err]} {
		incr wrok
	    }
	    if {$wrok} {
		unset wretry($nodeid)
	    } else {
		lappend wretry($nodeid) $err
	    }
	}

	# pass 3: cleanup HasTypeDefinition refs to BaseDataVariableType
	# from the nodes in array fn
	lassign [translate $handle [root] / Types / VariableTypes / \
		    BaseVariableType / BaseDataVariableType] bt
	foreach nodeid [array names fn] {
	    catch {delete $handle Reference $nodeid HasTypeDefinition $bt 1 1}
	}
	# cleanup automatically created nodes
	foreach nodeid [array names _lnodes] {
	    catch {delete $handle Node $nodeid 1}
	}
	unset _lnodes

	# try to add method output types to newly created methods
	foreach {nodeid _ _} [methods $handle] {
	    if {$nodeid in $old_meths} {
		continue
	    }
	    catch {
		set oa [translate $handle $nodeid \
			    HasProperty OutputArguments]
		set oid [lindex $oa 0]
		methods $handle $nodeid [read $handle $oid]
	    }
	}

	# done
	return {}
    }

    proc import {handle args} {
	set schema {}
	if {[llength $args] % 2} {
	    return -code error -errorcode [list opcua::sqlmodel import -1] \
		"odd number of arguments"
	}
	foreach {option value} $args {
	    if {[catch {::tcl::prefix match -message "option" {
		-file -db -schema -data -channel
	    } $option} opt]} {
		return -code error \
		    -errorcode [list opcua::sqlmodel import -1] $opt
	    }
	    switch -exact -- $opt {
		-file {
		    set mode file
		    set name $value
		}
		-db {
		    set mode db
		    set name $value
		}
		-schema {
		    set schema $value
		}
		-data {
		    set mode data
		    set data $value
		}
		-channel {
		    set mode chan
		    set name $value
		}
	    }
	}
	if {![::info exists mode]} {
	    return -code error \
		-errorcode [list opcua::sqlmodel import -1] "mode missing"
	}
	if {$mode eq "db"} {
	    tailcall _db_import $handle $name $schema
	}
	if {$mode eq "data"} {
	    set db [::namespace current]::_dbtemp
	    sqlite3 $db :memory:
	    set deschema $schema
	    if {$deschema eq {}} {
		set deschema main
	    }
	    try {
		$db deserialize $deschema $data
		_db_import $handle $db $schema
	    } on error {result opts} {
		return -code error -options $opts $result
	    } finally {
		$db close
	    }
	    return {}
	}
	if {$mode eq "chan"} {
	    set db [::namespace current]::_dbtemp
	    sqlite3 $db :memory:
	    set deschema $schema
	    if {$deschema eq {}} {
		set deschema main
	    }
	    try {
		$db deserialize $deschema [::read $name]
		_db_import $handle $db $schema
	    } on error {result opts} {
		return -code error -options $opts $result
	    } finally {
		$db close
	    }
	    return {}
	}
	# mode is "file"
	set db [::namespace current]::_dbtemp
	sqlite3 $db file:[file normalize $name] -uri 1 -readonly 1
	try {
	    _db_import $handle $db {}
	} on error {result opts} {
	    return -code error -options $opts $result
	} finally {
	    $db close
	}
	return {}
    }

    ######################################################################
    # Load full namespace zero from local database "ns0.db.gz" which
    # must be located in script's directory.

    variable ns0full
    set ns0full [file normalize \
			[file join [file dirname [::info script]] ns0.db.gz]]
    if {![file readable $ns0full]} {
	unset ns0full
    }

    proc ns0full {handle} {
	variable ns0full
	set ret 0
	if {[::info exists ns0full] && ![catch {::open $ns0full rb} f]} {
	    try {
		zlib push gunzip $f
		import $handle -channel $f
		set ret 1
	    } finally {
		close $f
	    }
	}
	return $ret
    }

    ######################################################################
    # Make database of companion specs by loading ZIP from OPC Foundation's
    # github repository. This produces a single SQLite3 database for all
    # available companion specs of about 3 MByte (as of July '23 for the
    # github tag "latest").

    proc _curl_get {url args} {
	set pairs {}
	foreach {name value} $args {
	    lappend pairs "[curl::escape $name]=[curl::escape $value]"
	}
	if {[llength $pairs]} {
	    append url ? [join $pairs &]
	}
	set handle [curl::init]
	$handle configure -url $url -bodyvar data -failonerror 1 \
	    -sslverifypeer 0
	catch {$handle perform} errnum
	$handle cleanup
	if {$errnum != 0} {
	    return -code error [list $errnum [curl::easystrerror $errnum]]
	}
	return $data
    }

    proc _process_xmlfile {db name tail} {
	try {
	    # open/read binary for later gzip
	    set f [open $name rb]
	    set data [::read $f]
	    if {[string range $data 0 2] eq "\xef\xbb\xbf"} {
		# strip the utf-8 encoded BOM, since tdom doesn't like it
		set data [string range $data 3 end]
	    }
	    set udata [encoding convertfrom utf-8 $data]
	    set doc [dom parse -- $udata]
	    set root [$doc documentElement]
	    set ns [$root namespaceURI]
	    foreach node [$root selectNodes -namespaces [list n $ns] \
			      /n:UANodeSet/n:Models/n:Model] {
		if {[$node hasAttribute ModelUri]} {
		    set prov [$node getAttribute ModelUri]
		    set vers {}
		    catch {set vers [$node getAttribute Version]}
		    set pdat {}
		    catch {set pdat [$node getAttribute PublicationDate]}
		    set data [zlib gzip $data -level 9]
		    set name [string range $prov 5 end]
		    set name [string trim $name /]
		    set name [lrange [split $name /] 1 end]
		    set name0 [lindex $name 0]
		    switch -nocase -- $name0 {
			UA - OPCUA {
			    set name [lrange $name 1 end]
			}
		    }
		    if {[string match http://sercos.org/* $prov]} {
			set name SERCOS
		    }
		    set name0 [lindex $name 0]
		    while {[string is integer -strict $name0]} {
			set name [lrange $name 1 end]
			set name0 [lindex $name 0]
		    }
		    set name [join $name /]
		    $db eval {
			INSERT OR REPLACE INTO Models
			    (Model, Name, Version, PublicationDate, XML)
			VALUES($prov, $name, $vers, $pdat, @data)
		    }
		    foreach child [$node childNodes] {
			if {[$child nodeName] ne "RequiredModel"} {
			    continue
			}
			if {[$child hasAttribute ModelUri]} {
			    set req [$child getAttribute ModelUri]
			    set vers {}
			    catch {
				set vers [$child getAttribute Version]
			    }
			    set pdat {}
			    catch {
				set pdat [$child getAttribute PublicationDate]
			    }
			    $db eval {
				INSERT OR REPLACE INTO Requires
				    (Model, RequiredModel,
				     RequiredVersion, RequiredPublicationDate)
				VALUES($prov, $req, $vers, $pdat)
			    }
			}
		    }
		}
	    }
	} on error {result opts} {
	    return -code error -options $opts "processing \"$tail\": $result"
	} finally {
	    if {[::info exists doc]} {
		$doc delete
	    }
	    if {[::info exists f]} {
		::close $f
	    }
	}
    }

    proc _process_xsdfile {db name tail} {
	try {
	    # open/read binary for later gzip
	    set f [open $name rb]
	    set data [::read $f]
	    if {[string range $data 0 2] eq "\xef\xbb\xbf"} {
		# strip the utf-8 encoded BOM, since tdom doesn't like it
		set data [string range $data 3 end]
	    }
	    set udata [encoding convertfrom utf-8 $data]
	    set doc [dom parse -- $udata]
	    set root [$doc documentElement]
	    set ns [$root namespaceURI]
	    foreach node [$root selectNodes -namespaces [list n $ns] \
			  /n:schema] {
		if {[$node hasAttribute targetNamespace]} {
		    set tns [$node getAttribute targetNamespace]
		    set data [zlib gzip $data -level 9]
		    $db eval {
			INSERT OR REPLACE INTO XmlSchema(Name, XML)
			VALUES($tns, @data)
		    }
		}
	    }
	} on error {result opts} {
	    return -code error -options $opts "processing \"$tail\": $result"
	} finally {
	    if {[::info exists doc]} {
		$doc delete
	    }
	    if {[::info exists f]} {
		::close $f
	    }
	}
    }

    proc _process_unece {db name} {
	try {
	    package require csv
	    set f [open $name r]
	    gets $f
	    while {![eof $f]} {
		array unset a
		lassign [csv::split [gets $f] ","] \
		    a(UNECECode) a(UnitId) a(DisplayName) a(Description)
		$db eval {
		    INSERT OR REPLACE INTO UNECE
			(UNECECode, UnitId, DisplayName, Description)
		    VALUES(
			$a(UNECECode), $a(UnitId),
			$a(DisplayName), $a(Description)
		    );
		}
	    }
	} finally {
	    close $f
	}
    }

    proc _find0 {dir} {
	set result {}
	if {[catch {glob -directory $dir -tails -nocomplain * .*} list]} {
	    return $result
	}
	foreach file $list {
	    if {($file eq ".") || ($file eq "..")} {
		continue
	    }
	    set file [file join $dir $file]
	    lappend result $file {*}[_find $file]
	}
	return $result
    }

    proc _find {dir} {
	tailcall lsort [_find0 $dir]
    }

    proc mkspecsdb {dbname {tag latest}} {
	package require TclCurl
	package require tdom
	if {[package provide zipfs] ne {}} {
	    package require zipfs
	    set zipfs 1
	} else {
	    package require vfs::zip
	    set zipfs 0
	    set zf [file tempfile zfile UATMP]
	}
	set zd [file normalize /UA-Nodeset[clock microseconds]]
	set db [::namespace current]::_dbtemp
	set url \
	    https://codeload.github.com/OPCFoundation/UA-Nodeset/zip/refs/heads/
	append url $tag
	try {
	    set zip [_curl_get $url]
	    if {$zipfs} {
		set zmnt [zipfs::mount -data $zip $zd]
	    } else {
		fconfigure $zf -translation binary
		puts -nonewline $zf $zip
		close $zf
		set zf [vfs::zip::Mount $zfile $zd]
	    }
	    sqlite3 $db $dbname
	    $db eval {
		BEGIN TRANSACTION;
		DROP TABLE IF EXISTS Models;
		CREATE TABLE Models(
		    Model VARCHAR NOT NULL,
		    Name VARCHAR UNIQUE NOT NULL,
		    Version VARCHAR DEFAULT '',
		    PublicationDate VARCHAR DEFAULT '',
		    XML BLOB NOT NULL,
		    PRIMARY KEY(Model)
		);
		DROP TABLE IF EXISTS Requires;
		CREATE TABLE Requires(
		    Model VARCHAR NOT NULL,
		    RequiredModel VARCHAR NOT NULL,
		    RequiredVersion VARCHAR DEFAULT '',
		    RequiredPublicationDate VARCHAR DEFAULT '',
		    PRIMARY KEY(Model, RequiredModel)
		);
		DROP TABLE IF EXISTS XmlSchema;
		CREATE TABLE XmlSchema(
		    Name VARCHAR UNIQUE NOT NULL,
		    XML BLOB NOT NULL
		);
		DROP TABLE IF EXISTS UNECE;
		CREATE TABLE UNECE(
		    UNECECode VARCHAR NOT NULL,
		    UnitId VARCHAR NOT NULL,
		    DisplayName VARCHAR NOT NULL,
		    Description VARCHAR NOT NULL,
		    PRIMARY KEY(UNECEcode)
		);
	    }
	    foreach item [_find $zd] {
		set tail [file tail $item]
		if {[file isfile $item]} {
		    switch -glob -- $tail {
			*.xml {
			    _process_xmlfile $db $item $tail
			}
			*.xsd {
			    _process_xsdfile $db $item $tail
			}
			UNECE_to_OPCUA.csv {
			    _process_unece $db $item
			}
		    }
		}
	    }
	    # remove NS0 information, it is assumed to be implicit
	    $db eval {
		DELETE FROM Requires
		WHERE Model = 'http://opcfoundation.org/UA/'
		OR RequiredModel = 'http://opcfoundation.org/UA/';
		DELETE FROM Models
		WHERE Model = 'http://opcfoundation.org/UA/';
		COMMIT TRANSACTION;
		VACUUM;
	    }
	} on error {result opts} {
	    return -code error -options $opts $result
	} finally {
	    if {$zipfs} {
		catch {zipfs::unmount $zmnt}
	    } else {
		catch {vfs::zip::Unmount $zf $zd}
		file delete -force $zfile
	    }
	    catch {$db close}
	}
    }

    ######################################################################
    # Load a companion spec, see comment on top.

    proc _resolve {db name} {
	upvar have have
	upvar want want
	$db eval {
	    SELECT RequiredModel FROM Requires WHERE Model = $name
	} data {
	    if {$data(RequiredModel) in $have} {
		continue
	    }
	    if {$data(RequiredModel) ni $want} {
		lappend want $data(RequiredModel)
		_resolve $db $data(RequiredModel)
	    } else {
		# the "want" list is in reverse order,
		# thus put "name" at it's end now
		set want [lsearch -not -all -inline $want $data(RequiredModel)]
		lappend want $data(RequiredModel)
	    }
	}
    }

    proc loadspec {handle name {dbname {}}} {
	if {[info $handle] ne "server"} {
	    return -code error -errorcode {opcua Internal 0 Good} \
		"not a server handle"
	}
	if {($dbname eq {}) && [::info exists ::env(HOME)]} {
	    set dbname [file normalize [file join $::env(HOME) uaspecs.db]]
	}
	set db [::namespace current]::_dbtemp
	set spec "***SETUP***"
	try {
	    sqlite3 $db file:$dbname -uri 1 -readonly 1
	    $db eval {
		SELECT Model FROM Models WHERE Name = $name OR Model = $name
	    } data {}
	    if {![::info exists data(Model)]} {
		return -code error "model \"$name\" not found"
	    }
	    set have [list]
	    foreach {i n} [namespace $handle] {
		lappend have $n
	    }
	    if {$data(Model) in $have} {
		return
	    }
	    lappend want $data(Model)
	    _resolve $db $data(Model)
	    foreach spec [lreverse $want] {
		$db eval {
		    SELECT XML FROM Models WHERE Model = $spec
		} data {
		    set udata \
			[encoding convertfrom utf-8 [zlib gunzip $data(XML)]]
		    loader $handle $udata
		}
	    }
	} on error {result opts} {
	    return -code error -options $opts "processing \"$spec\": $result"
	} finally {
	    catch {$db close}
	}
    }

    ######################################################################
    # List companion specs, see comment on top.

    proc listspecs {{dbname {}}} {
	if {($dbname eq {}) && [::info exists ::env(HOME)]} {
	    set dbname [file normalize [file join $::env(HOME) uaspecs.db]]
	}
	set db [::namespace current]::_dbtemp
	set ret [list]
	try {
	    sqlite3 $db file:$dbname -uri 1 -readonly 1
	    $db eval {
		SELECT Model, Name FROM Models
	    } data {
		lappend ret $data(Name) $data(Model)
	    }
	} on error {result opts} {
	    return -code error -options $opts $result
	} finally {
	    catch {$db close}
	}
	return $ret
    }

    ######################################################################
    # Get UNECE table, see comment on top.

    proc getunece {{dbname {}}} {
	if {($dbname eq {}) && [::info exists ::env(HOME)]} {
	    set dbname [file normalize [file join $::env(HOME) uaspecs.db]]
	}
	set db [::namespace current]::_dbtemp
	set ret [list]
	try {
	    sqlite3 $db file:$dbname -uri 1 -readonly 1
	    $db eval {
		SELECT UNECECode, UnitId, DisplayName, Description
		FROM UNECE
	    } data {
		lappend ret $data(UNECECode) $data(UnitId) \
		    $data(DisplayName) $data(Description)
	    }
	} on error {result opts} {
	    return -code error -options $opts $result
	} finally {
	    catch {$db close}
	}
	return $ret
    }

    ######################################################################

    ::namespace export {[a-z]*}

    ::namespace ensemble create -subcommands \
	    [list export getunece import listspecs loadspec ns0full mkspecsdb]

}
