# Module to calibrate resistive EVDEV touchscreens in SDL based Tk.
#
# Currently supports video drivers KSMDRM and RPI. Note that the
# calibration only covers simple touchscreens which report EV_ABS
# ABS_X and ABS_Y events. Multi-touch capable devices report event
# types ABS_MT_* which are assumed to be properly aligned with the
# display.
#
# Usage:
#
#	package require touchcal
#	set data [touchcal::calibrate .toplevel]
#	# A dialog of a four-point calibration
#	# sequence is presented in .toplevel and
#	# the result of that dialog is returned.
#	# On error, data is empty, on success, it
#	# contains a five element list of numbers
#	# which should be persisted and later re-
#	# loaded on next startup as calibration using
#	# the following command
#	sdltk touchcalibration {*}$data
#
# chw February 2019

namespace eval touchcal {

    if {[info command "sdltk"] eq ""} {
	error "no SDL2 aware Tk version loaded"
    }

    variable scrw
    variable scrh
    variable size 100
    variable csize 20
    variable lsize 50
    variable MSG1
    variable MSG2
    variable RESULT
    variable cdata
    variable lastcal
    variable running 0

    proc draw_pad {name x y} {
	variable size
	variable csize
	variable lsize
	variable scrw
	variable scrh

	set cx [expr {$size / 2 - $csize / 2}]
	set cy $cx
	set ccx [expr {$cx + $csize}]
	set ccy $ccx
	set vx [expr {$size / 2}]
	set voy [expr {($size -$lsize) / 2}]
	set vcy [expr {$voy + $lsize}]
	set hox [expr {($size - $lsize) / 2}]
	set hcx [expr {$hox + $lsize}]
	set hy [expr {$size / 2}]
	frame $name
	set anchor ""
	if {$y > 0} {
	    set y 0
	    append anchor n
	} elseif {$y < 0} {
	    set y [expr {$scrh - 0.5}]
	    append anchor s
	} else {
	    set y [expr {$scrh / 2}]
	}
	if {$x > 0} {
	    set x 0
	    append anchor w
	} elseif {$x < 0} {
	    set x [expr {$scrw - 0.5}]
	    append anchor e
	} else {
	    set x [expr {$scrw / 2}]
	    set anchor center
	}
	place $name -x $x -y $y -anchor $anchor -width $size -height $size
	canvas $name.m -height $size -width $size -background "#505075"
	$name.m create oval $cx $cy $ccx $ccy -outline white
	$name.m create line $vx $voy $vx $vcy -fill white
	$name.m create line $hox $hy $hcx $hy -fill white
	pack $name.m
    }

    proc show_calib {varname} {
	upvar #0 $varname msg

	set data {? ? ? ? ?}
	catch {set data [sdltk touchcalibration]}
	append msg "MinX:   [lindex $data 0]\n"
	append msg "MaxX:   [lindex $data 1]\n"
	append msg "MinY:   [lindex $data 2]\n"
	append msg "MaxY:   [lindex $data 3]\n"
	append msg "SwapXY: [lindex $data 4]"
    }

    proc swap {a b} {
	upvar $a va
	upvar $b vb
	set tmp $va
	set va $vb
	set vb $tmp
    }

    proc calib_seq {top which xdev ydev} {
	variable size
	variable scrw
	variable scrh
	variable MSG1
	variable MSG2
	variable RESULT
	variable cdata
	variable lastcal

	set cal {0 0 0 0 0}
	catch {set cal [sdltk touchcalibration]}
	lassign $cal xmin xmax ymin ymax
	set xrange [expr {$xmax - $xmin}]
	set yrange [expr {$ymax - $ymin}]
	set xdev [expr {$xmin + $xdev * $xrange / 10000.0}]
	set ydev [expr {$ymin + $ydev * $yrange / 10000.0}]

	set cdata(xdev,$which) $xdev
	set cdata(ydev,$which) $ydev
	if {$which == 0} {
	    $top.topleft.m configure -background "#505075"
	    $top.topright.m configure -background "#df94df"
	    bind $top <<FingerUp>> [subst {
		if {%s == 1} {
		    [namespace current]::calib_seq $top 1 %x %y
		}
	    }]
	} elseif {$which == 1} {
	    $top.topright.m configure -background "#505075"
	    $top.bottomright.m configure -background "#df94df"
	    bind $top <<FingerUp>> [subst {
		if {%s == 1} {
		    [namespace current]::calib_seq $top 2 %x %y
		}
	    }]
	} elseif {$which == 2} {
	    $top.bottomright.m configure -background "#505075"
	    $top.bottomleft.m configure -background "#df94df"
	    bind $top <<FingerUp>> [subst {
		if {%s == 1} {
		    [namespace current]::calib_seq $top 3 %x %y
		}
	    }]
	} elseif {$which == 3} {
	    $top.bottomleft.m configure -background "#505075"
	    bind $top <<FingerUp>> {}

	    set thresh [expr {abs(round($xrange * 0.1))}]
	    set swapxy 0
	    if {abs($cdata(xdev,0) - $cdata(xdev,3)) > $thresh ||
		abs($cdata(xdev,1) - $cdata(xdev,2)) > $thresh} {
		set swapxy 1
		swap cdata(xdev,0) cdata(ydev,0)
		swap cdata(xdev,1) cdata(ydev,1)
		swap cdata(xdev,2) cdata(ydev,2)
		swap cdata(xdev,3) cdata(ydev,3)
	    }

	    set boff [expr {double(($size / 2) + 1)}]
	    set wdev [expr {$cdata(xdev,1) - $cdata(xdev,0)}]
	    set wdev2 [expr {$cdata(xdev,2) - $cdata(xdev,3)}]
	    set wdev [expr {($wdev + $wdev2) / 2.0}]
	    set hdev [expr {$cdata(ydev,3) - $cdata(ydev,0)}]
	    set hdev2 [expr {$cdata(ydev,2) - $cdata(ydev,1)}]
	    set hdev [expr {($hdev + $hdev2) / 2.0}]
	    set wx [expr {1.0 * $scrw - (2 * $boff)}]
	    set hx [expr {1.0 * $scrh - (2 * $boff)}]

	    set xdevmin [expr {$cdata(xdev,0) - ($boff * $wdev / $wx)}]
	    set xdevmin2 [expr {$cdata(xdev,3) - ($boff * $wdev / $wx)}]
	    set xdevmin [expr {($xdevmin + $xdevmin2) / 2.0}]
	    set xdevmax [expr {$cdata(xdev,1) + ($boff * $wdev / $wx)}]
	    set xdevmax2 [expr {$cdata(xdev,2) + ($boff * $wdev / $wx)}]
	    set xdevmax [expr {($xdevmax + $xdevmax2) / 2.0}]

	    set ydevmin [expr {$cdata(ydev,0) - ($boff * $hdev / $hx)}]
	    set ydevmin2 [expr {$cdata(ydev,1) - ($boff * $hdev / $hx)}]
	    set ydevmin [expr {($ydevmin + $ydevmin2) / 2.0}]
	    set ydevmax [expr {$cdata(ydev,3) + ($boff * $hdev / $hx)}]
	    set ydevmax2 [expr {$cdata(ydev,2) + ($boff * $hdev / $hx)}]
	    set ydevmax [expr {($ydevmax + $ydevmax2) / 2.0}]

	    set data [list [expr {round($xdevmin)}] \
		[expr {round($xdevmax)}] \
		[expr {round($ydevmin)}] \
		[expr {round($ydevmax)}]]
	    if {$swapxy} {
		lappend data 1
	    }

	    $top.cal.doit configure -state normal
	    destroy $top.topleft $top.bottomleft \
		$top.topright $top.bottomright
	    $top configure -cursor {}

	    catch {sdltk touchcalibration {*}$lastcal}
	    set MSG1 "Old Calibration\n\n"
	    show_calib [namespace current]::MSG1
	    catch {sdltk touchcalibration {*}$data}
	    if {![catch {set lastcal [sdltk touchcalibration]}]} {
		set RESULT $lastcal
	    }
	    set MSG2 "New Calibration\n\n"
	    show_calib [namespace current]::MSG2
	}
    }

    proc start_calib {top} {
	variable MSG1
	variable MSG2
	variable lastcal
	$top.cal.doit configure -state disabled
	draw_pad $top.topleft 1 1
	draw_pad $top.bottomleft 1 -1
	draw_pad $top.topright -1 1
	draw_pad $top.bottomright -1 -1
	update
	# Start calib sequence
	$top.topleft.m configure -background "#df94df"
	bind $top <<FingerUp>> [subst {
	    if {%s == 1} {
		[namespace current]::calib_seq $top 0 %x %y
	    }
	}]
	set MSG1 "Old Calibration\n\n"
	set MSG2 ""
	show_calib [namespace current]::MSG1
	set lastcal {0 0 0 0 0}
	catch {set lastcal [sdltk touchcalibration]}
	# reset to device defaults
	catch {sdltk touchcalibration 0 0 0 0}
    }

    proc do_quit {top} {
	variable lastcal
	destroy $top
	catch {sdltk touchcalibration {*}$lastcal}
    }

    proc calibrate {top} {
	variable running
	variable size
	variable csize
	variable lsize
	variable scrw
	variable scrh
	variable MSG1
	variable MSG2
	variable RESULT

	if {$running} {
	    return -code error "calibration already in progress"
	}
	toplevel $top
	wm attributes $top -fullscreen 1
	set scrw [winfo screenwidth $top]
	set scrh [winfo screenheight $top]
	$top configure -cursor {} -background black
	set f_small [expr {round($scrw / -64.0)}]
	if {$f_small < -18} {
	    set f_small -18
	}
	set f_small [list {DejaVu Sans Mono} $f_small]
	set f_large [expr {round($scrw / -42.0)}]
	if {$f_large < -24} {
	    set f_large -24
	}
	set f_large [list {DejaVu Sans} $f_large]
	if {$scrw < 640 || $scrh < 480} {
	    set size 50
	    set csize 10
	    set lsize 25
	    set pady 10
	} else {
	    set size 100
	    set csize 20
	    set lsize 50
	    set pady 40
	}

	frame $top.msg -cursor {} -background black
	label $top.msg.m1 -textvariable [namespace current]::MSG1 \
	    -background black -foreground yellow \
	    -height 8 -width 40 -font $f_small \
	    -padx 20 -pady [expr {$pady / 2}] -justify left
	label $top.msg.m2 -textvariable [namespace current]::MSG2 \
	    -background black -foreground yellow \
	    -height 8 -width 40 -font $f_small \
	    -padx 20 -pady [expr {$pady / 2}] -justify left
	pack $top.msg -side bottom -fill x
	pack $top.msg.m1 $top.msg.m2 -side left -padx 20 -pady $pady -expand 1

	frame $top.cal -cursor {} -background black
	button $top.cal.doit -text "Calibrate Touchscreen" -underline 0 \
	    -command [list [namespace current]::start_calib $top] -width 22 \
	    -relief solid -background "#505075" -foreground white \
	    -activebackground "#df94df" -activeforeground white \
	    -highlightthickness 1 -borderwidth 0 -highlightbackground white \
	    -takefocus 0 -font $f_large -padx 20 -pady [expr {$pady / 2}]
	button $top.cal.quit -text "Quit" -underline 0 \
	    -command [list [namespace current]::do_quit $top] -width 22 \
	    -relief solid -background "#505075" -foreground white \
	    -activebackground "#df94df" -activeforeground white \
	    -highlightthickness 1 -borderwidth 0 -highlightbackground white \
	    -takefocus 0 -font $f_large -padx 20 -pady [expr {$pady / 2}]
	pack $top.cal -side bottom -fill both
	pack $top.cal.doit $top.cal.quit -side left -padx 20 -pady 20 -expand 1

	set MSG1 "Old Calibration\n\n"
	set MSG2 ""
	show_calib [namespace current]::MSG1

	bind $top <Escape> [list $top.cal.quit invoke]
	bind $top <q> [list $top.cal.quit invoke]
	bind $top <Q> [list $top.cal.quit invoke]
	bind $top <x> [list $top.cal.quit invoke]
	bind $top <X> [list $top.cal.quit invoke]
	bind $top <c> [list $top.cal.doit invoke]
	bind $top <C> [list $top.cal.doit invoke]

	set running 1
	update idletasks
	set oldfocus [focus]
	raise $top
	grab $top
	update idletasks
	focus -force $top
	set RESULT {}
	while {[winfo exists $top]} {
	    tkwait window $top
	}
	set running 0
	catch {focus $oldfocus}
	return $RESULT
    }

}

package provide touchcal 0.1
