# twDebugInspector.tcl -- a Tcl introspection tool for debugging
#
# Source http://wiki.tcl.tk/13989

package provide twDebugInspector 0.1

package require Tk 8.5
package require BWidget
package require ctext

namespace eval twDebug {
    variable state  ;# used to hold status data for the current item
    variable config ;# used to hold configuration data (like window paths)

    namespace export Inspector
}

# create a ctext widget and a browser for procedures and variables

proc twDebug::Inspector {toplevel} {
    variable config
    set w $toplevel
    if {[winfo exists $w]} {
	wm deiconify $w
	raise $w
	focus $w
	update idletasks
	return
    }

    toplevel $w
    wm title $w {Tcl in-process inspector}
    wm minsize $w 450 250
    wm protocol $w WM_DELETE_WINDOW [list twDebug::InspectorClose $w]

    # panedwindow holding tree and propc window:
    panedwindow $w.pane -orient horizontal -showhandle 0 -sashwidth 10 \
	-relief flat -borderwidth 1

    # the ctext widget holding a selected proc:
    twDebug::ScrolledWidget ctext $w.info 1 1 -width 60 -height 24 \
	-background white -wrap none -font TkFixedFont -undo 1
    set config(win,ctext) $w.info

    # the tree holding all inspectable information:
    twDebug::ScrolledWidget Tree $w.tree 0 1 -width 20 -height 20 \
	-background white -borderwidth 2 -relief flat \
	-linesfill gray -selectfill 1 -deltay 15 -deltax 10 \
	-selectcommand [list twDebug::InspectorShowItem $w.info]

    $w.pane add $w.tree -minsize 150
    $w.pane add $w.info -minsize 100

    # define highlight patterns (for the actual ctext widget...):
    twDebug::setHighlightTcl $w.info.list
    $w.info tag configure err -foreground red
    # change highlight patterns with other options than color:
    $w.info tag configure comments -background \#e0f0e0

    set btns [list \
	"Apply changes" twDebug::InspectorApply \
	"Save to file"  twDebug::InspectorSave \
	"Close window"  [list twDebug::InspectorClose $w] \
    ]
    if {[info command ::console] eq "::console"} {
	lappend btns "Console ..." twDebug::InspectorConsole
    }
    twDebug::ButtonRow $w.btn {*}$btns

    pack $w.btn  -fill x -side bottom
    pack $w.pane -fill both -expand yes -padx 5 -pady 5 -side top

    # initialize the tree view with procs and vars ...
    tk busy hold $w
    update idletasks
    tailcall twDebug::InspectorInit $w $w.tree.list
}

proc twDebug::InspectorProc {toplevel procedure} {
    twDebug::Inspector $toplevel
    set w $toplevel
    set ns [namespace qualifiers $procedure]
    if {$ns eq {}} {
	set ns ::
    }
    set nodeName "N[string map {: _} $ns]"
    set nodeName P${nodeName}[namespace tail $procedure]
    tailcall twDebug::InspectorShowItem $w.info $w.tree $nodeName
}

proc twDebug::InspectorInit {toplevel win} {
    $win delete [$win nodes root]
    # procedures sorted by namespace:
    twDebug::InspectorInitNS $win ::
    # array variables:
    twDebug::InspectorInitArray $win ::
    # scalar variables:
    twDebug::InspectorInitScalar $win ::
    # widgets:
    #$win insert end root widgets -text Widgets
    # bindings:
    #$win insert end root bindings -text Bindings
    tk busy forget $toplevel
}

proc twDebug::InspectorInitNS {win ns} {
    set parent "N[string map {: _} [namespace parent $ns]]"
    set nodeText $ns
    set nodeName "N[string map {: _} $ns]"
    if {$parent eq "N"} {
	set parent "root"
	set nodeText "Procedures"
    }
    # insert namespace:
    $win insert end $parent $nodeName -fill blue -text $nodeText
    # insert procedures:
    foreach procedure [lsort -dictionary [namespace eval $ns {::info procs}]] {
	if {($ns ne {}) && ($ns ne {::}) &&
	    ![catch {namespace origin ${ns}::$procedure} origin]} {
	    if {![string equal $origin ${ns}::$procedure]} {
		continue
	    }
	}
	$win insert end $nodeName "P$nodeName$procedure" \
	    -text $procedure -data P
    }
    # insert children:
    foreach myNS [lsort -dictionary [namespace children $ns]] {
	twDebug::InspectorInitNS $win $myNS
    }
}

proc twDebug::InspectorInitArray {win ns} {
    set parent "NN[string map {: _} [namespace parent $ns]]"
    set nodeText $ns
    set nodeName "NN[string map {: _} $ns]"
    if {$parent eq "NN"} {
	set parent "root"
	set nodeText "Array variables"
    }
    # insert namespace:
    $win insert end $parent $nodeName -fill green -text $nodeText
    # insert variables:
    foreach variable [lsort -dictionary [info vars ${ns}::*]] {
	if {[array exists $variable]} {
	    set newNode "A$nodeName[string map {: _} $variable]"
	    $win insert end $nodeName $newNode -text $variable -data A
	    set i 0
	    #foreach el [array names $variable] {
	    #    $win insert end $newNode "$newNode[incr i]" -text $el
	    #}
	}
    }
    # insert children:
    foreach myNS [lsort -dictionary [namespace children $ns]] {
	twDebug::InspectorInitArray $win $myNS
    }
}

proc twDebug::InspectorInitScalar {win ns} {
    set parent "NNN[string map {: _} [namespace parent $ns]]"
    set nodeText $ns
    set nodeName "NNN[string map {: _} $ns]"
    if {$parent eq "NNN"} {
	set parent "root"
	set nodeText "Scalar variables"
    }
    # insert namespace:
    $win insert end $parent $nodeName -fill brown -text $nodeText
    # insert variables:
    foreach variable [lsort -dictionary [info vars ${ns}::*]] {
	if {![array exists $variable]} {
	    set newNode "S$nodeName[string map {: _} $variable]"
	    $win insert end $nodeName $newNode -text $variable -data S
	}
    }
    # insert children:
    foreach myNS [lsort -dictionary [namespace children $ns]] {
	twDebug::InspectorInitScalar $win $myNS
    }
}

proc twDebug::InspectorShowItem {info tree node} {
    variable state
    set data [$tree itemcget $node -data]
    if {$data eq ""} {
	return
    }
    set state(itemType) $data
    set name [$tree itemcget $node -text]
    set NS [$tree itemcget [$tree parent $node] -text]
    switch $data {
	P {
	    if {$NS ne "Procedures"} {
		set name "${NS}::$name"
	    } else {
		set name "::$name"
	    }
	    $info delete 1.0 end
	    $info edit reset
	    set argList [list]
	    foreach arg [info args $name] {
		if {[info default $name $arg stdVar] == 1} {
		    lappend argList [list $arg $stdVar]
		} else {
		    lappend argList $arg
		}
	    }
	    $info fastinsert end "proc [list $name] [list $argList] {"
	    $info fastinsert end [info body $name]
	    $info fastinsert end "}"
	    $info edit reset
	    $info highlight 1.0 end
	}
	A {
	    $info delete 1.0 end
	    $info edit reset
	    foreach el [array names $name] {
		if {![catch {set ${name}($el)} value]} {
		    $info fastinsert end \
			"set [list $name]($el) [list $value]\n"
		}
	    }
	    $info edit reset
	}
	S {
	    $info delete 1.0 end
	    $info edit reset
	    catch {set ${name}} value
	    $info fastinsert end "set [list $name] [list $value]\n"
	    $info edit reset
	}
    }
}

proc twDebug::InspectorApply {args} {
    variable state
    variable config
    set w $config(win,ctext)
    set wp [winfo parent $w]
    switch $state(itemType) {
	P {
	    set data [$w get 1.0 "end - 1 char"]
	    if {[llength $data] != 4} {
		tk_messageBox -parent $wp -type ok -icon error \
		    -message "The procedure seems to have a wrong format. Please verify that is has: 'proc name args body'."
		return
	    }
	    if {[catch {uplevel #0 $data} error]} {
		tk_messageBox -parent $wp -type ok -icon error \
		    -message "Saving failed:\n\n $error"
	    }
	}
	A - S {
	    set data [$w get 1.0 "end - 1 char"]
	    if {[catch {uplevel #0 $data} error]} {
		tk_messageBox -parent $wp -type ok -icon error \
		    -message "Set failed:\n\n $error"
	    }
	}
    }
}

proc twDebug::InspectorClose {toplevel args} {
    destroy $toplevel
}

proc twDebug::InspectorConsole {args} {
    console show
}

proc twDebug::InspectorSave {args} {
    variable config
    set file [tk_getSaveFile -parent [winfo parent $config(win,ctext)]]
    if {$file eq ""} {
	return
    }
    catch {
	set fh [open $file w]
	puts $fh [$config(win,ctext) get 1.0 "end - 1 char"]
    }
    catch {
	close $fh
    }
}

# set hightlight patterns for the ctext widget

proc twDebug::setHighlightTcl {w} {
    set color(widgets) red
    set color(flags) orange
    set color(vars) blue
    set color(cmds) black
    set color(brackets) DeepPink
    set color(comments) black
    set color(strings) \#00bb00

    ctext::addHighlightClass $w widgets $color(widgets) \
	[list obutton button label text frame toplevel \
	     scrollbar checkbutton canvas listbox menu menubar menubutton \
	     radiobutton scale entry message spinbutton tk_chooseDir \
	     tk_getSaveFile tk_getOpenFile tk_chooseColor tk_optionMenu \
	     tk_dialog tk_messageBox panedwindow]

    ctext::addHighlightClass $w flags $color(flags) \
	[list -text -command -yscrollcommand \
	     -xscrollcommand -background -foreground -fg -bg \
	     -highlightbackground -y -x -highlightcolor -relief -width \
	     -height -wrap -font -fill -side -outline -style -insertwidth \
	     -textvariable -activebackground -activeforeground \
	     -insertbackground -anchor -orient -troughcolor -nonewline \
	     -expand -type -message -title -offset -in -after -yscroll \
	     -xscroll -forward -regexp -count -exact -padx -ipadx \
	     -filetypes -all -from -to -label -value -variable \
	     -regexp -backwards -forwards -bd -pady -ipady -state -row \
	     -column -cursor -highlightcolors -linemap -menu -tearoff \
	     -displayof -cursor -underline -tags -tag -length]

    ctext::addHighlightClassWithOnlyCharStart $w vars $color(vars) "\$"
    ctext::addHighlightClass $w cmds $color(cmds) \
	[list break case continue exit for foreach if then elseif else \
	    return switch while file info concat join lappend lindex linsert \
	    list llength lrange lreplace lsearch lsort split array parray \
	    append binary format regexp regsub scan string subst \
	    cd clock exec glob pid pwd close eof fblocked fconfigure fcopy \
	    fileevent flush gets open puts read seek socket tell interp \
	    package namespace variable after auto_execok auto_load \
	    auto_mkindex auto_reset bgerror catch error eval expr global \
	    history incr load proc rename set source time trace unknown \
	    unset update uplevel upvar vwait \
	    winfo wm bind event pack place grid font bell clipboard \
	    destroy focus grab lower option raise selection send tk \
	    tkwait tk_bisque tk_focusNext tk_focusPrev \
	    tk_focusFollowsMouse tk_popup tk_setPalette]

    ctext::addHighlightClassForSpecialChars $w brackets $color(brackets) {[]{}}
    ctext::addHighlightClassForRegexp $w comments $color(comments) {\#[^\n\r]*}
    ctext::addHighlightClassForRegexp $w strings $color(strings) {"(\\"|[^"])*"}
}

# build a row of buttons that are shown from left to right
#
# win  -> frame that holds all buttons
# args -> list with pairs of: "button_text button_command"
#
# Returns: a list of all paths to the buttons
#          in the order there where specified
#
# side-effect: the arguments of specified commands are also lappended
#              with the paths of the buttons

proc twDebug::ButtonRow {win args} {
    frame $win
    set index -1
    set width 0
    foreach {but cmd} $args {
	incr index
	if {[string length $but] > $width} {
	    set width [string length $but]
	}
	set b [::ttk::button $win.but$index -text $but]
	# remember command:
	set cmdArray($index) $cmd
	lappend blist $b
	pack $win.but$index -side left -padx 5 -pady 5
    }
    # configure all commands:
    for {set i 0} {$i <= $index} {incr i} {
	set command $cmdArray($i)
	foreach el $blist {
	    lappend command $el
	}
	$win.but$i configure -command $command
    }
    incr width 3
    # second pass to make the button widths equal:
    foreach widget $blist {
	$widget configure -width $width
    }
    return $blist
}

# create a standard widget with scrollbars around
#
# wigdet  -> name of the widget to be created
# parent  -> path to the frame, in which the widget and the scrollbars should
#            be created
# scrollx -> boolean; create horizontal scrollbar?
# scrolly -> boolean; create vertical scrollbar?
# args    -> additional arguments passed on the the widget
#
# returns: the path to the created widget (frame)

proc twDebug::ScrolledWidget {widget parent scrollx scrolly args} {
    # Create widget attached to scrollbars, pass thru $args
    frame $parent
    eval $widget $parent.list $args
    # Create scrollbars attached to the listbox
    if {$scrollx} {
	::ttk::scrollbar $parent.sx -orient horizontal \
	    -command [list $parent.list xview]
	grid $parent.sx -column 0 -row 1 -sticky ew -pady 2
	$parent.list configure -xscrollcommand [list $parent.sx set]
    }
    if {$scrolly} {
	::ttk::scrollbar $parent.sy -orient vertical \
	    -command [list $parent.list yview]
	grid $parent.sy -column 1 -row 0 -sticky ns -padx 2
	$parent.list configure -yscrollcommand [list $parent.sy set]
    }
    # Arrange them in the parent frame
    grid $parent.list  -column 0 -row 0 -sticky ewsn
    grid columnconfigure $parent 0 -weight 1
    grid rowconfigure $parent 0 -weight 1
    # hide the original widget command from the interpreter:
    interp hide {} $parent
    # Install the alias:
    interp alias {} $parent {} twDebug::ScrolledWidgetCmd $parent.list
    # fix the bindings for the listbox:
    bindtags $parent.list [lreplace [bindtags $parent.list] 0 0 $parent]
    #set tags [lrange [bindtags $parent.list] 1 end]
    #bindtags $parent.list "$parent $tags"
    return $parent
}

proc twDebug::ScrolledWidgetCmd {self cmd args} {
    switch -- $cmd {
	widgetPath {
	    return "$self.list"
	}
	default {
	    return [uplevel 1 [list $self $cmd] $args]
	}
    }
}


if {0} {
    package require twDebugInspector
    twDebug::Inspector .newToplevel

    # pkgIndex.tcl
    package ifneeded twDebugInspector 0.1 \
       [list source [file join $dir twDebugInspector.tcl]]
}
