## Itcl only itk replacement.
#
# \file		widget.tcl
# \author	Rene Zaumseil <r.zaumseil@freenet.de
# \copyright	BSD license

#===============================================================================

## Itcl namespace.
namespace eval itcl {}

# Save original configbody command for later use.
# \todo		better solution?
rename ::itcl::configbody ::itk::configbody

## Replacement of configbody command.
#
# \param name	Name of option to configure
# \param body	New configuration body of option
#
# The command checks if the body is for an itk_option.
# The body will be saved in the definition of the itk_option for later use.
# Otherwise the original configbody command is used.
proc ::itcl::configbody {name body} {
  #TODO Should we add the calling namespace?
  if {[string index $name 0] eq {:}} {
    if {[string index $name 1] ne {:}} {;# :..
      set myName :$name
    } else {;# ::..
      set myName $name
    }
  } else {;# ..
    set myName ::$name
  }
  set myNr [string last :: $myName]
  if {$myNr > 3} {
    set myName [string range $myName 0 [incr myNr]]-[string range $myName [incr myNr] end]
    if {[info exists ::itk::option($myName)]} {
      set ::itk::option($myName) [lreplace $::itk::option($myName) 4 4 $body]
      return
    }
  }
  tailcall ::itk::configbody $name $body
}

#===============================================================================

## Namespace containing tcl only itk features.
namespace eval ::itk {

## Variable holding information about available hull's.
# Each array element consist of list with createcommand and optionscommands.
  variable hull
  array set hull {
    toplevel {
	{toplevel $itk_hull -class [namespace tail [info class]]}
	{keep -menu -background -cursor -takefocus}
    }
    frame {
        {frame $itk_hull -class [namespace tail [info class]]}
        {keep -background -cursor}
    }
    labelframe {
        {labelframe $itk_hull -class [namespace tail [info class]]}
        {keep -*}
    }
    labelframe {
        {labelframe $itk_hull -class [namespace tail [info class]]}
        {keep -* ; ignore -class}
    }
    ttk_frame {
        {ttk::frame $itk_hull -class [namespace tail [info class]]}
        {keep -* ; ignore -class}
    }
    ttk_labelframe {
        {ttk::labelframe $itk_hull -class [namespace tail [info class]]}
        {keep -* ; ignore -class}
    }
  }

## Access default hull-handling commands for a mega-widget component.
#
# \param args	?tag? ?createcommand? ?optioncommands?
#
# Return sorted list of all provided hull tags:
# \verbatim
#   hull
# \endverbatim
#
# Return commands of given \a hull :
# \verbatim
#   hull <tag>
# \endverbatim
#
# Set commands of \a tag :
# \verbatim
#   hull <tag> <createcommand> <optioncommands>
# \endverbatim
#
  proc hull {args} {
#puts "##hull $args"
    variable hull
    switch -- [llength $args] {
      0 {return [lsort [array names hull]}
      1 {if {[info exists hull($args)]} {return $hull($args)}}
      2 - 3 {set hull([lindex $args 0]) [lrange $args 1 2]; return}
      default {error "wrong # of args, should be: hull ?tag? ??createcommand? ?optioncommands?"}
    }
  }

## Internal variable holding all usual informations.
  variable usual
  array set usual {}

## Access default option-handling commands for a mega-widget component.
#
# \param args	?tag? ?commands?
#
# Return sorted list of all provided usual tags:
# \verbatim
#   usual
# \endverbatim
#
# Return commands of given \a tag :
# \verbatim
#   usual <tag>
# \endverbatim
#
# Set commands of \a tag :
# \verbatim
#   usual <tag> <commands>
# \endverbatim
#
  proc usual {args} {
#puts "##usual $args"
    # keep, rename, ignore, usual -> see component options
    variable usual
    switch -- [llength $args] {
      0 {return [lsort [array names usual]}
      1 {if {[info exists usual($args)]} {return $usual($args)}}
      2 {set usual([lindex $args 0]) [lindex $args 1]; return}
      default {error "wrong # of args, should be: usual ?tag? ?commands?"}
    }
  }

#-------------------------------------------------------------------------------

## Internal variable holding all option informations.
# \verbatim
#   ::itk::option(::<class>::-<opt>)	{-opt res cls init conf}
# \endverbatim
  variable option
  array set option {}

## Command to work with class options.
#
# \param command	define, add or remove
# \param args		additional arguments depending on given "command"
#
# Inside class definitions options can be defined with:
# \verbatim
#   itk_option define switchName resourceName resourceClass init ?config?
# \endverbatim
#
# The add and remove commands are available inside class methods:
# \verbatim
#   itk_option add optName ?optName optName ...?
#   itk_option remove optName ?optName optName ...?
# \endverbatim
#
  proc ::itk_option {command args} {
    switch -- $command {
      define {
        switch -- [llength $args] {
	  4 {lappend args {}}
	  5 {}
	  default {error "itk_option: wrong # args"}
	}
	set myOpt [lindex $args 0]
        if {[string index $myOpt 0] ne {-}} {error "itk_option: no switch name"}
#TODO info frame
	set myTmp [dict get [::info frame [expr {[::info frame]-2}]] cmd]
	if {[lindex $myTmp 0] ni {::itcl::class itcl::class class}} {error "wrong itk_option define"}
	set myNs ::[string trimleft [lindex $myTmp 1] :]
	set ::itk::option(${myNs}::$myOpt) $args
        uplevel 1 [list protected method itk_configbody {body} {eval $body}]
      }
      add {error "move to constructor"}
      remove {error "move to constructor"}
      default {error "wrong command '$command', should be one of add, define or remove"}
    }
  }
}

#===============================================================================

## The widget class is the basis of all [itk] mega-widgets.
# Provides facilities to merge widget options into a composite list of options
# for the overall widget.
# Derived classes add widgets and methods to specialize behavior.
::itcl::class itk::widget {

## Hook to store data.
# This does not affect the widget operation in any way.
# It is simply a hook that clients can use to store a bit of data with each
# widget.
# This can come in handy when using widgets to build applications.
  itk_option define -clientdata clientData ClientData ""

## Array variable containing all available components.
# The "itk_component" array returns the real window path name for a component
# widget with the symbolic name name.
# The same information can be queried using the component method, but
# accessing this array is faster and more convenient.
  protected variable itk_component

## Array variable containing all available options.
# The "itk_option" array returns the current option value for the composite
# widget option named option.
# Here, the option name should include a leading "-" sign.
# The same information can be queried using the cget method, but accessing
# this array is faster and more convenient.
  protected variable itk_option

## This variable contains the name of the window that acts as a parent for
# internal components.
# It is initialized to the name of the "hull" component provided by the Widget
# and Toplevel classes.
# Derived classes can override the initial setting to point to another interior
# window to be used for further-derived classes.
  protected variable itk_interior .

## Name of internal hull widget.
  private variable itk_hull ""

## Array variable containing option definitions.
#
# \verbatim
#   _option(<comp>.opt)	{-opt res cls init win -winopt}
#   _option(-opt)	{::<class>|<comp>.opt ..}
# \endverbatim
  private variable _option

## Original widget path of itk widget.
  private variable _path

## Internal intepreter to parse option commands.
  private common _interp [interp create -safe --]

## Variable containing protected/public component names.
  protected variable _component

#-------------------------------------------------------------------------------

## Build up itk infrastructure.
#
#  \param args	option value list to initialize object.
#
# First check on new class usages and initialize class related variables.
# Then initialize all internal itk variables off the current object.
# At least call the itk_initialize function.
#
  constructor {hull args} {
#puts "==$this [namespace current] itk::widget constructor $args"
#TODO    if {[string range $this 0 2] ne {::.}} {error "wrong _path name: $this"}
    # Initialize all internal itk variables off the current object.
    array set itk_component {}
    set _component [list]
    array set itk_option {{} 1}
    array set _option {}

    # Build up hull component
    if {$hull ne {}} {
      set itk_hull [namespace tail $this]
      set itk_interior $itk_hull

      itk_component add hull {*}$::itk::hull($hull)
      bind itk-delete-$itk_hull <Destroy> [list itcl::delete object $this]

      set tags [bindtags $itk_hull]
      bindtags $itk_hull [linsert $tags 0 itk-delete-$itk_hull]
    }

    # At least call the itk_initialize function.
    eval itk_initialize $args
  }

#-------------------------------------------------------------------------------

## Cleanup object storage.
  destructor {
#puts "==$this itk::widget destructor"
    if {[winfo exists $itk_hull]} {
      set tags [bindtags $itk_hull]
      set i [lsearch $tags itk-delete-$itk_hull]
      if {$i >= 0} {
        bindtags $itk_hull [lreplace $tags $i $i]
      }
      destroy $itk_hull
    }
    itk_component delete hull

    set components [component]
    foreach component $components {
      set path($component) [component $component]
    }
    foreach component $components {
      if {[winfo exists $path($component)]} {
        destroy $path($component)
      }
    }
    destroy $_path
  }

#-------------------------------------------------------------------------------

## Used to query or access component widgets within a mega-widget.
#
# \param args ?name? ?command arg arg ...?
#
# Invokes the given "command" as a method on the component called "name".
#
# \verbatim
#   pathName component ?name? ?command arg arg ...?
# \endverbatim
# 
  public method component {args} {
#puts "==$this component $args"
    switch -- [llength $args] {
      0 {return $_component}
      1 {
	if {[lsearch $_component $args] != -1} {
	  return $itk_component($args)
	}
      }
      default {
	set myComp [lindex $args 0]
	if {$myComp eq {hull}} {
	  $_path {*}[lrange $args 1 end]
	} elseif {[lsearch $_component $myComp] != -1} {
          $itk_component($myComp) {*}[lrange $args 1 end]}
	}
    }
  }

#-------------------------------------------------------------------------------

## Returns the current value of the configuration option given by option.
#
# \param option	Name of composite configuration option for the mega-widget
#
# Individual components integrate their own configuration options onto the
# composite list when they are registered by the "itk_component add" method.
  public method cget {option} {return $itk_option($option)}

#-------------------------------------------------------------------------------

## Query or modify the configuration options of the widget.
#
# \param args	Option-value list to configure the widget
#
# Return list of all available configuration options:
# \verbatim
#   pathName configure
# \endverbatim
#
# Return information about given configuration option:
# \verbatim
#   pathName configure ?option?
# \endverbatim
#
# Set value of configuration options:
# \verbatim
#   pathName configure ?option? ?value option value ...?
# \endverbatim
#
  public method configure {args} {
    set myNr [llength $args]
    if {$myNr == 0} {
      set myRet [list]
      foreach o [lsort [array names _option -*]] {
	lappend myRet [_option_info $o]
      }
      return $myRet
    } elseif {$myNr == 1} {
      return [_option_info $args]
    } elseif {$myNr%2 == 0} {
      foreach {o v} $args {
	set myOld $itk_option($o)
	set itk_option($o) $v
        if {[catch {
          foreach myName $_option($o) {
            if {[string index $myName 0] eq {:}} {
	      ${myName}::itk_configbody [lindex $::itk::option(${myName}::$o) 4]
	    } else {
              [lindex $_option($myName) 4] configure [lindex $_option($myName) 5] $v
	    }
          }
	} myMsg]} {
	  set itk_option($o) $myOld
	  foreach myName1 $_option($o) {
	    if {$myName1 eq $myName} break
            if {[string index $myName1 0] eq {:}} {
	      ${myName1}::itk_configbody [lindex $::itk::option(${myName1}::$o) 4]
	    } else {
              [lindex $_option($myName1) 4] configure [lindex $_option($myName1) 5] $myOld
	    }
	  }
	  error $myMsg
        }
      }
      return
    }
    error "value for '[lindex $args end]' missing"
  }

## See documentation of configure.
  public method config {args} {tailcall $this configure {*}$args}

#-------------------------------------------------------------------------------

## This method must be invoked within the constructor for each class in a
# mega-widget hierarchy.
#
# \param args	?option value option value...?
#
  protected method itk_initialize {args} {
#puts "==$this itk_initialize $args"
    set myCls [uplevel 1 {namespace current}]
    # Build standard initialization
    _option_add [array names ::itk::option ${myCls}::-*]
    foreach {o v} $args {
      set itk_option($o) $v
    }
    # we are in the outermost class
    if {[info class] eq $myCls} {
      unset itk_option()
      set myArgs [list]
      foreach o [lsort [array names _option -*]] {
	lappend myArgs $o $itk_option($o)
      }
      if {[llength $myArgs]} {configure {*}$myArgs}
    }
  }

#-------------------------------------------------------------------------------

## Command to work with class options.
#
# \param command	One of add, remove or define
# \param args		Command related arguments
#
# \verbatim
#   itk_option add name ?name name ...? 
# \endverbatim
# Adds the option "name" belonging to a class or component
# widget into the option list.  Options can be added even
# if they were not originally kept when the component was
# created.
#
# \verbatim
#   itk_option remove name ?name...?
# \endverbatim
# Removes the option "name" belonging to a class or component
# widget from the option list.  This allows a derived class
# to turn off or redefine undesirable options inherited from
# a base class.
#
  protected method itk_option {command args} {
    switch -- $command {
      add {_option_add $args}
      remove {_option_remove $args}
      define {error "move to class definition"}
      default {error "usage"}
    }
  }

#-------------------------------------------------------------------------------

## The method is used in derived classes as part of the implementation for a
# mega-widget.
#
# \param command	One of add or delete
# \param args		Command related arguments
#
# \verbatim
#   itk_component add ?-protected? ?-private? ?--? name createCmds ?optionCmds?
# \endverbatim
# Creates a component widget and merges its options into
# the composite option list for the overall widget.
#
# \verbatim
#   itk_component delete name ?name name ...?
# \endverbatim
# Destroys a component widget and removes its options from
# the composite option list.
#
  protected method itk_component {command args} {
#puts "==$this itk_component $command $args"
    switch -- $command {
      add {_component_add $args}
      delete {_component_delete $args}
      default {error "wrong command '$command', should be add or delete"}
    }
  }

#-------------------------------------------------------------------------------

## Build up option info list.
#
# \param option	Name of option
  private method _option_info {option} {
    set myName [lindex $_option($option) 0]
    if {[string index $myName 0] eq {:}} {
      return [list {*}[lrange $::itk::option(${myName}::$option) 0 3] $itk_option($option)]
    } else {
      foreach {myOpt myRes myCls myIni myWin myNew} $_option($myName) break
      return [list $myOpt $myRes $myCls $myIni [$myWin cget $myNew]]
    }
  }

## Silent on found.
#
# \param list	List of options to add
  private method _option_add {list} {
#puts "$this ##_option_add $list"
    foreach myName $list {
      # Component <comp>.<optpattern> -> <comp>.<opt>
      set myNr [string last . $myName]
      if {$myNr > -1} {
        set myComp [string range $myName 0 [incr myNr -1]]
        set myOpt -[string range $myName [incr myNr 2] end]
        if {$myComp eq {hull}} {set myWin $_path} else {set myWin $itk_component($myComp)}
        set myOk 0
        foreach myVal [lsearch -index 0 -glob -inline -all [$myWin configure]\
		$myOpt] {
          if {[llength $myVal] != 5} continue
	  incr myOk
          set myIni [lindex $myVal 4]
	  set myOpt [lindex $myVal 0]
	  set myVal [lrange $myVal 0 3]
	  lappend myVal $myWin $myOpt
	  set _option($myName) $myVal
          if {[::info exists _option($myOpt)]} {
            if {[lsearch $_option($myOpt) $myName] == -1} {
              lappend _option($myOpt) $myName
	      if {![info exists itk_option()]} {
                $myWin configure [lindex $_option($myName) 5] $itk_option($myOpt)
	      }
            }
	    continue
	  }
	  set _option($myOpt) $myName
          set itk_option($myOpt) [option get $itk_interior {*}[lrange $myVal 1 2]]
          if {$itk_option($myOpt) eq {}} {
            set itk_option($myOpt) [lindex $myVal 3]
          } else {
            set _option($myName) [lreplace $myVal 3 3 $itk_option($myOpt)]
          }
	  if {![info exists itk_option()]} {
            $myWin configure [lindex $_option($myName) 5] $itk_option($myOpt)
	  }
          continue
	}
        if {$myOk == 0} {
          error "component option name '$myName' not found\n$myWin [winfo class $myWin]\n[join [$myWin configure] \n]"
        }
	continue
      }
      # Class <class>::<opt> -> ::<class>
      set myNr [string last :: $myName]
      if {$myNr > -1} {
	if {[string range $myName 0 1] ne {::}} {
	  set myName ::$myName
	  incr myNr 2
	}
	#TODO test on class in heritage
        set myCls [string range $myName 0 [incr myNr -1]]
        set myOpt [string range $myName [incr myNr 3] end]
 	if {[string index $myOpt 0] ne {-}} {set myOpt -$myOpt}
        set myName ${myCls}::$myOpt
        if {![::info exists ::itk::option($myName)]} {
          error "class option name '$myName' not found"
        }
        if {[::info exists _option($myOpt)]} {
          if {[lsearch $_option($myOpt) $myCls] == -1} {
            lappend _option($myOpt) $myCls
	    if {![info exists itk_option()]} {
	      ${myCls}::itk_configbody [lindex $::itk::option($myName) 4]
	    }
          }
	  continue
	}
	set _option($myOpt) $myCls
        set itk_option($myOpt) [option get $itk_interior {*}[lrange $::itk::option($myName) 1 2]]
        if {$itk_option($myOpt) eq {}} {
          set itk_option($myOpt) [lindex $::itk::option($myName) 3]
        } else {
          set ::itk::option($myName) [lreplace $::itk::option($myName) 3 3 $itk_option($myOpt)]
        }
	if {![info exists itk_option()]} {
	  ${myCls}::itk_configbody [lindex $::itk::option($myName) 4]
	}
	continue
      }
      error "option name '$myName' not found"
    }
  }

#-------------------------------------------------------------------------------

## Silent on not found.
#
# \param list	List of option to remove
#
  private method _option_remove {list} {
#puts "##_option_remove $list\n[parray _option]"
    foreach myName $list {
      set myNr [string last . $myName]
      if {$myNr > -1} {
        set myComp [string range $myName 0 [incr myNr -1]]
        set myOpt -[string range $myName [incr myNr 2] end]
	set myNr [lsearch $_option($myOpt) $myName]
	if {$myNr == -1} continue
	unset _option($myName)
	set _option($myOpt) [lreplace $_option($myOpt) $myNr $myNr]
	if {[llength $_option($myOpt)] == 0} {
	  unset _option($myOpt)
	  unset itk_option($myOpt)
	}
	continue
      }
      set myNr [string last :: $myName]
      if {$myNr > -1} {
       	if {[string range $myName 0 1] ne {::}} {
	  set myName ::$myName
	  incr myNr 2
	}
        set myCls [string range $myName 0 [incr myNr -1]]
        set myOpt -[string range $myName [incr myNr 3] end]
        set myName ${myCls}::$myOpt
	set myNr [lsearch $_option($myOpt) $myCls]
	if {$myNr == -1} continue
	set _option($myOpt) [lreplace $_option($myOpt) $myNr $myNr]
	if {[llength $_option($myOpt)] == 0} {
	  unset _option($myOpt)
	  unset itk_option($myOpt)
	}
	continue
      }
    }
  }

#-------------------------------------------------------------------------------

## Add new component.
#
# \param list	Component add statement
#
# Recognized commands:
# - ignore option ?option option ...?
# - keep option ?option option ...?
# - rename option switchName resourceName resourceClass
# - usual ?tag?
  private method _component_add {list} {
#puts "$this ##_component_add $list"
    set myLevel public
    set i 0
    foreach mySwitch $list {
      switch -- $mySwitch {
        -protected {set myLevel protected;incr i}
	-private {set myLevel private;incr i}
        -- {incr i; break}
        default {
	  if {[string index $mySwitch 0] eq {-}} {
	    error "bad option \"$mySwitch\": should be -private, -protected or --"
	  }
	  break
	}
      }
    }
    if {$i} {set list [lrange $list $i end]}
    if {[llength $list] == 1} {
      error {wrong # args: should be "itk_component add ?-protected? ?-private? ?--? name createCmds ?optionCmds?"}
    }
    if {[llength $list] > 3} {
      error {wrong # args: should be "add ?-protected? ?-private? ?--? name createCmds ?optionCmds?"}
    }
    foreach {myComp myCmd myOpts} $list break
    if {[::info exists itk_component($myComp)]} {
      error "component '$myComp' already exists\n[parray itk_component]\n[parray _option]"
    }
    # Check on special 'hull' component.
    if {$myComp eq {hull}} {
      set myThis ::[namespace tail $this]
      rename $this ::itk::tmp$myThis
    }
    # Create new widget.
    set itk_component($myComp) [uplevel 2 $myCmd]
    if {$myLevel ne {private}} {
      lappend _component $myComp
    }
    # Check on special 'hull' component.
    if {$myComp eq {hull}} {
      set _path .itk$itk_component(hull)
      rename $itk_component(hull) $_path
      set itk_interior $itk_component(hull)
      rename ::itk::tmp$myThis $myThis
    } else {
      # Remove widget options if destroyed
      set myList [bindtags $itk_component($myComp)]
      bind itk-destroy-$itk_component($myComp) <Destroy> [itcl::code $this itk_component delete $myComp]
      bindtags $itk_component($myComp) [list itk-destroy-$itk_component($myComp) {*}$myList]
    }

    # Process the option related part
    set myTag [winfo class $itk_component($myComp)]
    if {$myOpts eq {}} {set myOpts usual}
    interp eval $_interp {
      set ::result {}
      proc rename {args} {lappend ::result rename [lindex $args 0] [lrange $args 1 end]}
      proc keep {args} {foreach o $args {lappend ::result keep $o {}}}
      proc ignore {args} {foreach o $args {lappend ::result ignore $o {}}}
    }
    interp eval $_interp "proc usual {{tag {$myTag}}} {eval \[::itk::usual \$tag\]}"
    interp alias $_interp ::itk::usual {} ::itk::usual
    interp eval $_interp $myOpts
    foreach {myCmd myNew myArgs} [interp eval $_interp {set ::result}] {
      switch -- $myCmd {
	ignore {;# ignore optionpattern
	  foreach myTmp [array names _option $myNew] {
	    _option_remove $myComp.[string range $myTmp 1 end]
	  }
	}
	keep {;# keep optionpattern
	  _option_add $myComp.[string range $myNew 1 end]
	}
	rename {;# rename option switchName resourceName resourceClass
	  if {[llength $myArgs] != 3} {error "wrong rename args"}
          if {$myComp eq {hull}} {set myWin $_path} else {set myWin $itk_component($myComp)}
	  # check window option
          set myVal [lsearch -index 0 -inline [$myWin configure] $myNew]
          if {[llength $myVal] != 5} {error "component option rename '$myNew $myArgs' not found\n$myVal"}
#TODO test
	  set myOpt [lindex $myArgs 0]
	  if {[string index $myOpt 0] ne {-}} {error "component wrong switch '$myNew $myArgs'"}
	  set myName $myComp.[string range $myOpt 1 end]
	  set myVal [list {*}$myArgs [lindex $myVal 3]]
	  lappend myVal $myWin $myNew
	  set _option($myName) $myVal
          if {[::info exists _option($myOpt)]} {
            if {[lsearch $_option($myOpt) $myName] == -1} {
              lappend _option($myOpt) $myName
	      if {![info exists itk_option()]} {;#TODO
                $myWin configure [lindex $_option($myName) 5] $itk_option($myOpt)
	      }
            }
	  } else {
	    set _option($myOpt) $myName
            set itk_option($myOpt) [option get $itk_interior {*}[lrange $myVal 1 2]]
            if {$itk_option($myOpt) eq {}} {
              set itk_option($myOpt) [lindex $myVal 3]
            } else {
              set _option($myName) [lreplace $myVal 3 3 $itk_option($myOpt)]
            }
	    if {![info exists itk_option()]} {;#TODO
              $myWin configure [lindex $_option($myName) 5] $itk_option($myOpt)
	    }
	  }
	}
      }
    }
    interp eval $_interp {set ::result {}}
    return $myComp
  }

#-------------------------------------------------------------------------------

## Delete all given components.
#
# \param list	List of component names
#
  private method _component_delete {list} {
#puts "## _component_delete $list"
    foreach myName $list {
      if {![::info exists itk_component($myName)]} {error "component '$myName' does not exist"}
      _option_remove [array names _option $myName.*]
      ::itk::remove_destroy_hook $itk_component($myName)
      unset itk_component($myName)
      set myNr [lsearch $_component $myName]
      if {$myNr != -1} {
	set _component [lreplace $_component $myNr $myNr]
      }
    }
  }

#-------------------------------------------------------------------------------

}

#===============================================================================
