extern "C" {
#include "tkSDLInt.h"
#include "tkFont.h"
}
#include "SdlTkInt.h"
#include "Xregion.h"

#include "agg_bezier_arc.h"
#include "agg_conv_curve.h"
#include "agg_ellipse.h"
#include "agg_font_freetype.h"
#include "agg_pixfmt_rgb.h"
#include "agg_pixfmt_rgb_packed.h"
#include "agg_pixfmt_rgba.h"
#include "agg_pixfmt_gray.h"
#include "agg_renderer_base.h"
#include "agg_renderer_mclip.h"
#include "agg_renderer_primitives.h"
#include "agg_renderer_outline_aa.h"
#include "agg_rasterizer_outline_aa.h"
#include "agg_rasterizer_scanline_aa.h"
#include "agg_scanline_u.h"
#include "agg_span_pattern_rgba.h"
#include "agg_span_allocator.h"
#include "agg_renderer_scanline.h"
#include "agg_conv_stroke.h"
#include "agg_vcgen_stroke.h"

/* This is a span generator. It is used when drawing text and primitives
 * using a Bitmap as a stipple pattern. */
template<class ColorT,
	class Order, 
	class WrapModeX,
	class WrapModeY,
	class Allocator = agg::span_allocator<ColorT> > 
class span_stipple : public agg::span_pattern_rgba<ColorT, Order, WrapModeX, WrapModeY, Allocator>
{
public:
    typedef ColorT color_type;
    typedef Order order_type;
    typedef Allocator alloc_type;
    typedef agg::span_pattern_rgba<color_type, order_type, WrapModeX, WrapModeY, alloc_type> base_type;
    typedef typename color_type::value_type value_type;

    span_stipple(alloc_type& alloc,
	const agg::rendering_buffer& src, 
	unsigned offset_x, unsigned offset_y) :
            base_type(alloc, src, offset_x, offset_y),
            m_wrap_mode_x(src.width()),
            m_wrap_mode_y(src.height())
    {}

    //--------------------------------------------------------------------
    color_type* generate(int x, int y, unsigned len)
    {   
	color_type* span = base_type::allocator().span();
	unsigned sx = m_wrap_mode_x(x - base_type::offset_x());
	const value_type* row_ptr = 
	    (const value_type*)base_type::base_type::source_image().row(
		m_wrap_mode_y(
		    y - base_type::offset_y()));
	do
	{
	    const value_type* p = row_ptr + sx; /* 1 byte-per-pixel */
	    if (!p[0]) {
		*span = m_color;
	    } else {
		span->clear();
	    }
	    sx = ++m_wrap_mode_x;
	    ++span;
	}
	while(--len);
	return base_type::allocator().span();
    }

    //---------------------------------------------------------------------
    void color(const color_type& c) { m_color = c; }
    const color_type& color() const { return m_color; }

private:
    color_type m_color;
    WrapModeX m_wrap_mode_x;
    WrapModeY m_wrap_mode_y;
};

template<class PixelFormat>
void
doDrawArc(Drawable d, GC gc, int x, int y,
    unsigned int width, unsigned int height, int start, int extent)
{
    SDL_Surface *sdl;
    int xOff = 0, yOff = 0;
    long i;
    REGION *rgn = 0;

    if (IS_WINDOW(d)) {
	rgn = SdlTkGetVisibleRegion((_Window *) d);

	/* Window is unmapped or totally obscured */
	if (XEmptyRegion(rgn))
	    return;
    }

    sdl = SdlTkGetDrawableSurface(d, &xOff, &yOff, NULL);

    /* Lock surface */
    if (SDL_MUSTLOCK(sdl)) {
	if (SDL_LockSurface(sdl) < 0) {
	    return;
	}
    }

    /* Rendering buffer, points to SDL_Surface memory */
    agg::rendering_buffer rbuf((agg::int8u *) sdl->pixels, sdl->w, sdl->h, sdl->pitch);

    /* Pixel-format renderer, a low-level pixel-rendering object */
    PixelFormat ren_pixf(rbuf);

    /* A basic renderer that does clipping to multiple boxes */
    typedef agg::renderer_mclip<PixelFormat> t_renderer_mclip;
    t_renderer_mclip ren_mclip(ren_pixf);

    /* The color, in the format used by the pixel-format renderer */
    Uint8 r, g, b;
    SDL_GetRGB(gc->foreground, SdlTk_sdlsurf->format, &r, &g, &b);
    agg::rgba8 c(r, g, b);

    /* Apparently agg::arc is deprecated */
    agg::bezier_arc arc(xOff + x + width / 2.0, yOff + y + height / 2.0,
	width / 2.0, height / 2.0,
	agg::deg2rad(start / 64), agg::deg2rad(extent / 64));
    typedef agg::conv_curve<agg::bezier_arc, agg::curve3_div, agg::curve4_div> t_conv_curve;
    t_conv_curve curve(arc);
    agg::conv_stroke<t_conv_curve> stroke(curve);
    stroke.width((double) gc->line_width);

    /* Thing that generates scanlines */
    agg::rasterizer_scanline_aa<> rasterizer;
    rasterizer.reset();
    rasterizer.add_path(stroke);

    /* Scanline needed by the rasterizer -> renderer */
    agg::scanline_u8 scanline;

    if (rgn) {
	for (i = 0; i < rgn->numRects; i++) {
	    BoxPtr box = &rgn->rects[i];
	    ren_mclip.add_clip_box(xOff + box->x1, yOff + box->y1,
		xOff + box->x2 - 1, yOff + box->y2 - 1);
	}
    }

    /* FIXME: FillOpaqueStippled not implemented */
    if ((gc->fill_style == FillStippled
	    || gc->fill_style == FillOpaqueStippled)
	    && gc->stipple != None) {

	_Pixmap *stipple = (_Pixmap *) gc->stipple;

	/* Rendering buffer that points to the bitmap */
	agg::rendering_buffer stipple_buf((agg::int8u *) stipple->sdl->pixels,
	    stipple->sdl->w, stipple->sdl->h, stipple->sdl->pitch);

	/* A span allocator holds 1 line of pixels */
	agg::span_allocator<agg::rgba8> span_allocator;

	/* Generates spans (lines of pixels) from a source buffer */
	typedef span_stipple<agg::rgba8, agg::order_argb,
	    agg::wrap_mode_repeat, agg::wrap_mode_repeat> t_span_stipple;
	t_span_stipple span_stipple(span_allocator, stipple_buf, gc->ts_x_origin, gc->ts_y_origin);
	span_stipple.color(c);

	typedef agg::renderer_scanline_aa<t_renderer_mclip, t_span_stipple> t_renderer_scanline_aa;
	t_renderer_scanline_aa ren_scanline_aa(ren_mclip, span_stipple);

	agg::render_scanlines(rasterizer, scanline, ren_scanline_aa);
    } else {
	agg::renderer_scanline_aa_solid<t_renderer_mclip> ren_scanline(ren_mclip);
	ren_scanline.color(c);

	agg::render_scanlines(rasterizer, scanline, ren_scanline);
    }

    /* Unlock surface */
    if (SDL_MUSTLOCK(sdl)) {
	SDL_UnlockSurface(sdl);
    }
}

void
SdlTkGfxDrawArc(Drawable d, GC gc, int x, int y,
    unsigned int width, unsigned int height, int start, int extent)
{
    SDL_Surface *sdl;
    int format;

    sdl = SdlTkGetDrawableSurface(d, NULL, NULL, &format);
    start = -start;
    extent = -extent;

    switch (format) {
	case SDLTK_RGB565: doDrawArc<agg::pixfmt_rgb565>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_BGR565: doDrawArc<agg::pixfmt_bgr565>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_RGB24: doDrawArc<agg::pixfmt_rgb24>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_BGR24: doDrawArc<agg::pixfmt_bgr24>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_RGBA32: doDrawArc<agg::pixfmt_rgba32>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_ARGB32: doDrawArc<agg::pixfmt_argb32>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_BGRA32: doDrawArc<agg::pixfmt_bgra32>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_ABGR32: doDrawArc<agg::pixfmt_abgr32>(d, gc, x, y, width, height, start, extent); break;
    }
}

template<class PixelFormat>
void
doDrawBitmap(
    Drawable src,
    Drawable dest,
    GC gc,
    int src_x, int src_y,
    unsigned int width, unsigned int height,
    int dest_x, int dest_y)
{
    SDL_Surface *sdl;
    int xOff = 0, yOff = 0;
    long i;
    REGION *rgn = 0;
    Region tmpRgn = 0;
    TkpClipMask *clipPtr = (TkpClipMask *) gc->clip_mask;

    if (IS_WINDOW(dest)) {
	rgn = SdlTkGetVisibleRegion((_Window *) dest);

	/* Window is unmapped or totally obscured */
	if (XEmptyRegion(rgn))
	    return;
    }

    sdl = SdlTkGetDrawableSurface(dest, &xOff, &yOff, NULL);

    /* Lock surface */
    if (SDL_MUSTLOCK(sdl)) {
	if (SDL_LockSurface(sdl) < 0) {
	    return;
	}
    }

    /* Rendering buffer, points to SDL_Surface memory */
    agg::rendering_buffer rbuf((agg::int8u *) sdl->pixels, sdl->w, sdl->h, sdl->pitch);

    /* Pixel-format renderer, a low-level pixel-rendering object */
    /* The pixel format should match that of the SDL_Surface */
    PixelFormat ren_pixf(rbuf);

    /* A basic renderer that does clipping to multiple boxes */
    typedef agg::renderer_mclip<PixelFormat> t_renderer_mclip;
    t_renderer_mclip ren_mclip(ren_pixf);

    /* If the clipping region is specified, intersect it with the visible
     * region of the window, or if this is a pixmap then use the clipping
     * region unmodified. */
    if (clipPtr && clipPtr->type == TKP_CLIP_REGION) {
	Region clipRgn = (Region) clipPtr->value.region;
	if (rgn) {
	    tmpRgn = SdlTkRgnPoolGet();
	    XIntersectRegion(rgn, clipRgn, tmpRgn);
	    rgn = tmpRgn;
	} else {
	    rgn = clipRgn;
	}
    }

    if (rgn) {
	for (i = 0; i < rgn->numRects; i++) {
	    BoxPtr box = &rgn->rects[i];
	    ren_mclip.add_clip_box(xOff + box->x1, yOff + box->y1,
		xOff + box->x2 - 1, yOff + box->y2 - 1);
	}
    }

    /* The color, in the format used by the pixel-format renderer */
    Uint8 r, g, b;
    SDL_GetRGB(gc->foreground, SdlTk_sdlsurf->format, &r, &g, &b);
    agg::rgba8 fg(r, g, b);

    SDL_GetRGB(gc->background, SdlTk_sdlsurf->format, &r, &g, &b);
    agg::rgba8 bg(r, g, b);
    bool transparent = true;

    if (clipPtr && clipPtr->type == TKP_CLIP_PIXMAP) {
	if (clipPtr->value.pixmap == src) {
	    transparent = true;
	} else {
	}
    } else {
	transparent = false;
    }

    Uint8 *pixels = (Uint8 *) ((_Pixmap *) src)->sdl->pixels;
    int pitch = ((_Pixmap *) src)->sdl->pitch;
    unsigned int x, y;
    for (y = src_y; y < src_y + height; y++) {
	Uint8 *row = pixels + y * pitch;
	for (x = src_x; x < src_x + width; x++) {
	    if (!row[x])
		ren_mclip.copy_pixel(xOff + dest_x + (x - src_x),
		    yOff + dest_y + (y - src_y), fg);
	    else if (!transparent)
		ren_mclip.copy_pixel(xOff + dest_x + (x - src_x),
		    yOff + dest_y + (y - src_y), bg);
	}
    }

    if (tmpRgn)
	SdlTkRgnPoolFree(tmpRgn);

    /* Unlock surface */
    if (SDL_MUSTLOCK(sdl)) {
	SDL_UnlockSurface(sdl);
    }
}

void
SdlTkGfxDrawBitmap(
    Drawable src,
    Drawable dest,
    GC gc,
    int src_x, int src_y,
    unsigned int width, unsigned int height,
    int dest_x, int dest_y)
{
    SDL_Surface *sdl;
    int format;

    sdl = SdlTkGetDrawableSurface(dest, NULL, NULL, &format);

    switch (format) {
	case SDLTK_RGB565: doDrawBitmap<agg::pixfmt_rgb565>(src, dest, gc, src_x, src_y, width, height, dest_x, dest_y); break;
	case SDLTK_BGR565: doDrawBitmap<agg::pixfmt_bgr565>(src, dest, gc, src_x, src_y, width, height, dest_x, dest_y); break;
	case SDLTK_RGB24: doDrawBitmap<agg::pixfmt_rgb24>(src, dest, gc, src_x, src_y, width, height, dest_x, dest_y); break;
	case SDLTK_BGR24: doDrawBitmap<agg::pixfmt_bgr24>(src, dest, gc, src_x, src_y, width, height, dest_x, dest_y); break;
	case SDLTK_RGBA32: doDrawBitmap<agg::pixfmt_rgba32>(src, dest, gc, src_x, src_y, width, height, dest_x, dest_y); break;
	case SDLTK_ARGB32: doDrawBitmap<agg::pixfmt_argb32>(src, dest, gc, src_x, src_y, width, height, dest_x, dest_y); break;
	case SDLTK_BGRA32: doDrawBitmap<agg::pixfmt_bgra32>(src, dest, gc, src_x, src_y, width, height, dest_x, dest_y); break;
	case SDLTK_ABGR32: doDrawBitmap<agg::pixfmt_abgr32>(src, dest, gc, src_x, src_y, width, height, dest_x, dest_y); break;
    }
}

class VertexSource_XPoints {
public:
    VertexSource_XPoints(XPoint *points, int npoints, int xOff, int yOff) :
	m_points(points),
	m_npoints(npoints),
	m_xOff(xOff),
	m_yOff(yOff),
	m_idx(0)
    {
    }
    void rewind(unsigned path_id)
    {
	m_idx = 0;
    }
    unsigned vertex(double *x, double *y)
    {
	if (m_idx == 0) {
	    *x = m_xOff + m_points[m_idx].x;
	    *y = m_yOff + m_points[m_idx].y;
	    ++m_idx;
	    return agg::path_cmd_move_to;
	}
	if (m_idx < m_npoints) {
	    *x = m_xOff + m_points[m_idx].x;
	    *y = m_yOff + m_points[m_idx].y;
	    ++m_idx;
	    return agg::path_cmd_line_to;
	}
	return agg::path_cmd_stop;
    }
private:
    XPoint *m_points;
    int m_npoints;
    int m_xOff, m_yOff;
    int m_idx;
};

template<class PixelFormat>
void
doDrawLines(Drawable d, GC gc, XPoint *points, int npoints, int mode)
{
    SDL_Surface *sdl;
    int xOff = 0, yOff = 0;
    long i;
    REGION *rgn = 0;

    if (IS_WINDOW(d)) {
	rgn = SdlTkGetVisibleRegion((_Window *) d);

	/* Window is unmapped or totally obscured */
	if (XEmptyRegion(rgn))
	    return;
    }

    sdl = SdlTkGetDrawableSurface(d, &xOff, &yOff, NULL);

    /* Lock surface */
    if (SDL_MUSTLOCK(sdl)) {
	if (SDL_LockSurface(sdl) < 0) {
	    return;
	}
    }

    /* Rendering buffer, points to SDL_Surface memory */
    agg::rendering_buffer rbuf((agg::int8u *) sdl->pixels, sdl->w, sdl->h, sdl->pitch);

    /* Pixel-format renderer, a low-level pixel-rendering object */
    PixelFormat ren_pixf(rbuf);

    /* A basic renderer that does clipping to multiple boxes */
    typedef agg::renderer_mclip<PixelFormat> t_renderer_mclip;
    t_renderer_mclip ren_mclip(ren_pixf);

    /* The color, in the format used by the pixel-format renderer */
    Uint8 r, g, b;
    SDL_GetRGB(gc->foreground, SdlTk_sdlsurf->format, &r, &g, &b);
    agg::rgba8 c(r, g, b);

    VertexSource_XPoints vertexSrc(points, npoints, xOff, yOff);
    agg::conv_stroke<VertexSource_XPoints> stroke(vertexSrc);
    stroke.width((double) gc->line_width);

    if (gc->line_width >= 2) {
	switch (gc->cap_style) {
	    case CapNotLast:
	    case CapButt:
		stroke.line_cap(agg::butt_cap);
		break;
	    case CapRound:
		stroke.line_cap(agg::round_cap);
		break;
	    default:
		stroke.line_cap(agg::square_cap);
		break;
	}
	switch (gc->join_style) {
	    case JoinMiter:
		stroke.line_join(agg::miter_join);
		break;
	    case JoinRound:
		stroke.line_join(agg::round_join);
		break;
	    default:
		stroke.line_join(agg::bevel_join);
		break;
	}
    }

    /* Thing that generates scanlines */
    agg::rasterizer_scanline_aa<> rasterizer;
    rasterizer.reset();
    rasterizer.add_path(stroke);

    /* Scanline needed by the rasterizer -> renderer */
    agg::scanline_u8 scanline;

    if (rgn) {
	for (i = 0; i < rgn->numRects; i++) {
	    BoxPtr box = &rgn->rects[i];
	    ren_mclip.add_clip_box(xOff + box->x1, yOff + box->y1,
		xOff + box->x2 - 1, yOff + box->y2 - 1);
	}
    }

    if ((gc->fill_style == FillStippled
	    || gc->fill_style == FillOpaqueStippled)
	    && gc->stipple != None) {

	_Pixmap *stipple = (_Pixmap *) gc->stipple;

	/* Rendering buffer that points to the bitmap */
	agg::rendering_buffer stipple_buf((agg::int8u *) stipple->sdl->pixels,
	    stipple->sdl->w, stipple->sdl->h, stipple->sdl->pitch);

	/* A span allocator holds 1 line of pixels */
	agg::span_allocator<agg::rgba8> span_allocator;

	/* Generates spans (lines of pixels) from a source buffer */
	typedef span_stipple<agg::rgba8, agg::order_argb,
	    agg::wrap_mode_repeat, agg::wrap_mode_repeat> t_span_stipple;
	t_span_stipple span_stipple(span_allocator, stipple_buf, gc->ts_x_origin, gc->ts_y_origin);
	span_stipple.color(c);

	typedef agg::renderer_scanline_aa<t_renderer_mclip, t_span_stipple> t_renderer_scanline_aa;
	t_renderer_scanline_aa ren_scanline_aa(ren_mclip, span_stipple);

	agg::render_scanlines(rasterizer, scanline, ren_scanline_aa);
    } else {
	typedef agg::renderer_scanline_aa_solid<t_renderer_mclip> t_renderer_scanline_aa_solid;
	t_renderer_scanline_aa_solid ren_scanline(ren_mclip);
	ren_scanline.color(c);

	agg::render_scanlines(rasterizer, scanline, ren_scanline);
    }

    /* Unlock surface */
    if (SDL_MUSTLOCK(sdl)) {
	SDL_UnlockSurface(sdl);
    }
}

void
SdlTkGfxDrawLines(Drawable d, GC gc, XPoint *points, int npoints, int mode)
{
    SDL_Surface *sdl;
    int format;

    sdl = SdlTkGetDrawableSurface(d, NULL, NULL, &format);

    switch (format) {
	case SDLTK_RGB565: doDrawLines<agg::pixfmt_rgb565>(d, gc, points, npoints, mode); break;
	case SDLTK_BGR565: doDrawLines<agg::pixfmt_bgr565>(d, gc, points, npoints, mode); break;
	case SDLTK_RGB24: doDrawLines<agg::pixfmt_rgb24>(d, gc, points, npoints, mode); break;
	case SDLTK_BGR24: doDrawLines<agg::pixfmt_bgr24>(d, gc, points, npoints, mode); break;
	case SDLTK_RGBA32: doDrawLines<agg::pixfmt_rgba32>(d, gc, points, npoints, mode); break;
	case SDLTK_ARGB32: doDrawLines<agg::pixfmt_argb32>(d, gc, points, npoints, mode); break;
	case SDLTK_BGRA32: doDrawLines<agg::pixfmt_bgra32>(d, gc, points, npoints, mode); break;
	case SDLTK_ABGR32: doDrawLines<agg::pixfmt_abgr32>(d, gc, points, npoints, mode); break;
    }
}

/* http://www.libsdl.org/cgi/docwiki.cgi/Pixel_20Access */
static void
putpixel(SDL_Surface *surface, int x, int y, Uint32 pixel)
{
    int bpp = surface->format->BytesPerPixel;
    /* Here p is the address to the pixel we want to set */
    Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * bpp;

    switch(bpp) {
    case 1:
        *p = pixel;
        break;

    case 2:
        *(Uint16 *)p = pixel;
        break;

    case 3:
        if(SDL_BYTEORDER == SDL_BIG_ENDIAN) {
            p[0] = (pixel >> 16) & 0xff;
            p[1] = (pixel >> 8) & 0xff;
            p[2] = pixel & 0xff;
        } else {
            p[0] = pixel & 0xff;
            p[1] = (pixel >> 8) & 0xff;
            p[2] = (pixel >> 16) & 0xff;
        }
        break;

    case 4:
        *(Uint32 *)p = pixel;
        break;
    }
}

void
SdlTkGfxDrawPoint(Drawable d, GC gc, int x, int y)
{
    SDL_Surface *sdl;
    int xOff = 0, yOff = 0;
    REGION *rgn = 0;

    if (IS_WINDOW(d)) {
	rgn = SdlTkGetVisibleRegion((_Window *) d);

	/* Window is unmapped or totally obscured */
	if (XEmptyRegion(rgn))
	    return;
    }

    sdl = SdlTkGetDrawableSurface(d, &xOff, &yOff, NULL);

    /* Lock surface */
    if (SDL_MUSTLOCK(sdl)) {
	if (SDL_LockSurface(sdl) < 0) {
	    return;
	}
    }

    if (x >= 0 && x < sdl->w && y >= 0 && y < sdl->h) {
	if (rgn) {
	    if (XPointInRegion(rgn, x, y)) {
		putpixel(sdl, x, y, gc->foreground);
	    }
	} else {
	    putpixel(sdl, x, y, gc->foreground);
	}
    }

    /* Unlock surface */
    if (SDL_MUSTLOCK(sdl)) {
	SDL_UnlockSurface(sdl);
    }
}

class VertexSource_XRectangle {
public:
    VertexSource_XRectangle(int x, int y, int w, int h) :
	m_idx(0)
    {
	m_rect.x = x, m_rect.y = y, m_rect.width = w, m_rect.height = h;
    }
    void rewind(unsigned path_id)
    {
	m_idx = 0;
    }
    unsigned vertex(double *x, double *y)
    {
	if (m_idx == 0) {
	    *x = m_rect.x;
	    *y = m_rect.y;
	    ++m_idx;
	    return agg::path_cmd_move_to;
	}
	if (m_idx == 1) {
	    *x = m_rect.x + m_rect.width;
	    *y = m_rect.y;
	    ++m_idx;
	    return agg::path_cmd_line_to;
	}
	if (m_idx == 2) {
	    *x = m_rect.x + m_rect.width;
	    *y = m_rect.y + m_rect.height;
	    ++m_idx;
	    return agg::path_cmd_line_to;
	}
	if (m_idx == 3) {
	    *x = m_rect.x;
	    *y = m_rect.y + m_rect.height;
	    ++m_idx;
	    return agg::path_cmd_line_to;
	}
	if (m_idx == 4) {
	    *x = m_rect.x;
	    *y = m_rect.y;
	    ++m_idx;
	    return agg::path_cmd_end_poly | agg::path_flags_close;
	}
	return agg::path_cmd_stop;
    }
private:
    XRectangle m_rect;
    int m_idx;
};

template<class PixelFormat>
void
doDrawRect(Drawable d, GC gc, int x, int y, int w, int h)
{
    SDL_Surface *sdl;
    int xOff = 0, yOff = 0;
    long i;
    REGION *rgn = 0;
    Region tmpRgn = 0;
    TkpClipMask *clipPtr = (TkpClipMask *) gc->clip_mask;

    if (IS_WINDOW(d)) {
	rgn = SdlTkGetVisibleRegion((_Window *) d);

	/* Window is unmapped or totally obscured */
	if (XEmptyRegion(rgn))
	    return;
    }

    sdl = SdlTkGetDrawableSurface(d, &xOff, &yOff, NULL);
    x += xOff;
    y += yOff;

    /* Lock surface */
    if (SDL_MUSTLOCK(sdl)) {
	if (SDL_LockSurface(sdl) < 0) {
	    return;
	}
    }

    /* Rendering buffer, points to SDL_Surface memory */
    agg::rendering_buffer rbuf((agg::int8u *) sdl->pixels, sdl->w, sdl->h, sdl->pitch);

    /* Pixel-format renderer, a low-level pixel-rendering object */
    PixelFormat ren_pixf(rbuf);

    /* A basic renderer that does clipping to multiple boxes */
    typedef agg::renderer_mclip<PixelFormat> t_renderer_mclip;
    t_renderer_mclip ren_mclip(ren_pixf);

    /* The color, in the format used by the pixel-format renderer */
    Uint8 r, g, b;
    SDL_GetRGB(gc->foreground, SdlTk_sdlsurf->format, &r, &g, &b);
    agg::rgba8 c(r, g, b);

#if 1
    /* If the clipping region is specified, intersect it with the visible
     * region of the window, or if this is a pixmap then use the clipping
     * region unmodified. */
    if (clipPtr && clipPtr->type == TKP_CLIP_REGION) {
	Region clipRgn = (Region) clipPtr->value.region;
	if (rgn) {
	    tmpRgn = SdlTkRgnPoolGet();
	    XIntersectRegion(rgn, clipRgn, tmpRgn);
	    rgn = tmpRgn;
	} else {
	    rgn = clipRgn;
	}
    }

    if (rgn) {
	for (i = 0; i < rgn->numRects; i++) {
	    BoxPtr box = &rgn->rects[i];
	    ren_mclip.add_clip_box(xOff + box->x1, yOff + box->y1,
		xOff + box->x2 - 1, yOff + box->y2 - 1);
	}
    }

    /* A 1-pixel thick line is inside the top-left, but outside the
     * bottom-right (that is what Tk expects and how Win32 draws it) */
    if (gc->line_width == 1) {
	agg::renderer_primitives<t_renderer_mclip> ren_prim(ren_mclip);
	ren_prim.line_color(c);
	ren_prim.rectangle(x, y, x + w, y + h);

    /* This handles 1-pixel thick lines correctly but is slower */
    } else {
	int thick = gc->line_width;
	int half = thick / 2;
	int noDups = thick; /* to avoid drawing pixels twice */
	ren_mclip.copy_bar(
	    x - half,
	    y - half,
	    x + w - half + thick - 1,
	    y - half + thick - 1, c); /* top */
	ren_mclip.copy_bar(
	    x - half,
	    y + h - half,
	    x + w - half + thick - 1,
	    y + h - half + thick - 1, c); /* bottom */
	ren_mclip.copy_bar(
	    x - half,
	    y - half + noDups,
	    x - half + thick - 1,
	    y + h - half + thick - 1 - noDups, c); /* left */
	ren_mclip.copy_bar(
	    x + w - half,
	    y - half + noDups,
	    x + w - half + thick - 1,
	    y + h - half + thick - 1 - noDups, c); /* right */
    }

#else /* aa */
    typedef agg::renderer_scanline_aa_solid<t_renderer_mclip> t_renderer_scanline_aa_solid;
    t_renderer_scanline_aa_solid ren_scanline(ren_mclip);
    ren_scanline.color(c);

/* A 1-pixel thick line appears as 2-pixel thick anti-aliased line. But that
 * is outside the bounds of a canvas rect item. */
if (gc->line_width & 1) { x += 1, y += 1, w -= 2, h -= 2; }

    VertexSource_XRectangle vertexSrc(x, y, w, h);
    agg::conv_stroke<VertexSource_XRectangle> stroke(vertexSrc);
    stroke.width((double) gc->line_width);

    /* Thing that generates scanlines */
    agg::rasterizer_scanline_aa<> rasterizer;
    rasterizer.reset();
    rasterizer.add_path(stroke);

    /* Scanline needed by the rasterizer -> renderer */
    agg::scanline_u8 scanline;

    /* If the clipping region is specified, intersect it with the visible
     * region of the window, or if this is a pixmap then use the clipping
     * region unmodified. */
    if (clipPtr && clipPtr->type == TKP_CLIP_REGION) {
	Region clipRgn = (Region) clipPtr->value.region;
	if (rgn) {
	    tmpRgn = SdlTkRgnPoolGet();
	    XIntersectRegion(rgn, clipRgn, tmpRgn);
	    rgn = tmpRgn;
	} else {
	    rgn = clipRgn;
	}
    }

    if (rgn) {
	for (i = 0; i < rgn->numRects; i++) {
	    BoxPtr box = &rgn->rects[i];
	    ren_mclip.add_clip_box(xOff + box->x1, yOff + box->y1,
		xOff + box->x2 - 1, yOff + box->y2 - 1);
	}
    }

    agg::render_scanlines(rasterizer, scanline, ren_scanline);
#endif

    if (tmpRgn)
	SdlTkRgnPoolFree(tmpRgn);

    /* Unlock surface */
    if (SDL_MUSTLOCK(sdl)) {
	SDL_UnlockSurface(sdl);
    }
}

/* This is a pixel-format renderer that simply XORs the destination pixel */
/* For 3 bytes-per-pixel */
class pixfmt_3Bpp_xor
{
public:
    typedef agg::rendering_buffer::row_data row_data;
    typedef agg::rendering_buffer::span_data span_data;
    typedef agg::rgba8 color_type;

    //--------------------------------------------------------------------
    pixfmt_3Bpp_xor(agg::rendering_buffer& rb) :
	m_rbuf(&rb)
    {}

    //--------------------------------------------------------------------
    AGG_INLINE unsigned width()  const { return m_rbuf->width();  }
    AGG_INLINE unsigned height() const { return m_rbuf->height(); }

    //--------------------------------------------------------------------
    void copy_pixel(int x, int y, const color_type& c)
    {
	Uint8* p = (Uint8*)m_rbuf->row(y) + x;
	p[0] ^= 0xFF;
	p[1] ^= 0xFF;
	p[2] ^= 0xFF;
    }

    //--------------------------------------------------------------------
    void copy_hline(int x, int y, 
			    unsigned len, 
			    const color_type& c)
    {
	Uint8* p = (Uint8*)m_rbuf->row(y) + x + x + x;
	do
	{
	    p[0] ^= 0xFF;
	    p[1] ^= 0xFF;
	    p[2] ^= 0xFF;
	    p += 3;
	}
	while(--len);
    }

    //--------------------------------------------------------------------
    void blend_hline(int x, int y,
		    unsigned len, 
		    const color_type& c,
		    agg::int8u cover)
    {
	Uint8* p = (Uint8*)m_rbuf->row(y) + x + x + x;
	do
	{
	    p[0] ^= 0xFF;
	    p[1] ^= 0xFF;
	    p[2] ^= 0xFF;
	    p += 3;
	}
	while(--len);
    }

    //--------------------------------------------------------------------
    void blend_vline(int x, int y,
		    unsigned len, 
		    const color_type& c,
		    agg::int8u cover)
    {
	Uint8* p = (Uint8*)m_rbuf->row(y) + x + x + x;
	do
	{
	    p[0] ^= 0xFF;
	    p[1] ^= 0xFF;
	    p[2] ^= 0xFF;
	    p = (Uint8*)m_rbuf->next_row(p);
	}
	while(--len);
    }

private:
    agg::rendering_buffer* m_rbuf;
};

/* This is a pixel-format renderer that simply XORs the destination pixel */
/* For 1, 2 or 4 bytes-per-pixel */
template<class Type> class pixfmt_1_2_4Bpp_xor
{
public:
    typedef agg::rendering_buffer::row_data row_data;
    typedef agg::rendering_buffer::span_data span_data;
    typedef agg::rgba8 color_type;

    //--------------------------------------------------------------------
    pixfmt_1_2_4Bpp_xor(agg::rendering_buffer& rb) :
	m_rbuf(&rb)
    {}

    //--------------------------------------------------------------------
    AGG_INLINE unsigned width()  const { return m_rbuf->width();  }
    AGG_INLINE unsigned height() const { return m_rbuf->height(); }

    //--------------------------------------------------------------------
    void copy_pixel(int x, int y, const color_type& c)
    {
	Type* p = (Type*)m_rbuf->row(y) + x;
	*p ^= 0xFFFFFFFF;
    }

    //--------------------------------------------------------------------
    void copy_hline(int x, int y, 
		    unsigned len, 
		    const color_type& c)
    {
	Type* p = (Type*)m_rbuf->row(y) + x;
	do
	{
	    *p ^= 0xFFFFFFFF;
	    p += 1;
	}
	while(--len);
    }

    //--------------------------------------------------------------------
    void blend_hline(int x, int y,
		    unsigned len, 
		    const color_type& c,
		    agg::int8u cover)
    {
	Type* p = (Type*)m_rbuf->row(y) + x;
	do
	{
	    *p ^= 0xFFFFFFFF;
	    p += 1;
	}
	while(--len);
    }

    //--------------------------------------------------------------------
    void blend_vline(int x, int y,
		    unsigned len, 
		    const color_type& c,
		    agg::int8u cover)
    {
	Type* p = (Type*)m_rbuf->row(y) + x;
	do
	{
	    *p ^= 0xFFFFFFFF;
	    p = (Type*)m_rbuf->next_row(p);
	}
	while(--len);
    }

private:
    agg::rendering_buffer* m_rbuf;
};

void
SdlTkGfxDrawRect(Drawable d, GC gc, int x, int y, int w, int h)
{
    SDL_Surface *sdl;
    int format;

    sdl = SdlTkGetDrawableSurface(d, NULL, NULL, &format);

    if (gc->function == GXinvert) {
	switch (sdl->format->BitsPerPixel) {
	    case 16: doDrawRect<pixfmt_1_2_4Bpp_xor<Uint16> >(d, gc, x, y, w, h); break;
	    case 24: doDrawRect<pixfmt_3Bpp_xor>(d, gc, x, y, w, h); break;
	    case 32: doDrawRect<pixfmt_1_2_4Bpp_xor<Uint32> >(d, gc, x, y, w, h); break;
	}
	return;
    }

    switch (format) {
	case SDLTK_RGB565: doDrawRect<agg::pixfmt_rgb565>(d, gc, x, y, w, h); break;
	case SDLTK_BGR565: doDrawRect<agg::pixfmt_bgr565>(d, gc, x, y, w, h); break;
	case SDLTK_RGB24: doDrawRect<agg::pixfmt_rgb24>(d, gc, x, y, w, h); break;
	case SDLTK_BGR24: doDrawRect<agg::pixfmt_bgr24>(d, gc, x, y, w, h); break;
	case SDLTK_RGBA32: doDrawRect<agg::pixfmt_rgba32>(d, gc, x, y, w, h); break;
	case SDLTK_ARGB32: doDrawRect<agg::pixfmt_argb32>(d, gc, x, y, w, h); break;
	case SDLTK_BGRA32: doDrawRect<agg::pixfmt_bgra32>(d, gc, x, y, w, h); break;
	case SDLTK_ABGR32: doDrawRect<agg::pixfmt_abgr32>(d, gc, x, y, w, h); break;
    }
}

typedef agg::font_engine_freetype_int16 t_font_engine;
typedef agg::font_cache_manager<t_font_engine> t_font_manager;
static t_font_engine *feng = 0;
static t_font_manager *fman = 0;

XFontStruct *
SdlTkGfxAllocFontStruct(_Font *_f)
{
#if 1
    XFontStruct *fs = (XFontStruct *) ckalloc(sizeof (XFontStruct));
    memset(fs, 0, sizeof(XFontStruct));

    if (!feng) {
	feng = new t_font_engine;
	fman = new t_font_manager(*feng);
    }
    (void) feng->load_font(_f->file, _f->index, agg::glyph_ren_agg_gray8,
                           (const char *) SdlTkGetFTStream(_f->file));
    feng->flip_y(true);
    feng->height(_f->size);

    fs->fid = (Font) _f;
    fs->ascent = (int) feng->ascender();
    fs->descent = (int) -feng->descender();
#if 1
    fs->ascent++;
    fs->descent++;
#endif
    fs->max_bounds.width = 10; /* FIXME */
    return fs;
#else
    XFontStruct *fs = new XFontStruct;
    int i;

    if (!feng) {
	feng = new t_font_engine;
	fman = new t_font_manager(*feng);
    }
    (void) feng->load_font(_f->file, _f->index, agg::glyph_ren_agg_gray8,
                           (const char *) SdlTkGetFTStream(_f->file));
    feng->flip_y(true);
    feng->height(_f->size);

    memset(fs, 0, sizeof(XFontStruct));
    fs->fid = (Font) _f;
    fs->ascent = (int) feng->ascender();
    fs->descent = (int) -feng->descender();

    /* Unicode */
    if (_f->chars != NULL) {
	int minLo = 255, maxLo = 0, minHi = 255, maxHi = 0;
	int i, limit, scale;
	for (i = 0; i < _f->nchars; i++) {
	    unsigned char *p = (unsigned char *) &_f->chars[i];
	    int hi = p[0], lo = p[1];
	    if (hi < minHi) minHi = hi;
	    if (hi > maxHi) maxHi = hi;
	    if (lo < minLo) minLo = lo;
	    if (lo > maxLo) maxLo = lo;
	}
	scale = maxLo - minLo + 1;
	limit = (maxHi - minHi + 1) * (maxLo - minLo + 1);
	/* FIXME: 256 * 256 * sizeof(XCharStruct) == 768 KB */
	fs->per_char = (XCharStruct *) ckalloc(sizeof(XCharStruct) * limit);
	memset(fs->per_char, 0, sizeof(XCharStruct) * limit);
	for (i = 0; i < _f->nchars; i++) {
	    const agg::glyph_cache *glyph = fman->glyph(_f->chars[i]);
	    if (glyph) {
		unsigned char *p = (unsigned char *) &_f->chars[i];
		int hi = p[0], lo = p[1];
		int n = (hi - minHi) * scale + lo - minLo;
		fs->per_char[n].width = (short) glyph->advance_x;
		if (glyph->advance_x > fs->max_bounds.width)
		    fs->max_bounds.width = (short) glyph->advance_x;
	    }
	}

	fs->min_byte1 = minHi; /* min row */
	fs->max_byte1 = maxHi; /* max row */
	fs->min_char_or_byte2 = minLo; /* min column */
	fs->max_char_or_byte2 = maxLo; /* max column */

    /* Ascii */
    } else {
	fs->per_char = (XCharStruct *) ckalloc(sizeof(XCharStruct) * (126 - 32 + 1));
	memset(fs->per_char, 0, sizeof(XCharStruct) * (126 - 32 + 1));
	for (i = 32; i < 126; i++) {
	    const agg::glyph_cache *glyph = fman->glyph((char)i);
	    if (glyph) {
		fs->per_char[i-32].width = (short) glyph->advance_x;
		if (glyph->advance_x > fs->max_bounds.width)
		    fs->max_bounds.width = (short) glyph->advance_x;
	    } else {
		fs->per_char[i-32].width = 0;
	    }
	}

	fs->min_char_or_byte2 = 32; /* ascii */
	fs->max_char_or_byte2 = 126; /* ascii */
    }

    return fs;
#endif
}

extern "C" unsigned SdlTkGetNthGlyphIndex(_Font *_f, const char *s, int n);

extern "C" void SdlTkFontInitCharWidths(Font f, int widths[256]);

void
SdlTkFontInitCharWidths(
    Font f,
    int widths[256]
)
{
    _Font *_f = (_Font *) f;
    int i;

    (void) feng->load_font(_f->file, _f->index, agg::glyph_ren_agg_gray8,
                           (const char *) SdlTkGetFTStream(_f->file));
    feng->flip_y(true);
    feng->height(_f->size);

    for (i = 0; i < 256; i++) {
	const agg::glyph_cache *glyph = fman->glyph(i);
	if (glyph)
	    widths[i] = (int) glyph->advance_x;
	else
	    widths[i] = 0;
    }
}

int SdlTkGfxTextWidth(Font f, const char *string, int length)
{
    _Font *_f = (_Font *) f;
    int i;
    double w = 0.0;

    (void) feng->load_font(_f->file, _f->index, agg::glyph_ren_agg_gray8,
                           (const char *) SdlTkGetFTStream(_f->file));
    feng->flip_y(true);
    feng->height(_f->size);

    length /= sizeof(unsigned int) /* FcChar32 */;

    for (i = 0; i < length; i++) {
	const agg::glyph_cache *glyph = fman->glyph(SdlTkGetNthGlyphIndex(_f, string, i));
	if (glyph)
	    w += glyph->advance_x;
    }
    return (int) w;
}

template<class PixelFormat>
void
doDrawString(Drawable d, GC gc, int x, int y, const char *string, int length)
{
    SDL_Surface *sdl;
    int xOff = 0, yOff = 0;
    long i;
    _Font *_f = (_Font *) gc->font;
    double fx, fy;
    TkpClipMask *clipPtr = (TkpClipMask *) gc->clip_mask;
    REGION *rgn = 0;
    Region tmpRgn = 0;

    if (IS_WINDOW(d)) {
	rgn = SdlTkGetVisibleRegion((_Window *) d);

	/* Window is unmapped or totally obscured */
	if (XEmptyRegion(rgn))
	    return;
    }

    sdl = SdlTkGetDrawableSurface(d, &xOff, &yOff, NULL);

    /* Lock surface */
    if (SDL_MUSTLOCK(sdl)) {
	if (SDL_LockSurface(sdl) < 0) {
	    return;
	}
    }

    /* Rendering buffer, points to SDL_Surface memory */
    agg::rendering_buffer rbuf((agg::int8u *) sdl->pixels, sdl->w, sdl->h,
	sdl->pitch);

    /* Pixel-format renderer, a low-level pixel-rendering object */
    PixelFormat ren_pixf(rbuf);

    /* A basic renderer that does clipping to multiple boxes */
    typedef agg::renderer_mclip<PixelFormat> t_renderer_mclip;
    t_renderer_mclip ren_mclip(ren_pixf);

    /* The color, in the format used by the pixel-format renderer */
    Uint8 r, g, b;
    SDL_GetRGB(gc->foreground, SdlTk_sdlsurf->format, &r, &g, &b);
    agg::rgba8 c(r, g, b);

    fx = xOff + x;
    fy = yOff + y;

    /* agg::glyph_ren_agg_gray8 is BROKEN with MS Gothic japanese chars */
    (void) feng->load_font(_f->file, _f->index, agg::glyph_ren_native_gray8,
                           (const char *) SdlTkGetFTStream(_f->file));
    feng->flip_y(true);
    feng->height(_f->size);

    /* If the clipping region is specified, intersect it with the visible
     * region of the window, or if this is a pixmap then use the clipping
     * region unmodified. */
    if (clipPtr && clipPtr->type == TKP_CLIP_REGION) {
	Region clipRgn = (Region) clipPtr->value.region;
	if (rgn) {
	    tmpRgn = SdlTkRgnPoolGet();
	    XIntersectRegion(rgn, clipRgn, tmpRgn);
	    rgn = tmpRgn;
	} else {
	    rgn = clipRgn;
	}
    }

    if (rgn) {
	for (i = 0; i < rgn->numRects; i++) {
	    BoxPtr box = &rgn->rects[i];
	    ren_mclip.add_clip_box(xOff + box->x1, yOff + box->y1,
		xOff + box->x2 - 1, yOff + box->y2 - 1);
	}
    }

    length /= sizeof(unsigned int) /* FcChar32 */;

    /* FIXME: FillOpaqueStippled not implemented */
    if ((gc->fill_style == FillStippled
	    || gc->fill_style == FillOpaqueStippled)
	    && gc->stipple != None) {

	_Pixmap *stipple = (_Pixmap *) gc->stipple;

	/* Rendering buffer that points to the bitmap */
	agg::rendering_buffer stipple_buf((agg::int8u *) stipple->sdl->pixels,
	    stipple->sdl->w, stipple->sdl->h, stipple->sdl->pitch);

	/* A span allocator holds 1 line of pixels */
	agg::span_allocator<agg::rgba8> span_allocator;

	/* Generates spans (lines of pixels) from a source buffer */
	typedef span_stipple<agg::rgba8, agg::order_argb,
	    agg::wrap_mode_repeat, agg::wrap_mode_repeat> t_span_stipple;
	/* FIXME: stippled text doesn't line up with other stippled primitives. */
	t_span_stipple span_stipple(span_allocator, stipple_buf, gc->ts_x_origin + 1, gc->ts_y_origin);
	span_stipple.color(c);

	typedef agg::renderer_scanline_aa<t_renderer_mclip, t_span_stipple> t_renderer_scanline_aa;
	t_renderer_scanline_aa ren_scanline_aa(ren_mclip, span_stipple);

	for (i = 0; i < length; i++) {
	    const agg::glyph_cache *glyph = fman->glyph(SdlTkGetNthGlyphIndex(_f, string, i));
	    if (glyph) {
		fman->init_embedded_adaptors(glyph, fx, fy);
		agg::render_scanlines(fman->gray8_adaptor(),
		    fman->gray8_scanline(), ren_scanline_aa);
		fx += glyph->advance_x;
		fy += glyph->advance_y;
	    }
	}
    } else {
	typedef agg::renderer_scanline_aa_solid<t_renderer_mclip> t_renderer_scanline_aa_solid;
	t_renderer_scanline_aa_solid ren_aa(ren_mclip);
	ren_aa.color(c);

	for (i = 0; i < length; i++) {
	    const agg::glyph_cache *glyph = fman->glyph(SdlTkGetNthGlyphIndex(_f, string, i));
	    if (glyph) {
		fman->init_embedded_adaptors(glyph, fx, fy);
		agg::render_scanlines(fman->gray8_adaptor(),
		    fman->gray8_scanline(), ren_aa);
		fx += glyph->advance_x;
		fy += glyph->advance_y;
	    }
	}
    }

    if (tmpRgn)
	SdlTkRgnPoolFree(tmpRgn);

    /* Unlock surface */
    if (SDL_MUSTLOCK(sdl)) {
	SDL_UnlockSurface(sdl);
    }
}

template<class PixelFormat>
void
doDrawStringGray(Drawable d, GC gc, int x, int y, const char *string,
		 int length)
{
    SDL_Surface *sdl;
    int xOff = 0, yOff = 0;
    long i;
    _Font *_f = (_Font *) gc->font;
    double fx, fy;
    TkpClipMask *clipPtr = (TkpClipMask *) gc->clip_mask;
    REGION *rgn = 0;
    Region tmpRgn = 0;

    if (IS_WINDOW(d)) {
	rgn = SdlTkGetVisibleRegion((_Window *) d);

	/* Window is unmapped or totally obscured */
	if (XEmptyRegion(rgn))
	    return;
    }

    sdl = SdlTkGetDrawableSurface(d, &xOff, &yOff, NULL);

    /* Lock surface */
    if (SDL_MUSTLOCK(sdl)) {
	if (SDL_LockSurface(sdl) < 0) {
	    return;
	}
    }

    /* Rendering buffer, points to SDL_Surface memory */
    agg::rendering_buffer rbuf((agg::int8u *) sdl->pixels, sdl->w, sdl->h,
	sdl->pitch);

    /* Pixel-format renderer, a low-level pixel-rendering object */
    PixelFormat ren_pixf(rbuf);

    /* A basic renderer that does clipping to multiple boxes */
    typedef agg::renderer_mclip<PixelFormat> t_renderer_mclip;
    t_renderer_mclip ren_mclip(ren_pixf);

    /* The color, in the format used by the pixel-format renderer */
    Uint8 r, g, b;
    SDL_GetRGB(gc->foreground, SdlTk_sdlsurf->format, &r, &g, &b);
    agg::rgba8 c(r, g, b);

    fx = xOff + x;
    fy = yOff + y;

    /* agg::glyph_ren_agg_gray8 is BROKEN with MS Gothic japanese chars */
    (void) feng->load_font(_f->file, _f->index, agg::glyph_ren_native_gray8,
                           (const char *) SdlTkGetFTStream(_f->file));
    feng->flip_y(true);
    feng->height(_f->size);

    /* If the clipping region is specified, intersect it with the visible
     * region of the window, or if this is a pixmap then use the clipping
     * region unmodified. */
    if (clipPtr && clipPtr->type == TKP_CLIP_REGION) {
	Region clipRgn = (Region) clipPtr->value.region;
	if (rgn) {
	    tmpRgn = SdlTkRgnPoolGet();
	    XIntersectRegion(rgn, clipRgn, tmpRgn);
	    rgn = tmpRgn;
	} else {
	    rgn = clipRgn;
	}
    }

    if (rgn) {
	for (i = 0; i < rgn->numRects; i++) {
	    BoxPtr box = &rgn->rects[i];
	    ren_mclip.add_clip_box(xOff + box->x1, yOff + box->y1,
		xOff + box->x2 - 1, yOff + box->y2 - 1);
	}
    }

    length /= sizeof(unsigned int) /* FcChar32 */;

    typedef agg::renderer_scanline_aa_solid<t_renderer_mclip> t_renderer_scanline_aa_solid;
    t_renderer_scanline_aa_solid ren_aa(ren_mclip);
    ren_aa.color(c);

    for (i = 0; i < length; i++) {
        const agg::glyph_cache *glyph = fman->glyph(SdlTkGetNthGlyphIndex(_f, string, i));
	if (glyph) {
	    fman->init_embedded_adaptors(glyph, fx, fy);
	    agg::render_scanlines(fman->gray8_adaptor(),
		fman->gray8_scanline(), ren_aa);
	    fx += glyph->advance_x;
	    fy += glyph->advance_y;
	}
    }

    if (tmpRgn)
	SdlTkRgnPoolFree(tmpRgn);

    /* Unlock surface */
    if (SDL_MUSTLOCK(sdl)) {
	SDL_UnlockSurface(sdl);
    }
}

void
SdlTkGfxDrawString(Drawable d, GC gc, int x, int y, const char *string, int length)
{
    SDL_Surface *sdl;
    int format;

    sdl = SdlTkGetDrawableSurface(d, NULL, NULL, &format);

    switch (format) {
	case SDLTK_RGB565: doDrawString<agg::pixfmt_rgb565>(d, gc, x, y, string, length); break;
	case SDLTK_BGR565: doDrawString<agg::pixfmt_bgr565>(d, gc, x, y, string, length); break;
	case SDLTK_RGB24: doDrawString<agg::pixfmt_rgb24>(d, gc, x, y, string, length); break;
	case SDLTK_BGR24: doDrawString<agg::pixfmt_bgr24>(d, gc, x, y, string, length); break;
	case SDLTK_RGBA32: doDrawString<agg::pixfmt_rgba32>(d, gc, x, y, string, length); break;
	case SDLTK_ARGB32: doDrawString<agg::pixfmt_argb32>(d, gc, x, y, string, length); break;
	case SDLTK_BGRA32: doDrawString<agg::pixfmt_bgra32>(d, gc, x, y, string, length); break;
	case SDLTK_ABGR32: doDrawString<agg::pixfmt_abgr32>(d, gc, x, y, string, length); break;
	case SDLTK_GRAY8: doDrawStringGray<agg::pixfmt_gray8>(d, gc, x, y, string, length); break;
    }
}

template<class PixelFormat>
void
doFillArc(Drawable d, GC gc, int x, int y,
    unsigned int width, unsigned int height, int start, int extent)
{
    SDL_Surface *sdl;
    int xOff = 0, yOff = 0;
    long i;
    REGION *rgn = 0;

    if (IS_WINDOW(d)) {
	rgn = SdlTkGetVisibleRegion((_Window *) d);

	/* Window is unmapped or totally obscured */
	if (XEmptyRegion(rgn))
	    return;
    }

    sdl = SdlTkGetDrawableSurface(d, &xOff, &yOff, NULL);

    /* Lock surface */
    if (SDL_MUSTLOCK(sdl)) {
	if (SDL_LockSurface(sdl) < 0) {
	    return;
	}
    }

    /* Rendering buffer, points to SDL_Surface memory */
    agg::rendering_buffer rbuf((agg::int8u *) sdl->pixels, sdl->w, sdl->h,
	sdl->pitch);

    /* Pixel-format renderer, a low-level pixel-rendering object */
    PixelFormat ren_pixf(rbuf);

    /* A basic renderer that does clipping to multiple boxes */
    typedef agg::renderer_mclip<PixelFormat> t_renderer_mclip;
    t_renderer_mclip ren_mclip(ren_pixf);

    /* The color, in the format used by the pixel-format renderer */
    Uint8 r, g, b;
    SDL_GetRGB(gc->foreground, SdlTk_sdlsurf->format, &r, &g, &b);
    agg::rgba8 c(r, g, b);

    /* Apparently agg::arc is deprecated */
    agg::bezier_arc arc(xOff + x + width / 2.0, yOff + y + height / 2.0,
	width / 2.0, height / 2.0,
	agg::deg2rad(start / 64), agg::deg2rad(extent / 64),
        gc->arc_mode == ArcPieSlice ? true : false);
    typedef agg::conv_curve<agg::bezier_arc, agg::curve3_div, agg::curve4_div> t_conv_curve;
    t_conv_curve curve(arc);

    /* Thing that generates scanlines */
    agg::rasterizer_scanline_aa<> rasterizer;
    rasterizer.reset();
    rasterizer.add_path(curve);

    /* Scanline needed by the rasterizer -> renderer */
    agg::scanline_u8 scanline;

    if (rgn) {
	for (i = 0; i < rgn->numRects; i++) {
	    BoxPtr box = &rgn->rects[i];
	    ren_mclip.add_clip_box(xOff + box->x1, yOff + box->y1,
		xOff + box->x2 - 1, yOff + box->y2 - 1);
	}
    }

    /* FIXME: FillOpaqueStippled not implemented */
    if ((gc->fill_style == FillStippled
	    || gc->fill_style == FillOpaqueStippled)
	    && gc->stipple != None) {

	_Pixmap *stipple = (_Pixmap *) gc->stipple;

	/* Rendering buffer that points to the bitmap */
	agg::rendering_buffer stipple_buf((agg::int8u *) stipple->sdl->pixels,
	    stipple->sdl->w, stipple->sdl->h, stipple->sdl->pitch);

	/* A span allocator holds 1 line of pixels */
	agg::span_allocator<agg::rgba8> span_allocator;

	/* Generates spans (lines of pixels) from a source buffer */
	typedef span_stipple<agg::rgba8, agg::order_argb,
	    agg::wrap_mode_repeat, agg::wrap_mode_repeat> t_span_stipple;
	t_span_stipple span_stipple(span_allocator, stipple_buf, gc->ts_x_origin, gc->ts_y_origin);
	span_stipple.color(c);

	typedef agg::renderer_scanline_aa<t_renderer_mclip, t_span_stipple> t_renderer_scanline_aa;
	t_renderer_scanline_aa ren_scanline_aa(ren_mclip, span_stipple);

	agg::render_scanlines(rasterizer, scanline, ren_scanline_aa);
    } else {
	typedef agg::renderer_scanline_aa_solid<t_renderer_mclip> t_renderer_scanline_aa_solid;
	t_renderer_scanline_aa_solid ren_scanline(ren_mclip);
	ren_scanline.color(c);

	agg::render_scanlines(rasterizer, scanline, ren_scanline);
    }

    /* Unlock surface */
    if (SDL_MUSTLOCK(sdl)) {
	SDL_UnlockSurface(sdl);
    }
}

void
SdlTkGfxFillArc(Drawable d, GC gc, int x, int y,
    unsigned int width, unsigned int height, int start, int extent)
{
    SDL_Surface *sdl;
    int format;

    sdl = SdlTkGetDrawableSurface(d, NULL, NULL, &format);
    start = -start;
    extent = -extent;

    switch (format) {
	case SDLTK_RGB565: doFillArc<agg::pixfmt_rgb565>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_BGR565: doFillArc<agg::pixfmt_bgr565>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_RGB24: doFillArc<agg::pixfmt_rgb24>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_BGR24: doFillArc<agg::pixfmt_bgr24>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_RGBA32: doFillArc<agg::pixfmt_rgba32>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_ARGB32: doFillArc<agg::pixfmt_argb32>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_BGRA32: doFillArc<agg::pixfmt_bgra32>(d, gc, x, y, width, height, start, extent); break;
	case SDLTK_ABGR32: doFillArc<agg::pixfmt_abgr32>(d, gc, x, y, width, height, start, extent); break;
    }
}

template<class PixelFormat>
void
doFillPolygon(Drawable d, GC gc, XPoint *points, int npoints, int shape, int mode)
{
    SDL_Surface *sdl;
    int xOff = 0, yOff = 0;
    long i;
    REGION *rgn = 0;

    if (IS_WINDOW(d)) {
	rgn = SdlTkGetVisibleRegion((_Window *) d);

	/* Window is unmapped or totally obscured */
	if (XEmptyRegion(rgn))
	    return;
    }

    sdl = SdlTkGetDrawableSurface(d, &xOff, &yOff, NULL);

    /* Lock surface */
    if (SDL_MUSTLOCK(sdl)) {
	if (SDL_LockSurface(sdl) < 0) {
	    return;
	}
    }

    /* Rendering buffer, points to SDL_Surface memory */
    agg::rendering_buffer rbuf((agg::int8u *) sdl->pixels, sdl->w, sdl->h,
	sdl->pitch);

    /* Pixel-format renderer, a low-level pixel-rendering object */
    PixelFormat ren_pixf(rbuf);

    /* A basic renderer that does clipping to multiple boxes */
    typedef agg::renderer_mclip<PixelFormat> t_renderer_mclip;
    t_renderer_mclip ren_mclip(ren_pixf);

    /* The color, in the format used by the pixel-format renderer */
    Uint8 r, g, b;
    SDL_GetRGB(gc->foreground, SdlTk_sdlsurf->format, &r, &g, &b);
    agg::rgba8 c(r, g, b);

    /* Thing that creates scanlines for a filled polygon (with aa) */
    agg::rasterizer_scanline_aa<> rasterizer;
    rasterizer.reset();
#if 1
    VertexSource_XPoints vertexSrc(points, npoints, xOff, yOff);
    rasterizer.add_path(vertexSrc);
#else
    rasterizer.move_to((xOff + points[0].x) << 8, (yOff + points[0].y) << 8);
    for (i = 1; i < npoints; i++)
	rasterizer.line_to((xOff + points[i].x) << 8, (yOff + points[i].y) << 8);
#endif
    /* Scanline needed by the rasterizer -> renderer */
    agg::scanline_u8 scanline;

    if (rgn) {
	for (i = 0; i < rgn->numRects; i++) {
	    BoxPtr box = &rgn->rects[i];
	    ren_mclip.add_clip_box(xOff + box->x1, yOff + box->y1,
		xOff + box->x2 - 1, yOff + box->y2 - 1);
	}
    }

    /* FIXME: FillOpaqueStippled not implemented */
    if ((gc->fill_style == FillStippled
	    || gc->fill_style == FillOpaqueStippled)
	    && gc->stipple != None) {

	_Pixmap *stipple = (_Pixmap *) gc->stipple;

	/* Rendering buffer that points to the bitmap */
	agg::rendering_buffer stipple_buf((agg::int8u *) stipple->sdl->pixels,
	    stipple->sdl->w, stipple->sdl->h, stipple->sdl->pitch);

	/* A span allocator holds 1 line of pixels */
	agg::span_allocator<agg::rgba8> span_allocator;

	/* Generates spans (lines of pixels) from a source buffer */
	typedef span_stipple<agg::rgba8, agg::order_argb,
	    agg::wrap_mode_repeat, agg::wrap_mode_repeat> t_span_stipple;
	t_span_stipple span_stipple(span_allocator, stipple_buf, gc->ts_x_origin, gc->ts_y_origin);
	span_stipple.color(c);

	typedef agg::renderer_scanline_aa<t_renderer_mclip, t_span_stipple> t_renderer_scanline_aa;
	t_renderer_scanline_aa ren_scanline_aa(ren_mclip, span_stipple);

	agg::render_scanlines(rasterizer, scanline, ren_scanline_aa);
    } else {
	/* Thing that renders the scanlines */
	typedef agg::renderer_scanline_aa_solid<t_renderer_mclip> t_renderer_scanline_aa_solid;
	t_renderer_scanline_aa_solid ren_scanline(ren_mclip);
	ren_scanline.color(c);

	/* Thing that calculates the scanlines that make up the shape */
	agg::render_scanlines(rasterizer, scanline, ren_scanline);
    }

    /* Unlock surface */
    if (SDL_MUSTLOCK(sdl)) {
	SDL_UnlockSurface(sdl);
    }
}

void
SdlTkGfxFillPolygon(Drawable d, GC gc, XPoint *points, int npoints, int shape, int mode)
{
    SDL_Surface *sdl;
    int format;

    sdl = SdlTkGetDrawableSurface(d, NULL, NULL, &format);

    switch (format) {
	case SDLTK_RGB565: doFillPolygon<agg::pixfmt_rgb565>(d, gc, points, npoints, shape, mode); break;
	case SDLTK_BGR565: doFillPolygon<agg::pixfmt_bgr565>(d, gc, points, npoints, shape, mode); break;
	case SDLTK_RGB24: doFillPolygon<agg::pixfmt_rgb24>(d, gc, points, npoints, shape, mode); break;
	case SDLTK_BGR24: doFillPolygon<agg::pixfmt_bgr24>(d, gc, points, npoints, shape, mode); break;
	case SDLTK_RGBA32: doFillPolygon<agg::pixfmt_rgba32>(d, gc, points, npoints, shape, mode); break;
	case SDLTK_ARGB32: doFillPolygon<agg::pixfmt_argb32>(d, gc, points, npoints, shape, mode); break;
	case SDLTK_BGRA32: doFillPolygon<agg::pixfmt_bgra32>(d, gc, points, npoints, shape, mode); break;
	case SDLTK_ABGR32: doFillPolygon<agg::pixfmt_abgr32>(d, gc, points, npoints, shape, mode); break;
    }
}

template<class PixelFormat>
void
doFillRect(Drawable d, GC gc, int x, int y, int w, int h)
{
    SDL_Surface *sdl;
    int xOff = 0, yOff = 0;
    long i;
    TkpClipMask *clipPtr = (TkpClipMask*)gc->clip_mask;
    REGION *rgn = 0;
    Region tmpRgn = 0;

    if (IS_WINDOW(d)) {
	rgn = SdlTkGetVisibleRegion((_Window *) d);

	/* Window is unmapped or totally obscured */
	if (XEmptyRegion(rgn))
	    return;
    }

    sdl = SdlTkGetDrawableSurface(d, &xOff, &yOff, NULL);
    x += xOff;
    y += yOff;

    /* Lock surface */
    if (SDL_MUSTLOCK(sdl)) {
	if (SDL_LockSurface(sdl) < 0) {
	    return;
	}
    }

    /* Rendering buffer, points to SDL_Surface memory */
    agg::rendering_buffer rbuf((agg::int8u *) sdl->pixels, sdl->w, sdl->h,
	sdl->pitch);

    /* Pixel-format renderer, a low-level pixel-rendering object */
    PixelFormat ren_pixf(rbuf);

    /* A basic renderer that does clipping to multiple boxes */
    typedef agg::renderer_mclip<PixelFormat> t_renderer_mclip;
    t_renderer_mclip ren_mclip(ren_pixf);

    /* The color, in the format used by the pixel-format renderer */
    Uint8 r, g, b;
    SDL_GetRGB(gc->foreground, SdlTk_sdlsurf->format, &r, &g, &b);
    agg::rgba8 c(r, g, b);

    /* If the clipping region is specified, intersect it with the visible
     * region of the window, or if this is a pixmap then use the clipping
     * region unmodified. */
    if (clipPtr && clipPtr->type == TKP_CLIP_REGION) {
	Region clipRgn = (Region) clipPtr->value.region;
	if (rgn) {
	    tmpRgn = SdlTkRgnPoolGet();
	    XIntersectRegion(rgn, clipRgn, tmpRgn);
	    rgn = tmpRgn;
	} else {
	    rgn = clipRgn;
	}
    }

    if (rgn) {
	for (i = 0; i < rgn->numRects; i++) {
	    BoxPtr box = &rgn->rects[i];
	    ren_mclip.add_clip_box(xOff + box->x1, yOff + box->y1,
		xOff + box->x2 - 1, yOff + box->y2 - 1);
	}
    }

    /* FIXME: FillOpaqueStippled not implemented */
    if ((gc->fill_style == FillStippled
	    || gc->fill_style == FillOpaqueStippled)
	    && gc->stipple != None) {

	_Pixmap *stipple = (_Pixmap *) gc->stipple;

	/* Rendering buffer that points to the bitmap */
	agg::rendering_buffer stipple_buf((agg::int8u *) stipple->sdl->pixels,
	    stipple->sdl->w, stipple->sdl->h, stipple->sdl->pitch);
#if 1
	agg::wrap_mode_repeat wrap_x(stipple_buf.width());
	agg::wrap_mode_repeat wrap_y(stipple_buf.height());
	unsigned wy = wrap_y(y - gc->ts_y_origin);
	while (h--) {
	    agg::int8u *row_ptr = stipple_buf.row(wy);
	    unsigned x1 = x;
	    unsigned wx = wrap_x(x - gc->ts_x_origin);
	    unsigned w1 = w;
	    while (w1--) {
		agg::int8u *p = row_ptr + wx;
		if (!p[0]) {
		    ren_mclip.copy_pixel(x1, y, c);
		}
		wx = ++wrap_x;
		++x1;
	    }
	    wy = ++wrap_y;
	    ++y;
	}
#else
	/* A span allocator holds 1 line of pixels */
	agg::span_allocator<agg::rgba8> span_allocator;

	/* Generates spans (lines of pixels) from a source buffer */
	typedef span_stipple<agg::rgba8, agg::order_argb,
	    agg::wrap_mode_repeat, agg::wrap_mode_repeat> t_span_stipple;
	t_span_stipple span_stipple(span_allocator, stipple_buf, gc->ts_x_origin, gc->ts_y_origin);
	span_stipple.color(c);


	typedef agg::renderer_scanline_aa<t_renderer_mclip, t_span_stipple> t_renderer_scanline_aa;
	t_renderer_scanline_aa ren_scanline_aa(ren_mclip, span_stipple);

	VertexSource_XRectangle vertexSrc(x, y, w, h);
	agg::rasterizer_scanline_aa<> rasterizer;
	rasterizer.reset();
	rasterizer.add_path(vertexSrc);

	/* Scanline needed by the rasterizer -> renderer */
	agg::scanline_u8 scanline;

	render_scanlines(rasterizer, scanline, ren_scanline_aa);
#endif
    } else {
	ren_mclip.copy_bar(x, y, x + w - 1, y + h - 1, c);
    }

    if (tmpRgn)
	SdlTkRgnPoolFree(tmpRgn);

    /* Unlock surface */
    if (SDL_MUSTLOCK(sdl)) {
	SDL_UnlockSurface(sdl);
    }
}

void
SdlTkGfxFillRect(Drawable d, GC gc, int x, int y, int w, int h)
{
    SDL_Surface *sdl;
    int format;

    sdl = SdlTkGetDrawableSurface(d, NULL, NULL, &format);

    if (gc->function == GXinvert) {
	switch (sdl->format->BitsPerPixel) {
	    case 16: doFillRect<pixfmt_1_2_4Bpp_xor<Uint16> >(d, gc, x, y, w, h); break;
	    case 24: doFillRect<pixfmt_3Bpp_xor>(d, gc, x, y, w, h); break;
	    case 32: doFillRect<pixfmt_1_2_4Bpp_xor<Uint32> >(d, gc, x, y, w, h); break;
	}
	return;
    }

    switch (format) {
	case SDLTK_RGB565: doFillRect<agg::pixfmt_rgb565>(d, gc, x, y, w, h); break;
	case SDLTK_BGR565: doFillRect<agg::pixfmt_bgr565>(d, gc, x, y, w, h); break;
	case SDLTK_RGB24: doFillRect<agg::pixfmt_rgb24>(d, gc, x, y, w, h); break;
	case SDLTK_BGR24: doFillRect<agg::pixfmt_bgr24>(d, gc, x, y, w, h); break;
	case SDLTK_RGBA32: doFillRect<agg::pixfmt_rgba32>(d, gc, x, y, w, h); break;
	case SDLTK_ARGB32: doFillRect<agg::pixfmt_argb32>(d, gc, x, y, w, h); break;
	case SDLTK_BGRA32: doFillRect<agg::pixfmt_bgra32>(d, gc, x, y, w, h); break;
	case SDLTK_ABGR32: doFillRect<agg::pixfmt_abgr32>(d, gc, x, y, w, h); break;
	case SDLTK_GRAY8: doFillRect<agg::pixfmt_gray8>(d, gc, x, y, w, h); break;
    }
}

