#include <X11/Xlib.h>
#include "tkInt.h"
#include "default.h"
#include "SdlTkInt.h"

enum {
    DEC_BUTTON_NONE,
    DEC_BUTTON_CLOSE,
    DEC_BUTTON_MAXIMIZE,
    DEC_BUTTON_MINIMIZE,
    DEC_NUM_BUTTONS = DEC_BUTTON_MINIMIZE
};

enum {
    DEC_HIT_NONE,
    DEC_HIT_TITLE,
    DEC_HIT_NW,
    DEC_HIT_N,
    DEC_HIT_NE,
    DEC_HIT_W,
    DEC_HIT_E,
    DEC_HIT_SW,
    DEC_HIT_S,
    DEC_HIT_SE,
    DEC_HIT_BUTTON
};

enum {
    DEC_TRACK_NONE,
    DEC_TRACK_BUTTON,
    DEC_TRACK_DRAG,
    DEC_TRACK_RESIZE
};

/* Info about decorative frame window which has titlebar, min/max/close
 * buttons */
struct _DecFrame
{
    Tk_3DBorder border;
    int draw; /* TRUE if frame needs redrawing */
    int button; /* DEC_BUTTON_XXX mouse is over */
    int pressed; /* TRUE if mouse is down in a button */
    int active; /* TRUE if frame should be drawn active */
};

int SdlTkX_drag_lock = 0;

static XFontStruct *titlebar_font_struct = NULL;

static _Window *track_toplevel = NULL;
static int track_action = DEC_TRACK_NONE;
static int track_button = DEC_BUTTON_NONE;
static int track_edge = DEC_HIT_NONE;
static int track_x0 = 0;
static int track_y0 = 0;

static _Window *motion_toplevel = NULL;
static int motion_button = DEC_BUTTON_NONE;

static void
GetButtonBounds(_Window *_w, int button, int *x, int *y, int *w, int *h)
{
    int buttonSize = DEC_TITLE_HEIGHT - 6;

    *x = _w->atts.width - (5 + buttonSize) * button;
    *y = 3;
    *w = *h = buttonSize;
}

static int
HitTestFrame(_Window *_w, int x, int y, int *button)
{
    int i;
    int n = 0, s = 0, w = 0, e = 0;
    int buttonX, buttonY, buttonW, buttonH;

    *button = DEC_BUTTON_NONE;

    for (i = 1; i <= DEC_NUM_BUTTONS; i++) {
	GetButtonBounds(_w, i, &buttonX, &buttonY, &buttonW, &buttonH);
	if (x >= buttonX && x < buttonX + buttonW &&
	    y >= buttonY && y < buttonY + buttonH) {
	    *button = i;
	    return DEC_HIT_BUTTON;
	}
    }

    if (y < 4) {
	n = 1;
    }
    if (y >= _w->atts.height - 4) {
	s = 1;
    }
    if (x < 4) {
	w = 1;
    }
    if (x >= _w->atts.width - 4) {
	e = 1;
    }

    if (n && w) {
        return DEC_HIT_NW;
    }
    if (n && e) {
        return DEC_HIT_NE;
    }
    if (s && w) {
        return DEC_HIT_SW;
    }
    if (s && e) {
        return DEC_HIT_SE;
    }
    if (n) {
        return DEC_HIT_N;
    }
    if (s) {
        return DEC_HIT_S;
    }
    if (w) {
        return DEC_HIT_W;
    }
    if (e) {
        return DEC_HIT_E;
    }

    if (y < DEC_TITLE_HEIGHT) {
	return DEC_HIT_TITLE;
    }

    return DEC_HIT_NONE;
}

static void
EnterButton(_Window *_w, int button)
{
    if (track_action == DEC_TRACK_BUTTON &&
	(track_toplevel != _w || track_button != button)) {
	return;
    }
    _w->dec->button = button;
    _w->dec->pressed = track_action == DEC_TRACK_BUTTON;

    _w->dec->draw = 1;
    SdlTkScreenChanged();
}

static void
LeaveButton(_Window *_w, int button)
{
    if (track_action == DEC_TRACK_BUTTON &&
	(track_toplevel != _w || track_button != button)) {
	return;
    }
    _w->dec->button = DEC_BUTTON_NONE;
    _w->dec->pressed = False;

    _w->dec->draw = 1;
    SdlTkScreenChanged();
}

/* Handle mouse event in a decframe */
int
SdlTkDecFrameEvent(_Window *_w, SDL_Event *sdl_event, int x, int y)
{
    int hit, button, dummy;

    switch (sdl_event->type) {
	case SDL_MOUSEBUTTONDOWN:
	    if (sdl_event->button.button != SDL_BUTTON_LEFT) {
		break;
	    }
	    if (_w->dec == NULL) {
		break;
	    }
	    if (!SdlTkGrabCheck(_w, &dummy)) {
		break;
	    }
	    hit = HitTestFrame(_w, x - _w->atts.x, y - _w->atts.y, &button);
	    switch (hit) {
		case DEC_HIT_BUTTON:
		    track_toplevel = _w;
		    track_action = DEC_TRACK_BUTTON;
		    track_button = button;
		    EnterButton(_w, button); /* redraw pressed */
		    break;
		case DEC_HIT_TITLE:
		    track_toplevel = _w;
		    track_action = DEC_TRACK_DRAG;
		    track_x0 = x;
		    track_y0 = y;
		    break;
		default:
		    track_toplevel = _w;
		    track_action = DEC_TRACK_RESIZE;
		    track_edge = hit;
		    track_x0 = x;
		    track_y0 = y;
		    break;
	    }
	    return 1;

	case SDL_MOUSEBUTTONUP:
	  if (sdl_event->button.button != SDL_BUTTON_LEFT) {
		break;
	  }
	    if (track_action != DEC_TRACK_NONE) {
		switch (track_action) {
		    case DEC_TRACK_BUTTON:
			if (motion_toplevel == _w &&
			    track_toplevel == _w &&
			    track_button == motion_button) {
			    if (track_button == DEC_BUTTON_CLOSE) {
				XEvent event;
				TkWindow *tkwin = _w->child->tkwin;
				event.type = ClientMessage;
				event.xclient.serial = tkwin->display->request++;
				event.xclient.send_event = False;
				event.xclient.display = tkwin->display;
				event.xclient.window = tkwin->window;
				event.xclient.message_type =
				    Tk_InternAtom((Tk_Window) tkwin, "WM_PROTOCOLS");
				event.xclient.format = 32;
				event.xclient.data.l[0] =
				    Tk_InternAtom((Tk_Window) tkwin, "WM_DELETE_WINDOW");
				Tk_QueueWindowEvent(&event, TCL_QUEUE_TAIL);
			    }

			    /* redraw not pressed */
			    _w->dec->pressed = 0;
			    _w->dec->draw = 1;
			    SdlTkScreenChanged();
			} else if (motion_button != DEC_BUTTON_NONE) {
			    motion_toplevel->dec->draw = 1;
			    motion_toplevel->dec->button = motion_button;
			    SdlTkScreenChanged();
			}
			break;
		}
		track_toplevel = NULL;
		track_action = DEC_TRACK_NONE;
		return 1;
	    }
	    break;

	case SDL_MOUSEMOTION:
	{
	    int dx, dy;
/*	    if (!(sdl_event->motion.state & SDL_BUTTON(SDL_BUTTON_LEFT)))
		break;*/
	    dx = sdl_event->motion.xrel;
	    dy = sdl_event->motion.yrel;
	    if (track_action == DEC_TRACK_DRAG) {
		/*
 		 * Note: dragging the wrapper to the new position
		 * of the decframe
		 */
drag:
		SdlTkX_drag_lock = 1;
		XMoveWindow(SdlTkX_display,
			    (Window) track_toplevel->child,
			    track_toplevel->atts.x + dx,
			    track_toplevel->atts.y + dy);
		SdlTkX_drag_lock = 0;
		return 1;
	    }
	    if (track_action == DEC_TRACK_RESIZE) {
		int dw, dh;

		if (track_toplevel->child->size.flags & PResizeInc) {
		    dw = track_toplevel->child->size.width_inc;
		    if (dw > 0) {
			dx = x - track_x0;
			dx = dx / dw;
			dx *= dw;
		    }
		    dh = track_toplevel->child->size.height_inc;
		    if (dh > 0) {
			dy = y - track_y0;
			dy = dy / dh;
			dy *= dh;
		    }
		}
		if ((track_toplevel->child->size.flags & PMinSize) &&
		    (track_toplevel->child->size.flags & PMaxSize) &&
		    track_toplevel->child->size.min_width ==
		    track_toplevel->child->size.max_width &&
		    track_toplevel->child->size.min_height ==
		    track_toplevel->child->size.max_height) {
		    goto drag;
		}
		dw = dh = 0;
		switch (track_edge) {
		    case DEC_HIT_NW:
		    case DEC_HIT_SW:
		    case DEC_HIT_W:
			dw = track_toplevel->child->atts.width - dx;
			break;
		    case DEC_HIT_NE:
		    case DEC_HIT_SE:
		    case DEC_HIT_E:
			dw = track_toplevel->child->atts.width + dx;
			break;
		}
		if (track_toplevel->child->size.flags & PMinSize) {
		    if (dw < track_toplevel->child->size.min_width) {
			dx = 0;
		    }
		}
		if (track_toplevel->child->size.flags & PMaxSize) {
		    if (dw > track_toplevel->child->size.max_width) {
			dx = 0;
		    }
		}
		switch (track_edge) {
		    case DEC_HIT_NW:
		    case DEC_HIT_NE:
		    case DEC_HIT_N:
			dh = track_toplevel->child->atts.height - dy;
			break;
		    case DEC_HIT_SW:
		    case DEC_HIT_SE:
		    case DEC_HIT_S:
			dh = track_toplevel->child->atts.height + dy;
			break;
		}
		if (track_toplevel->child->size.flags & PMinSize) {
		    if (dh < track_toplevel->child->size.min_height) {
			dy = 0;
		    }
		}
		if (track_toplevel->child->size.flags & PMaxSize) {
		    if (dh > track_toplevel->child->size.max_height) {
			dy = 0;
		    }
		}
		if (dx == 0 && dy == 0) {
		    return 1;
		}
		switch (track_edge) {
		    case DEC_HIT_NW:
			XMoveResizeWindow(SdlTkX_display,
					  (Window) track_toplevel->child,
					  track_toplevel->atts.x + dx,
					  track_toplevel->atts.y + dy,
					  track_toplevel->child->atts.width - dx,
					  track_toplevel->child->atts.height - dy);
			break;
		    case DEC_HIT_NE:
			XMoveResizeWindow(SdlTkX_display,
					  (Window) track_toplevel->child,
					  track_toplevel->atts.x,
					  track_toplevel->atts.y + dy,
					  track_toplevel->child->atts.width + dx,
					  track_toplevel->child->atts.height - dy);
			break;
		    case DEC_HIT_SW:
			XMoveResizeWindow(SdlTkX_display,
					  (Window) track_toplevel->child,
					  track_toplevel->atts.x + dx,
					  track_toplevel->atts.y,
					  track_toplevel->child->atts.width - dx,
					  track_toplevel->child->atts.height + dy);
			break;
		    case DEC_HIT_SE:
		        XResizeWindow(SdlTkX_display,
				      (Window) track_toplevel->child,
				      track_toplevel->child->atts.width + dx,
				      track_toplevel->child->atts.height + dy);
			break;
		    case DEC_HIT_N:
		        XMoveResizeWindow(SdlTkX_display,
					  (Window) track_toplevel->child,
					  track_toplevel->atts.x,
					  track_toplevel->atts.y + dy,
					  track_toplevel->child->atts.width,
					  track_toplevel->child->atts.height - dy);
			break;
		    case DEC_HIT_S:
		        XResizeWindow(SdlTkX_display,
				      (Window) track_toplevel->child,
				      track_toplevel->child->atts.width,
				      track_toplevel->child->atts.height + dy);
			break;
		    case DEC_HIT_W:
		        XMoveResizeWindow(SdlTkX_display,
					  (Window) track_toplevel->child,
					  track_toplevel->atts.x + dx,
					  track_toplevel->atts.y,
					  track_toplevel->child->atts.width - dx,
					  track_toplevel->child->atts.height);
			break;
		    case DEC_HIT_E:
			XResizeWindow(SdlTkX_display,
				      (Window) track_toplevel->child,
				      track_toplevel->child->atts.width + dx,
				      track_toplevel->child->atts.height);
			break;
		}
		track_x0 = x;
		track_y0 = y;
		return 1;
	    }

	    /* If the pointer is in the root, leave any button */
	    if (_w->dec == NULL) {
	        if (motion_button) {
		    LeaveButton(motion_toplevel, motion_button);
		}
		motion_toplevel = NULL;
		motion_button = DEC_BUTTON_NONE;

		/* Hide the event from Tk if mousebutton 1 is pressed */
		return (track_action != DEC_TRACK_NONE);
	    }

	    hit = HitTestFrame(_w, x - _w->atts.x, y - _w->atts.y, &button);
	    if (button != motion_button || _w != motion_toplevel) {
	        if (motion_button) {
		    LeaveButton(motion_toplevel, motion_button);
		}
		if (SdlTkGrabCheck(_w, &dummy)) {
		    motion_toplevel = _w;
		    motion_button = button;
		} else {
		    motion_toplevel = NULL;
		    motion_button = DEC_BUTTON_NONE;
		}
		if (motion_button) {
		    EnterButton(motion_toplevel, motion_button);
		}
	    }

	    /* Hide the event from Tk if mousebutton 1 is pressed */
	    return (track_action != DEC_TRACK_NONE);
	}
	case SDL_QUIT:
	{
	    TkMainInfo *info = TkGetMainInfoList();
	    TkWindow *tkwin;
	    int evsent = 0;

	    while (info != NULL) {
		tkwin = info->winPtr;
		if (tkwin != NULL) {
		    XEvent event;
		    event.type = ClientMessage;
		    event.xclient.serial = tkwin->display->request++;
		    event.xclient.send_event = False;
		    event.xclient.display = tkwin->display;
		    event.xclient.window = tkwin->window;
		    event.xclient.message_type =
		    Tk_InternAtom((Tk_Window) tkwin, "WM_PROTOCOLS");
		    event.xclient.format = 32;
		    event.xclient.data.l[0] =
		    Tk_InternAtom((Tk_Window) tkwin, "WM_DELETE_WINDOW");
		    Tk_QueueWindowEvent(&event, TCL_QUEUE_TAIL);
		    evsent++;
		}
		info = info->nextPtr;
	    }
	    if (evsent > 0) {
		return 1;
	    }
	    break;
	}
    }
    return 0;
}

void
SdlTkDecDrawFrame(_Window *_w)
{
    Drawable d = (Drawable) _w;
    const char *title = NULL;
    Tk_Window tkwin;
    Tk_3DBorder border;
    GC bgGC, lightGC, darkGC;
    int w, h;
    unsigned long savePixel;
    Uint32 titlePixel;
    int buttonX, buttonY, buttonW, buttonH;

    if (_w->dec == NULL) {
        Tcl_Panic("SdlTkRedrawFrame: not a decorative frame");
    }
    if (_w->child == NULL || _w->child->tkwin == NULL) {
	return;
    }

    /* decframe's don't have a TkWindow, get it from the child */
    tkwin = (Tk_Window) _w->child->tkwin;

    if (_w->dec->border == NULL) {
	/* Could be customized for each toplevel */
	/* Free it when dec is destroyed */
	_w->dec->border = Tk_Get3DBorder(NULL, tkwin, /*"#42a0ff"*/ NORMAL_BG);
    }
    border = _w->dec->border;
    bgGC = Tk_3DBorderGC(tkwin, border, TK_3D_FLAT_GC);
    lightGC = Tk_3DBorderGC(tkwin, border, TK_3D_LIGHT_GC);
    darkGC = Tk_3DBorderGC(tkwin, border, TK_3D_DARK_GC);

    w = _w->atts.width;
    h = _w->atts.height;

    /* top */
    SdlTkGfxFillRect(d, bgGC, 0, 0, w, DEC_TITLE_HEIGHT);
    /* bottom */
    SdlTkGfxFillRect(d, bgGC, 0, h - DEC_FRAME_WIDTH, w, DEC_FRAME_WIDTH);
    /* left */
    SdlTkGfxFillRect(d, bgGC, 0, 0, DEC_FRAME_WIDTH, h);
    /* right */
    SdlTkGfxFillRect(d, bgGC, w - DEC_FRAME_WIDTH, 0, DEC_FRAME_WIDTH, h);

    if (_w->dec->active) {
	unsigned long fg = darkGC->foreground;
	Uint8 r, g, b;

	SDL_GetRGB(fg, SdlTk_sdlsurf->format, &r, &g, &b);
	darkGC->foreground = SDL_MapRGB(SdlTk_sdlsurf->format,
					r | 0x10, g | 0x10, b | 0x10);
	titlePixel = darkGC->foreground;
	/* top */
	SdlTkGfxFillRect(d, darkGC, 0, 0, w, DEC_TITLE_HEIGHT - 1);
	/* bottom */
        SdlTkGfxFillRect(d, darkGC, 0, h - DEC_FRAME_WIDTH, w,
			 DEC_FRAME_WIDTH - 1);
	/* left */
        SdlTkGfxFillRect(d, darkGC, 0, 0, DEC_FRAME_WIDTH - 1, h);
	/* right */
        SdlTkGfxFillRect(d, darkGC, w - DEC_FRAME_WIDTH, 0,
			 DEC_FRAME_WIDTH + 2, h - 2);
	darkGC->foreground = fg;
    } else
	titlePixel = bgGC->foreground;

    /* XChangeProperty sets the title for the wrapper */
    title = _w->child->title;
    if (title != NULL) {
	XGCValues fakeGC;
	int lineHeight;
	Tcl_Encoding encoding;
	Tcl_DString ds;
	char buf[2]; buf[0] = 0x00; buf[1] = 'c';

	if (titlebar_font_struct == NULL) {
	    titlebar_font_struct = XLoadQueryFont(SdlTkX_display,
		"-*-arial-normal-r-*-*-12-*-*-*-*-*-*-*");
	}
	if (titlebar_font_struct != NULL) {
	    lineHeight = titlebar_font_struct->ascent +
		titlebar_font_struct->descent;

	    fakeGC.font = titlebar_font_struct->fid;
	    fakeGC.foreground = SdlTkX_screen->black_pixel;
	    fakeGC.clip_mask = None;
	    fakeGC.stipple = None;
	    fakeGC.fill_style = FillSolid;	

	    if (_w->dec->active) {
		fakeGC.foreground = lightGC->foreground;
	    }
	    encoding = Tcl_GetEncoding(NULL, "ucs-4");
	    Tcl_UtfToExternalDString(encoding, buf, 2, &ds);
	    Tcl_UtfToExternalDString(encoding, title, strlen(title), &ds);

	    SdlTkGfxDrawString(d, &fakeGC, DEC_FRAME_WIDTH * 2,
			       1 + (DEC_TITLE_HEIGHT - lineHeight) / 2 +
			       titlebar_font_struct->ascent,
			       Tcl_DStringValue(&ds), Tcl_DStringLength(&ds));

	    Tcl_FreeEncoding(encoding);
	    Tcl_DStringFree(&ds);
	}
    }

    /* Close box */
    {
        XPoint points[2];

	GetButtonBounds(_w, DEC_BUTTON_CLOSE, &buttonX, &buttonY,
			&buttonW, &buttonH);

	/* Button background (erase 1 pixel outside) */
	savePixel = bgGC->foreground;
	bgGC->foreground = titlePixel;
	SdlTkGfxFillRect(d, bgGC, buttonX - 1, buttonY - 1, buttonW + 2,
			 buttonH + 2);
	bgGC->foreground = savePixel;

	if (_w->dec->button == DEC_BUTTON_CLOSE) {
	    savePixel = bgGC->foreground;
	    bgGC->foreground = SDL_MapRGB(SdlTk_sdlsurf->format,
					  128 + 64 - _w->dec->pressed * 64,
					  0, 0);
	    SdlTkGfxFillRect(d, bgGC, buttonX, buttonY, buttonW, buttonH);
	    bgGC->foreground = savePixel;
	}

	SdlTkGfxDrawRect(d, lightGC, buttonX, buttonY, buttonW - 1,
			 buttonH - 1);

	lightGC->line_width = 2;

	points[0].x = buttonX + 3;
	points[0].y = buttonY + 3;
	points[1].x = buttonX + buttonW - 3;
	points[1].y = buttonY + buttonH - 3;
	SdlTkGfxDrawLines(d, lightGC, points, 2, CoordModeOrigin);

	points[0].x = buttonX + buttonW - 3;
	points[0].y = buttonY + 3;
	points[1].x = buttonX + 3;
	points[1].y = buttonY + buttonH - 3;
	SdlTkGfxDrawLines(d, lightGC, points, 2, CoordModeOrigin);

	lightGC->line_width = 1;
    }

#ifdef MIN_MAX_BOXES
    /* Maximize box */
    {
        GetButtonBounds(_w, DEC_BUTTON_MAXIMIZE, &buttonX, &buttonY,
			&buttonW, &buttonH);

	/* Button background (erase 1 pixel outside) */
	savePixel = bgGC->foreground;
	bgGC->foreground = titlePixel;
	SdlTkGfxFillRect(d, bgGC, buttonX - 1, buttonY - 1,
			 buttonW + 2, buttonH + 2);
	bgGC->foreground = savePixel;

	if (_w->dec->button == DEC_BUTTON_MAXIMIZE) {
	    savePixel = bgGC->foreground;
	    bgGC->foreground = SDL_MapRGB(SdlTk_sdlscreen->format, 0,
					  128 + 64 - _w->dec->pressed * 64, 0);
	    SdlTkGfxFillRect(d, bgGC, buttonX, buttonY, buttonW, buttonH);
	    bgGC->foreground = savePixel;
	}

	/* Outline */
	SdlTkGfxDrawRect(d, lightGC, buttonX, buttonY,
			 buttonW - 1, buttonH - 1);

	/* Symbol */
	SdlTkGfxDrawRect(d, lightGC, buttonX + 4, buttonY + 4,
			 buttonW - 9, buttonH - 9);
	SdlTkGfxFillRect(d, lightGC, buttonX + 4, buttonY + 5, buttonW - 8, 2);
    }

    /* Minimize box */
    {
        GetButtonBounds(_w, DEC_BUTTON_MINIMIZE, &buttonX, &buttonY,
			&buttonW, &buttonH);

	/* Button background (erase 1 pixel outside) */
	savePixel = bgGC->foreground;
	bgGC->foreground = titlePixel;
	SdlTkGfxFillRect(d, bgGC, buttonX - 1, buttonY - 1,
			 buttonW + 2, buttonH + 2);
	bgGC->foreground = savePixel;

	if (_w->dec->button == DEC_BUTTON_MINIMIZE) {
	    savePixel = bgGC->foreground;
	    bgGC->foreground = SDL_MapRGB(SdlTk_sdlscreen->format, 0, 0,
					  255 - _w->dec->pressed * 64);
	    SdlTkGfxFillRect(d, bgGC, buttonX, buttonY, buttonW, buttonH);
	    bgGC->foreground = savePixel;
	}

	/* Outline */
	SdlTkGfxDrawRect(d, lightGC, buttonX, buttonY,
			 buttonW - 1, buttonH - 1);

	/* Symbol */
	SdlTkGfxFillRect(d, lightGC, buttonX + 4,
			 buttonY + buttonH - 7, buttonW - 8, 3);
    }
#endif

    /* outer highlight */
    SdlTkGfxFillRect(d, darkGC, w - 1, 0, 1, h);
    SdlTkGfxFillRect(d, darkGC, 0, h - 1, w, 1);
    SdlTkGfxFillRect(d, lightGC, 0, 0, 1, h);
    SdlTkGfxFillRect(d, lightGC, 0, 0, w, 1);

    /* inner highlight */
    SdlTkGfxFillRect(d, darkGC, DEC_FRAME_WIDTH - 1, DEC_TITLE_HEIGHT - 1,
		     DEC_FRAME_WIDTH - 1,
		     h - DEC_TITLE_HEIGHT - DEC_FRAME_WIDTH + 2);
    SdlTkGfxFillRect(d, darkGC, DEC_FRAME_WIDTH - 1, DEC_TITLE_HEIGHT - 1,
		     w - DEC_FRAME_WIDTH * 2 + 2, 1);
    SdlTkGfxFillRect(d, lightGC, w - DEC_FRAME_WIDTH, DEC_TITLE_HEIGHT - 1,
		     1, h - DEC_TITLE_HEIGHT - DEC_FRAME_WIDTH + 2);
    SdlTkGfxFillRect(d, lightGC, DEC_FRAME_WIDTH - 1, h - DEC_FRAME_WIDTH,
		     w - DEC_FRAME_WIDTH * 2 + 2, 1);
}

int
SdlTkDecSetActive(_Window *_w, int active)
{
    if (active != -1) {
	_w->dec->active = active;
    }
    return _w->dec->active;
}

int
SdlTkDecSetDraw(_Window *_w, int draw)
{
    if (draw != -1) {
	_w->dec->draw = draw;
    }
    return _w->dec->draw;
}

void
SdlTkDecCreate(_Window *_w)
{
    _w->dec = (struct _DecFrame *) ckalloc(sizeof (struct _DecFrame));
    _w->dec->border = NULL;
    _w->dec->draw = 0;
    _w->dec->button = DEC_BUTTON_NONE;
    _w->dec->pressed = False;
    _w->dec->active = 0;
}

void
SdlTkDecDestroy(_Window *_w)
{
    if (_w->dec->border != NULL) {
	Tk_Free3DBorder(_w->dec->border);
    }
    ckfree((char *) _w->dec);
    _w->dec = NULL;

    if (track_toplevel == _w) {
	track_toplevel = NULL;
	track_action = DEC_TRACK_NONE;
    }
    if (motion_toplevel == _w) {
	motion_toplevel = NULL;
	motion_button = DEC_BUTTON_NONE;
    }
}
