#include "tkInt.h"
#include "SdlTkInt.h"

SDL_Window *SdlTk_sdlscreen = NULL;
SDL_Surface *SdlTk_sdlsurf = NULL;
SDL_Renderer *SdlTk_sdlrend = NULL;
SDL_Texture *SdlTk_sdltex = NULL;
_Window *SdlTkX_root = NULL;
Display *SdlTkX_display = NULL;
Screen *SdlTkX_screen = NULL;
Window SdlTkX_focus_window = None;
Window SdlTkX_focus_window_old = None;
Window SdlTkX_focus_window_not_override = None;
#ifdef ANDROID
int SdlTk_sdlfocus = 1;
#else
int SdlTk_sdlfocus = 0;
#endif

static Region screen_dirty_region = NULL;
static Region screen_update_region = NULL;
static TkWindow *tk_pointergrab_window = NULL;
static TkWindow *tk_capture_window = NULL;
static _Window *tk_mouse_window = NULL;
static int tk_mouse_x = 0;
static int tk_mouse_y = 0;
static int drawLater = 0;

/*
 *----------------------------------------------------------------------
 *
 * SdlTkSendAppEvent --
 *
 *	Send virtual application events to all toplevel windows
 *	recursively.
 *
 * Results:
 *	First window to receive application event.
 *
 *----------------------------------------------------------------------
 */

static _Window *
SdlTkSendAppEvent(XEvent *event, int *sentp, _Window *_w)
{
    _Window *result = NULL;

    while (_w != NULL) {
	if (_w->tkwin != NULL && (_w->tkwin->flags & TK_APP_TOP_LEVEL)) {
	    *sentp += 1;
	    if (*sentp == 1) {
		result = _w;
	    } else {
		event->xany.window = (Window) _w;
		Tk_QueueWindowEvent(event, TCL_QUEUE_TAIL);
	    }
	}
	if (_w->child != NULL) {
	    _Window *tmp = SdlTkSendAppEvent(event, sentp, _w->child);

	    if (result == NULL && tmp != NULL) {
		result = tmp;
	    }
	}
	_w = _w->next;
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkAttachTkWindow --
 *
 *	Associates a TkWindow structure with the given X window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Sets the tkwin field of the given _Window. Allocates a copy of
 *	the window name, if any.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkAttachTkWindow(Window w, TkWindow *tkwin)
{
    _Window *_w = (_Window *) w;

    _w->tkwin = tkwin;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkDrawEverything --
 *
 *	Redraw entire display.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkDrawEverything(ClientData clientData)
{
    _Window *child, *focus_window;
    Region tmpRgn;
#ifdef ANDROID
    static SDL_GLContext gl = NULL;
    SDL_GLContext currgl;

    /* Happens when app is/gets paused. */
    if ((currgl = SDL_GL_GetCurrentContext()) == NULL) {
	drawLater = 0;
	return;
    }
    /* Detect that GL context has changed. */
    if (gl != NULL && currgl != gl) {
	SDL_Texture *newtex;

	newtex = SDL_CreateTexture(SdlTk_sdlrend,
				   SDL_PIXELFORMAT_RGB565,
				   SDL_TEXTUREACCESS_STREAMING,
				   SdlTkX_screen->width,
				   SdlTkX_screen->height);
	if (newtex != NULL) {
	    SDL_DestroyTexture(SdlTk_sdltex);
	    SdlTk_sdltex = newtex;
	    gl = currgl;
	} else {
	    drawLater = 0;
	    return;
	}
    } else if (gl == NULL) {
	gl = currgl;
    }
#endif

    tmpRgn = SdlTkRgnPoolGet();
    if (screen_update_region == NULL) {
	screen_update_region = SdlTkRgnPoolGet();
    }
    /* If areas of the root window were exposed, paint them now */
    if (screen_dirty_region && !XEmptyRegion(screen_dirty_region)) {
	Uint32 pixel = SDL_MapRGB(SdlTk_sdlsurf->format, 0x00, 0x4E, 0x98);

	SdlTkGfxFillRegion(SdlTkX_screen->root, screen_dirty_region, pixel);
	XUnionRegion(screen_dirty_region, screen_update_region,
		     screen_update_region);
	XSetEmptyRegion(screen_dirty_region);
    }

    focus_window = (_Window *) SdlTkX_focus_window_not_override;
    if (focus_window != NULL) {
	child = focus_window->parent;
	while (child != NULL && child->dec == NULL) {
	    child = child->parent;
	}
	if (child != NULL) {
	    focus_window = child->child;
 	}
    }

    /* Check each toplevel from highest to lowest */
    for (child = ((_Window *) SdlTkX_screen->root)->child;
	child != NULL;
	child = child->next) {

        if (child->atts.map_state == IsUnmapped) {
	    continue;
	}
	/* Keep track of which decframe was the "active" one. Redraw frames
	* when changes occur. */
	if (child->dec != NULL) {
	    if (child->child == (_Window *) focus_window &&
		!SdlTkDecSetActive(child, -1) && SdlTk_sdlfocus) {
		SdlTkDecSetDraw(child, 1);
		SdlTkDecSetActive(child, 1);
	    } else if (child->child != (_Window *) focus_window &&
		       SdlTkDecSetActive(child, -1)) {
		SdlTkDecSetDraw(child, 1);
		SdlTkDecSetActive(child, 0);
	    }
	    if (SdlTkDecSetDraw(child, -1)) {
		SdlTkDecDrawFrame(child);
		XUnionRegion(child->visRgn, child->dirtyRgn, child->dirtyRgn);
		SdlTkDecSetDraw(child, 0);
	    }
	}

	if (!XEmptyRegion(child->dirtyRgn)) {

	    /* If the toplevel is double-buffered, blit the parts of it that
	     * were exposed or drawn into to the screen. */
	    XIntersectRegion(child->dirtyRgn, child->visRgnInParent, tmpRgn);

	    /* Add those areas to screen_update_region. */
	    XOffsetRegion(tmpRgn, child->atts.x, child->atts.y);
	    XUnionRegion(tmpRgn, screen_update_region, screen_update_region);

	    XSetEmptyRegion(child->dirtyRgn);
	}
    }

    SdlTkRgnPoolFree(tmpRgn);

    SdlTkGfxUpdateRegion(SdlTk_sdlscreen, screen_update_region);

    SDL_UpdateTexture(SdlTk_sdltex, NULL, SdlTk_sdlsurf->pixels,
		      SdlTk_sdlsurf->pitch);
    SDL_RenderCopy(SdlTk_sdlrend, SdlTk_sdltex, NULL, NULL);
    SDL_RenderPresent(SdlTk_sdlrend);

    XSetEmptyRegion(screen_update_region);

    drawLater = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkScreenChanged --
 *
 *	Dispatch redraw of entire display using the after idle
 *	mechanism.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkScreenChanged(void)
{
    if (drawLater) {
	Tcl_CancelIdleCall(SdlTkDrawEverything, NULL);
    }
    Tcl_DoWhenIdle(SdlTkDrawEverything, NULL);
    drawLater = 1;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkTranslateEvent --
 *
 *	Turn an SDL_Event into an XEvent.
 *
 * Results:
 *	Return 1 if the SDL event was translated into an XEvent;
 *	otherwise return 0.
 *
 * Side effects:
 *	Pointer events other than MouseWheelEvent are handled by
 *	Tk_UpdatePointer(), which has various side effects.
 *	If the event type is SDL_QUIT, Tcl_Exit() is called.
 *
 *----------------------------------------------------------------------
 */

int
SdlTkTranslateEvent(SDL_Event *sdl_event, XEvent *event)
{
    unsigned long nowms;
    int x, y, state = 0;
    SDL_Event fix_sdl_event;
    SDL_Event txt_sdl_event;
    const char *evname = NULL;

    nowms = TkpGetMS();

    switch (sdl_event->type) {

	case SDL_TEXTINPUT:
	case SDL_TEXTEDITING:
	case SDL_KEYDOWN:
	case SDL_KEYUP:
	    state = SDL_GetMouseState(&x, &y);

	    event->type = (sdl_event->type == SDL_KEYUP)
	                ? KeyRelease : KeyPress;
	    event->xkey.serial = SdlTkX_display->request;
	    event->xkey.send_event = False;
	    event->xkey.display = SdlTkX_display;
	    event->xkey.window = SdlTkX_focus_window ?
		SdlTkX_focus_window : SdlTkX_display->screens->root;
	    event->xkey.subwindow = None;
	    event->xkey.time = nowms;
	    event->xkey.x = x;
	    event->xkey.y = y;
	    event->xkey.x_root = x;
	    event->xkey.y_root = y;

	    event->xkey.state = 0;
	    if (state & SDL_BUTTON(1)) {
		event->xkey.state |= Button1Mask;
	    }
	    if (state & SDL_BUTTON(2)) {
		event->xkey.state |= Button2Mask;
	    }
	    if (state & SDL_BUTTON(3)) {
		event->xkey.state |= Button3Mask;
	    }

	    event->xkey.keycode = -1;

	    if (sdl_event->type != SDL_TEXTINPUT &&
		sdl_event->type != SDL_TEXTEDITING) {
		if (sdl_event->key.keysym.mod & KMOD_ALT) {
		    event->xkey.state |= ALT_MASK;
		}
		if (sdl_event->key.keysym.mod & KMOD_CAPS) {
		    event->xkey.state |= LockMask;
		}
		if (sdl_event->key.keysym.mod & KMOD_CTRL) {
		    event->xkey.state |= ControlMask;
		}
		if (sdl_event->key.keysym.mod & KMOD_NUM) {
		    event->xkey.state |= Mod1Mask;
		}
		if (sdl_event->key.keysym.mod & KMOD_SHIFT) {
		    event->xkey.state |= ShiftMask;
		}
		/* SDL_SCANCODE_xxx */
		event->xkey.keycode = sdl_event->key.keysym.scancode;
	    }

	    event->xkey.same_screen = True;

	    event->xkey.nbytes = 0;
	    if (sdl_event->type == SDL_TEXTINPUT ||
		sdl_event->type == SDL_TEXTEDITING) {
		event->xkey.nbytes = strlen(sdl_event->text.text);
		if (event->xkey.nbytes > sizeof (event->xkey.trans_chars)) {
		    event->xkey.nbytes = sizeof (event->xkey.trans_chars);
		}
		memcpy(event->xkey.trans_chars, sdl_event->text.text,
		       event->xkey.nbytes);
		/* ignore illegal UTF-8 */
		if ((event->xkey.trans_chars[0] & 0xFF) == 0xC0) {
		    return 0;
		}
	    } else if (SDL_PeepEvents(&txt_sdl_event, 1, SDL_PEEKEVENT,
			       SDL_TEXTINPUT, SDL_TEXTINPUT) == 1) {
	        SDL_PeepEvents(&txt_sdl_event, 1, SDL_GETEVENT,
			       SDL_TEXTINPUT, SDL_TEXTINPUT);
		event->xkey.nbytes = strlen(txt_sdl_event.text.text);
		if (event->xkey.nbytes > sizeof (event->xkey.trans_chars)) {
		    event->xkey.nbytes = sizeof (event->xkey.trans_chars);
		}
		memcpy(event->xkey.trans_chars, txt_sdl_event.text.text,
		       event->xkey.nbytes);
		/* ignore illegal UTF-8 */
		if ((event->xkey.trans_chars[0] & 0xFF) == 0xC0) {
		    return 0;
		}
		if (event->xkey.nbytes > 0 &&
		    (sdl_event->key.keysym.mod & KMOD_RALT)) {
		    event->xkey.state &= ~ALT_MASK;
		}
	    } else if (sdl_event->type == SDL_KEYDOWN) {
	        /* Keypad mapping */
		if ((event->xkey.keycode >= SDL_SCANCODE_KP_0) &&
		    (event->xkey.keycode <= SDL_SCANCODE_KP_9) &&
		    (sdl_event->key.keysym.mod & KMOD_NUM)) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '0' +
		        (event->xkey.keycode - SDL_SCANCODE_KP_0);
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_DIVIDE) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '/';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_MULTIPLY) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '*';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_MINUS) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '-';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_PLUS) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '+';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_ENTER) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '\r';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_PERIOD) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '.';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_COMMA) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = ',';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_EQUALS) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '=';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_LEFTPAREN) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '(';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_RIGHTPAREN) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = ')';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_LEFTBRACE) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '{';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_RIGHTBRACE) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '}';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_VERTICALBAR) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '|';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_TAB) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '\t';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_SPACE) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = ' ';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_EXCLAM) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '!';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_AT) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '@';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_HASH) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '#';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_COLON) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = ':';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_AMPERSAND) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '&';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_LESS) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '<';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_GREATER) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '>';
		} else if (event->xkey.keycode == SDL_SCANCODE_KP_PERCENT) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = '%';
		/* normal mapping */
		} else if (event->xkey.keycode == SDL_SCANCODE_SPACE) {
		    event->xkey.nbytes = 1;
		    event->xkey.trans_chars[0] = ' ';
		}
	    }
	    break;

	case SDL_MOUSEBUTTONDOWN:
	case SDL_MOUSEBUTTONUP:
	case SDL_MOUSEMOTION:
	{
	    _Window *_w;
	    int xstate = 0, othergrab = 0;
	    Tk_Window tkwin;
	    SDL_Keymod mod;

	    switch (sdl_event->type) {
		case SDL_MOUSEBUTTONUP:
		case SDL_MOUSEBUTTONDOWN:
		    x = sdl_event->button.x;
		    y = sdl_event->button.y;
		    state = SDL_BUTTON(sdl_event->button.button);
		    break;
	        case SDL_MOUSEMOTION: {
		    int dx, dy;

		    fix_sdl_event = *sdl_event;
		    sdl_event = &fix_sdl_event;
		    x = sdl_event->motion.x;
		    y = sdl_event->motion.y;
		    dx = sdl_event->motion.xrel;
		    dy = sdl_event->motion.yrel;
		    if (x <= 0 && dx < 0) {
		        dx = 0;
		    } else if (x >= SdlTk_sdlsurf->w - 1 && dx > 0) {
		        dx = 0;
		    }
		    if (y <= 0 && dy < 0) {
		        dy = 0;
		    } else if (y >= SdlTk_sdlsurf->h - 1 && dy > 0) {
		        dy = 0;
		    }
		    sdl_event->motion.xrel = dx;
		    sdl_event->motion.yrel = dy;
		    state = sdl_event->motion.state;
		    break;
		}
	    }

	    _w = SdlTkPointToWindow((_Window *) SdlTkX_screen->root,
				    x, y, True);
	    tk_mouse_window = _w;

	    /* Click in background window to raise it. */
	    /* FIXME: unless a Tk grab is on */
	    if (!IS_ROOT(_w) && SdlTkGrabCheck(_w, &othergrab) &&
		(sdl_event->type == SDL_MOUSEBUTTONDOWN) &&
		(sdl_event->button.button == SDL_BUTTON_LEFT)) {
		SdlTkBringToFrontIfNeeded(_w);
		XSetInputFocus(SdlTkX_display,
			       (Window) SdlTkWrapperForWindow(_w),
			       RevertToParent, CurrentTime);
		/* Frames need redrawing if the focus changed */
		SdlTkScreenChanged();
	    }

	    /* Possible event in decorative frame (button, drag, resize) */
	    /* If a menu is showing, give Tk the click so the
	     * menu will go away. I would still like to drag */
	    if (SdlTkDecFrameEvent(_w, sdl_event, x, y)) {
		/* Hide the event from Tk. */
		return 0;
	    }
	    if (othergrab) {
		return 0;
	    }

	    /* NULL for root and decorative frames */
	    tkwin = (Tk_Window) _w->tkwin;

	    if (tkwin == NULL) {
		tkwin = (Tk_Window) tk_capture_window;
	    }

	    if (state & SDL_BUTTON(1)) {
		xstate |= Button1Mask;
	    }
	    if (state & SDL_BUTTON(2)) {
		xstate |= Button2Mask;
	    }
	    if (state & SDL_BUTTON(3)) {
		xstate |= Button3Mask;
	    }

	    mod = SDL_GetModState();
	    if (mod & KMOD_ALT) {
		xstate |= ALT_MASK;
	    }
	    if (mod & KMOD_CAPS) {
		xstate |= LockMask;
	    }
	    if (mod & KMOD_CTRL) {
		xstate |= ControlMask;
	    }
	    if (mod & KMOD_NUM) {
		xstate |= Mod1Mask;
	    }
	    if (mod & KMOD_SHIFT) {
		xstate |= ShiftMask;
	    }

	    if (sdl_event->type == SDL_MOUSEBUTTONUP) {
	        if (state & SDL_BUTTON(1)) {
		    xstate &= ~Button1Mask;
		}
	        if (state & SDL_BUTTON(2)) {
		    xstate &= ~Button2Mask;
		}
	        if (state & SDL_BUTTON(3)) {
		    xstate &= ~Button3Mask;
		}
		if (xstate == 0) {
		    tk_pointergrab_window = NULL;
		}
	    }
	    if (sdl_event->type == SDL_MOUSEBUTTONDOWN) {
		int bstate = xstate;

	        if (state & SDL_BUTTON(1)) {
		    bstate &= ~Button1Mask;
		}
	        if (state & SDL_BUTTON(2)) {
		    bstate &= ~Button2Mask;
		}
	        if (state & SDL_BUTTON(3)) {
		    bstate &= ~Button3Mask;
		}
		if (bstate == 0) {
		    if (tk_pointergrab_window == NULL) {
			tk_pointergrab_window = (TkWindow *) tkwin;
		    }
		}
	    }
	    if (tk_pointergrab_window != NULL) {
		tkwin = (Tk_Window) tk_pointergrab_window;
	    }
	    tk_mouse_x = x;
	    tk_mouse_y = y;
	    Tk_UpdatePointer(tkwin, x, y, xstate);
	    return 0;
	}

        case SDL_MOUSEWHEEL: {
	    int xstate = 0;
	    Tk_Window tkwin = NULL;

	    if (tk_mouse_window != NULL) {
	        tkwin = (Tk_Window) tk_mouse_window->tkwin;
	    }
	    if (tk_pointergrab_window != NULL) {
		tkwin = (Tk_Window) tk_pointergrab_window;
	    }
	    if (sdl_event->wheel.y < 0) {
	        xstate |= Button5Mask;
	    } else if (sdl_event->wheel.y > 0) {
	        xstate |= Button4Mask;
	    }
	    if (tkwin && xstate) {
	        Tk_UpdatePointer(tkwin, tk_mouse_x, tk_mouse_y, xstate);
	        Tk_UpdatePointer(tkwin, tk_mouse_x, tk_mouse_y, 0);
	    }
	    return 0;
	}

	case SDL_QUIT:
	    if (SdlTkDecFrameEvent((_Window *) SdlTkX_screen->root,
				   sdl_event, 0, 0)) {
		return 0;
	    }
	    Tcl_Exit(0);
	    break;

	case SDL_APP_LOWMEMORY:
	    evname = "LowMemory";
	    goto doAppEvent;
	case SDL_APP_TERMINATING:
	    evname = "Terminating";
	    goto doAppEvent;
	case SDL_APP_WILLENTERBACKGROUND:
	    evname = "WillEnterBackground";
	    goto doAppEvent;
	case SDL_APP_DIDENTERBACKGROUND:
	    evname = "DidEnterBackground";
	    goto doAppEvent;
	case SDL_APP_WILLENTERFOREGROUND:
	    evname = "WillEnterForeground";
	    goto doAppEvent;
	case SDL_APP_DIDENTERFOREGROUND:
	    evname = "DidEnterForeground";
	doAppEvent: {
	    int nsent = 0;

	    memset(event, 0, sizeof (event));
	    event->xany.type = VirtualEvent;
	    event->xany.send_event = False;
	    event->xany.window = (Window) SdlTkX_screen->root;
	    event->xany.display = SdlTkX_display;
	    ((XVirtualEvent *) event)->name = Tk_GetUid(evname);
	    event->xany.serial = SdlTkX_display->request;
	    /* only TK_APP_TOP_LEVEL windows will get this */
	    event->xany.window = (Window)
		SdlTkSendAppEvent(event, &nsent,
				  ((_Window *) SdlTkX_screen->root)->child);
	    return nsent > 0;
	}

	case SDL_USEREVENT: {
	    int nsent = 0;

	    if (sdl_event->user.data1 != NULL) {
		evname = sdl_event->user.data1;
	    }
	    if (evname != NULL) {
		memset(event, 0, sizeof (event));
		event->xany.type = VirtualEvent;
		event->xany.send_event = False;
		event->xany.window = (Window) SdlTkX_screen->root;
		event->xany.display = SdlTkX_display;
		((XVirtualEvent *) event)->name = Tk_GetUid(evname);
		event->xany.serial = SdlTkX_display->request;
		event->xbutton.x = event->xbutton.y = sdl_event->user.code;
		event->xbutton.state = (long) sdl_event->user.data2;
		/* only TK_APP_TOP_LEVEL windows will get this */
		event->xany.window = (Window)
		    SdlTkSendAppEvent(event, &nsent,
				      ((_Window *) SdlTkX_screen->root)->child);
	    }
	    return nsent > 0;
	}

	case SDL_FINGERDOWN:
	    evname = "FingerDown";
	    goto doFingerEvent;
	case SDL_FINGERUP:
	    evname = "FingerUp";
	    goto doFingerEvent;
	case SDL_FINGERMOTION:
	    evname = "FingerMotion";
	doFingerEvent: {
	    _Window *_w;
	    Tk_Window tkwin = NULL;

	    _w = SdlTkPointToWindow((_Window *) SdlTkX_screen->root,
				    x, y, True);
	    if (_w != NULL) {
		tkwin = (Tk_Window) _w->tkwin;
	    }
	    if (tkwin == NULL) {
		tkwin = (Tk_Window) tk_capture_window;
	    }
	    if (tkwin != NULL) {
		int pressure;

		memset(event, 0, sizeof (event));
		event->xany.type = VirtualEvent;
		event->xany.serial = SdlTkX_display->request;
		event->xany.send_event = False;
		event->xany.window = Tk_WindowId(tkwin);
		event->xany.display = SdlTkX_display;
		event->xbutton.x = sdl_event->tfinger.x * 10000;
		if (event->xbutton.x >= 10000) {
		    event->xbutton.x = 9999;
		} else if (event->xbutton.x < 0) {
		    event->xbutton.x = 0;
		}
		event->xbutton.y = sdl_event->tfinger.y * 10000;
		if (event->xbutton.y >= 10000) {
		    event->xbutton.y = 9999;
		} else if (event->xbutton.y < 0) {
		    event->xbutton.y = 0;
		}
		event->xbutton.x_root = sdl_event->tfinger.dx * 10000;
		if (event->xbutton.x_root >= 10000) {
		    event->xbutton.x_root = 9999;
		} else if (event->xbutton.x_root < 0) {
		    event->xbutton.x_root = 0;
		}
		event->xbutton.y_root = sdl_event->tfinger.dy * 10000;
		if (event->xbutton.y_root >= 10000) {
		    event->xbutton.y_root = 9999;
		} else if (event->xbutton.y_root < 0) {
		    event->xbutton.y_root = 0;
		}
		pressure = sdl_event->tfinger.pressure * 10000;
		if (pressure >= 10000) {
		    pressure = 9999;
		} else if (pressure < 0) {
		    pressure = 0;
		}
		event->xbutton.time = pressure;
		event->xbutton.state = sdl_event->tfinger.fingerId + 1;
		((XVirtualEvent *) event)->name = Tk_GetUid(evname);
		return 1;
	    }
	    return 0;
	}

#ifdef SDL_GESTURES_ARE_WORKING
	case SDL_DOLLARGESTURE:
	    evname = "Gesture";
	    goto doGestureEvent;
	case SDL_DOLLARRECORD:
	    evname = "GestureRecord";
	doGestureEvent: {
	    _Window *_w;
	    Tk_Window tkwin = NULL;

	    _w = SdlTkPointToWindow((_Window *) SdlTkX_screen->root,
				    x, y, True);
	    if (_w != NULL) {
		tkwin = (Tk_Window) _w->tkwin;
	    }
	    if (tkwin == NULL) {
		tkwin = (Tk_Window) tk_capture_window;
	    }
	    if (tkwin != NULL) {
		int error;

		memset(event, 0, sizeof (event));
		event->xany.type = VirtualEvent;
		event->xany.serial = SdlTkX_display->request;
		event->xany.send_event = False;
		event->xany.window = Tk_WindowId(tkwin);
		event->xany.display = SdlTkX_display;
		event->xbutton.x = sdl_event->dgesture.x * 10000;
		if (event->xbutton.x >= 10000) {
		    event->xbutton.x = 9999;
		} else if (event->xbutton.x < 0) {
		    event->xbutton.x = 0;
		}
		event->xbutton.y = sdl_event->dgesture.y * 10000;
		if (event->xbutton.y >= 10000) {
		    event->xbutton.y = 9999;
		} else if (event->xbutton.y < 0) {
		    event->xbutton.y = 0;
		}
		error = sdl_event->dgesture.error * 10000;
		if (error >= 10000) {
		    error = 9999;
		} else if (error < 0) {
		    error = 0;
		}
		event->xbutton.time = error;
		event->xbutton.state = sdl_event->dgesture.gestureId + 1;
		event->xbutton.button = sdl_event->dgesture.numFingers;
		((XVirtualEvent *) event)->name = Tk_GetUid(evname);
		return 1;
	    }
	    return 0;
	}
#endif

	case SDL_CLIPBOARDUPDATE:
	    XSetSelectionOwner(SdlTkX_display, None, None, nowms);
	    return 0;

#ifdef ANDROID
	case SDL_JOYAXISMOTION: {
	    int nsent = 0;

	    memset(event, 0, sizeof (event));
	    event->xany.type = VirtualEvent;
	    event->xany.send_event = False;
	    event->xany.window = (Window) SdlTkX_screen->root;
	    event->xany.display = SdlTkX_display;
	    event->xany.serial = SdlTkX_display->request;
	    event->xbutton.x = event->xbutton.y = sdl_event->jaxis.value;
	    event->xbutton.state = sdl_event->jaxis.axis + 1;
	    ((XVirtualEvent *) event)->name = Tk_GetUid("Accelerometer");
	    /* only TK_APP_TOP_LEVEL windows will get this */
	    event->xany.window = (Window)
		SdlTkSendAppEvent(event, &nsent,
				  ((_Window *) SdlTkX_screen->root)->child);
	    return nsent > 0;
	}
#endif

	case SDL_WINDOWEVENT: {
	    int width, height;
	    SDL_PixelFormat *pfmt;
	    SDL_Surface *newsurf = NULL;
	    SDL_Texture *newtex = NULL;
	    _Window *_w;

	    switch (sdl_event->window.event) {

		case SDL_WINDOWEVENT_RESIZED:
		    width = sdl_event->window.data1;
		    height = sdl_event->window.data2;
		    pfmt = SdlTk_sdlsurf->format;
		    newsurf =
			SDL_CreateRGBSurface(SDL_SWSURFACE, width, height,
					     pfmt->BitsPerPixel, pfmt->Rmask,
					     pfmt->Gmask, pfmt->Bmask, 0);
		    newtex =
			SDL_CreateTexture(SdlTk_sdlrend,
#ifdef ANDROID
					  SDL_PIXELFORMAT_RGB565,
#else
					  SDL_PIXELFORMAT_RGB888,
#endif
					  SDL_TEXTUREACCESS_STREAMING,
					  width, height);
		    if (newsurf != NULL && newtex != NULL) {
			int oldw, oldh;
			SDL_Rect sr;
			Uint32 pixel;
			_Window *child;

			SDL_BlitSurface(SdlTk_sdlsurf, NULL, newsurf, NULL);
			SDL_FreeSurface(SdlTk_sdlsurf);
			SdlTk_sdlsurf = newsurf;
			SDL_DestroyTexture(SdlTk_sdltex);
			SdlTk_sdltex = newtex;
			oldw = SdlTkX_screen->width;
			oldh = SdlTkX_screen->height;
			SdlTkX_screen->width = width;
			SdlTkX_screen->height = height;
			_w = (_Window *) SdlTkX_screen->root;
			_w->sdl = newsurf;
			_w->atts.width = _w->parentWidth = width;
			_w->atts.height = _w->parentHeight = height;
			pixel = SDL_MapRGB(SdlTk_sdlsurf->format,
					   0x00, 0x4E, 0x78);
			if (width > oldw) {
			    sr.x = oldw;
			    sr.w = width - oldw;
			    sr.y = 0;
			    sr.h = height;
			    SDL_FillRect(SdlTk_sdlsurf, &sr, pixel);
			}
			if (height > oldh) {
			    sr.y = oldh;
			    sr.h = height - oldh;
			    sr.x = 0;
			    sr.w = width;
			    SDL_FillRect(SdlTk_sdlsurf, &sr, pixel);
			}
			if (width > oldw || height > oldh) {
			    SdlTkVisRgnChanged(_w, VRC_CHANGED, 0, 0);
			}

			child = _w->child;
			while (child != NULL) {
			    if (child->fullscreen) {
				int xx, yy, ww, hh;
				_Window *_ww = child;

				xx = 0;
				yy = 0;
				ww = width;
				hh = height;
				if (child->dec != NULL) {
				    xx -= DEC_FRAME_WIDTH;
				    yy -= DEC_TITLE_HEIGHT;
				    ww += 2 * DEC_FRAME_WIDTH;
				    hh += DEC_TITLE_HEIGHT + DEC_FRAME_WIDTH;
				    _ww = child->child;
				}
				child->fullscreen = 0;
				_ww->fullscreen = 0;
				XMoveResizeWindow(SdlTkX_display,
						  (Window) _ww,
						  xx, yy, ww, hh);
				_ww->fullscreen = 1;
				child->fullscreen = 1;
			    } 
			    child = child->next;
			}
			goto refresh;
		    } else {
			if (newsurf != NULL) {
			    SDL_FreeSurface(newsurf);
			}
			if (newtex != NULL) {
			    SDL_DestroyTexture(newtex);
			}
		    }
		    return 0;

		case SDL_WINDOWEVENT_FOCUS_GAINED:
		    if (!SdlTk_sdlfocus) {
			SdlTk_sdlfocus = 1;
			if (SdlTkX_focus_window_old != None) {
			    XSetInputFocus(SdlTkX_display,
					   SdlTkX_focus_window_old,
					   RevertToParent, CurrentTime);
		    	    goto refresh;
			}
		    }
		    return 0;

		case SDL_WINDOWEVENT_FOCUS_LOST:
		    if (SdlTk_sdlfocus) {
			SdlTkX_focus_window_old = SdlTkX_focus_window;
			SdlTk_sdlfocus = 0;
			if (SdlTkX_focus_window != None)
			    XSetInputFocus(SdlTkX_display, None,
					   RevertToNone, CurrentTime);
			    goto refresh;
			}
		    }
		    return 0;

		case SDL_WINDOWEVENT_SIZE_CHANGED:
		case SDL_WINDOWEVENT_RESTORED:
		case SDL_WINDOWEVENT_SHOWN:
		case SDL_WINDOWEVENT_EXPOSED:
		refresh:
		    SdlTkScreenChanged();
		    return 0;
	    }
	    return 0;

	default:
	    return 0;
    }

    return 1;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkCalculateVisibleRegion --
 *
 *	This procedure calculates the visible region for a window.
 *
 *	NOTE: The caller is responsible for freeing the old visRgn
 *	and visRgnInParent.
 *
 *	NOTE: The parent's visRgnInParent must be correct before
 *	this procedure may be called.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The visRgn and visRgnInParent members of the
 *	given window are changed.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkCalculateVisibleRegion(_Window *_w)
{
    Region rgn2;
    XRectangle r1, r2;
    _Window *ancestor, *child, *parent = _w->parent;

    /* Caller must free old ones */
    _w->visRgn = SdlTkRgnPoolGet();
    _w->visRgnInParent = SdlTkRgnPoolGet();

    /* An unmapped window has an empty visible region */
    if (_w->atts.map_state == IsUnmapped) {
	return;
    }

    /* If any ancestor is unmapped, this window has an empty
     * visible region. In X11, a window may be mapped even if an
     * ancestor isn't */
    if (parent != NULL) {
	ancestor = parent;
	while (!IS_ROOT(ancestor)) {
	    if (ancestor->atts.map_state == IsUnmapped) {
		return;
	    }
	    ancestor = ancestor->parent;
	}
    }

    /* Start with a rectangle as large as ourself in our parent */
    r1.x = _w->atts.x;
    r1.y = _w->atts.y;
    r1.width = _w->parentWidth;
    r1.height = _w->parentHeight;
    XUnionRectWithRegion(&r1, _w->visRgnInParent, _w->visRgnInParent);

    if (parent != NULL) {
	/* Intersect with parent's visRgnInParent */
	XIntersectRegion(_w->visRgnInParent, _w->parent->visRgnInParent,
			 _w->visRgnInParent);

	/* Subtract one rectangle for each mapped sibling higher in the
	* stacking order, unless this is a double-buffered window. */
	if (parent->child != _w) {
	    rgn2 = SdlTkRgnPoolGet();
	    child = parent->child;
	    while (child != _w) {
		if (child->atts.map_state != IsUnmapped) {
		    r2.x = child->atts.x;
		    r2.y = child->atts.y;
		    r2.width = child->parentWidth;
		    r2.height = child->parentHeight;
		    XUnionRectWithRegion(&r2, rgn2, rgn2);
		}
		child = child->next;
	    }
	    XSubtractRegion(_w->visRgnInParent, rgn2, _w->visRgnInParent);
	    SdlTkRgnPoolFree(rgn2);

	    /* A window's siblings may completely obscure it */
	    if (XEmptyRegion(_w->visRgnInParent)) {
	        if (!PARENT_IS_ROOT(_w)) {
		    return;
		}
	    }
	}
    }

    /* Calculation of visRgnInParent is complete. Copy visRgnInParent into
     * visRgn. */
    XUnionRegion(_w->visRgnInParent, _w->visRgn, _w->visRgn);

    /* Subtract one rectangle for each mapped child from visRgn (a window
     * can't draw over its children) */
    if (_w->child != NULL) {
	rgn2 = SdlTkRgnPoolGet();
	for (child = _w->child; child != NULL; child = child->next) {
	    if (child->atts.map_state != IsUnmapped) {
		r2.x = _w->atts.x + child->atts.x;
		r2.y = _w->atts.y + child->atts.y;
		r2.width = child->parentWidth;
		r2.height = child->parentHeight;
		XUnionRectWithRegion(&r2, rgn2, rgn2);
	    }
	}
	XSubtractRegion(_w->visRgn, rgn2, _w->visRgn);
	SdlTkRgnPoolFree(rgn2);
    }

    /* Make relative to this window's top-left corner */
    XOffsetRegion(_w->visRgn, -_w->atts.x, -_w->atts.y);
    XOffsetRegion(_w->visRgnInParent, -_w->atts.x, -_w->atts.y);
}

/*
 *----------------------------------------------------------------------
 *
 * blitMovedWindow --
 *
 *	Helper routine for SdlTkVisRgnChanged(). Copies the visible
 *	pixels of a window from its old location to its new location.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Pixels will be blitted within the drawing surface of the
 *	window's parent.
 *
 *----------------------------------------------------------------------
 */

static void
blitMovedWindow(_Window *_w, int x, int y)
{
    XGCValues fakeGC;
    TkpClipMask clip;
    int xOff, yOff;
    Region parentVisRgn;

    /* Hack: SdlTkGfxCopyArea will clip to our parent's visRgn. Need to
    * clip to our parent's visRgnInParent instead. */
    parentVisRgn = _w->parent->visRgn;
    _w->parent->visRgn = _w->parent->visRgnInParent;

    /* This window's visRgnInParent is used as the clip region. Put
    * it in the parent's coordinates. */
    XOffsetRegion(_w->visRgnInParent, _w->atts.x, _w->atts.y);

    clip.type = TKP_CLIP_REGION;
    clip.value.region = (TkRegion) _w->visRgnInParent;
    fakeGC.clip_mask = (Pixmap) &clip;
    fakeGC.graphics_exposures = False;
    fakeGC.clip_x_origin = 0;
    fakeGC.clip_y_origin = 0;

    /* What actually needs to be done is copy everything inside the
    * old visRgnInParent to the new location. */
    SdlTkGfxCopyArea((Drawable) _w->parent, (Drawable) _w->parent, &fakeGC,
	x, y, _w->parentWidth, _w->parentHeight, _w->atts.x, _w->atts.y);

    XOffsetRegion(_w->visRgnInParent, -_w->atts.x, -_w->atts.y);

    /* Undo hack */
    _w->parent->visRgn = parentVisRgn;

    /* If this is a double-buffered toplevel, add its visible area to the
     * screen's dirtyRgn */
    SdlTkRootCoords(_w, &xOff, &yOff);
    XOffsetRegion(_w->visRgnInParent, xOff, yOff);
    if (screen_update_region == NULL) {
	screen_update_region = SdlTkRgnPoolGet();
    }
    XUnionRegion(_w->visRgnInParent, screen_update_region, screen_update_region);
    XOffsetRegion(_w->visRgnInParent, -xOff, -yOff);
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkVisRgnChanged --
 *
 *	This procedure gets called whenever a window is mapped,
 *	unmapped, raised, lowered, moved, or resized. It recalculates
 *	the visible regions for all windows affected by the change.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The visRgn, visRgnInParent and dirtyRgn
 *	members of various windows may be updated. If the window
 *	was moved, then pixels may be copied within its parent's
 *	drawing surface. <Expose> events may be added to the event
 *	queue.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkVisRgnChanged(_Window *_w, int flags, int x, int y)
{
    static int depth = 0;
    Region visRgn, visRgnInParent = NULL;

    depth++;

    /*
     * If this is NOT the window that was mapped/unmapped/moved/
     * resized/restacked and it is unmapped, don't bother recalculating
     * visible regions for this window.
     */
    if ((flags & VRC_CHANGED) || (_w->atts.map_state != IsUnmapped)) {

	/* A window obscures part of its parent */
	/* Update parent's visible region before blitting this window */
        if ((flags & VRC_DO_PARENT) && (_w->parent != NULL)) {
	    SdlTkVisRgnChanged(_w->parent, VRC_SELF_ONLY, 0, 0);
	}

	/* Save the old regions temporarily so changes can be examined */
	visRgn = _w->visRgn;
	visRgnInParent = _w->visRgnInParent;

	SdlTkCalculateVisibleRegion(_w);

	/*
  	 * If this window moved within its parent, copy all the visible
  	 * pixels to the new location.
  	 */
	if ((flags & VRC_MOVE) && !XEmptyRegion(_w->visRgnInParent)) {
	    blitMovedWindow(_w, x, y);
	}

	if (_w->atts.map_state != IsUnmapped) {

	    /* Take away what was visible before */
	    XSubtractRegion(_w->visRgn, visRgn, visRgn);

	    /*
	     * Generate <Expose> events for newly-exposed areas.
	     * Only generate events for real Tk windows, not a
	     * decframe or wrapper
	     */
	    if (_w->tkwin != NULL) {
		(void) SdlTkGfxExposeRegion((Window) _w, visRgn);
	    } else if (_w->dec != NULL) {
		/*
		 * If the toplevel is double-buffered, the visRgn can only
		 * change if the toplevel is resized (because other toplevels
		 * don't obscure it.)
		 */
	        if (!XEmptyRegion(visRgn)) {
		   SdlTkDecSetDraw(_w, 1);
		}

	    } else if (IS_ROOT(_w)) {
		/*
		 * Can't erase right away, because it will erase the pixels of
		 * any toplevels which moved (I want to blit those pixels).
		 */
	        if (screen_dirty_region == NULL) {
		    screen_dirty_region = SdlTkRgnPoolGet();
		}
		XUnionRegion(screen_dirty_region, visRgn, screen_dirty_region);
	    }
	}

	SdlTkRgnPoolFree(visRgn);
    }

    if (!(flags & VRC_SELF_ONLY)) {
	/*
	 * If this is NOT the window that was mapped/unmapped/moved/
	 * resized/restacked and it is unmapped, don't bother
	 * recalculating visible regions for any descendants.
	 */
	if ((flags & VRC_CHANGED) || (_w->atts.map_state != IsUnmapped)) {
	    /*
	     * If this window's visRgnInParent did not change (such as when
	     * moving a toplevel), don't recalculate the visible
	     * regions of any descendants.
	     */
	    if (!XEqualRegion(_w->visRgnInParent, visRgnInParent)) {
		/*
		 * We only need to do our first child, because it will do
		 * its next sibling etc.
		 */
	        if (_w->child != NULL) {
		    SdlTkVisRgnChanged(_w->child, 0, 0, 0);
		}
	    }
	}

	/* A window may obscure siblings lower in the stacking order */
	if (_w->next != NULL) {
	    SdlTkVisRgnChanged(_w->next, 0, 0, 0);
	}
    }

    if (visRgnInParent != NULL) {
	SdlTkRgnPoolFree(visRgnInParent);
    }

    depth--;

    if (depth <= 0) {
        /* If areas of the root window were exposed, paint them now */
	if (screen_dirty_region && !XEmptyRegion(screen_dirty_region)) {
	    Uint32 pixel = SDL_MapRGB(SdlTk_sdlsurf->format, 0x00, 0x4E, 0x78);

	    SdlTkGfxFillRegion(SdlTkX_screen->root, screen_dirty_region, pixel);
	    if (screen_update_region == NULL) {
		screen_update_region = SdlTkRgnPoolGet();
	    }
	    XUnionRegion(screen_dirty_region, screen_update_region,
			 screen_update_region);
	    XSetEmptyRegion(screen_dirty_region);
	}
    }
}

Region
SdlTkGetVisibleRegion(_Window *_w)
{
    if (_w->visRgn == NULL) {
	_w->visRgn = SdlTkRgnPoolGet();
    }
    return _w->visRgn;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkGenerateConfigureNotify --
 *
 *	Generate a <Configure> event for a window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A window event is added to the event queue.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkGenerateConfigureNotify(Display *display, Window w)
{
    _Window *_w = (_Window *) w;
    XEvent event;

    event.type = ConfigureNotify;
    event.xconfigure.serial = display->request;
    event.xconfigure.send_event = False;
    event.xconfigure.display = display;
    event.xconfigure.event = w;
    event.xconfigure.window = w;
    event.xconfigure.border_width = _w->atts.border_width;
    event.xconfigure.override_redirect = _w->atts.override_redirect;
    event.xconfigure.x = _w->atts.x;
    event.xconfigure.y = _w->atts.y;
    event.xconfigure.width = _w->atts.width;
    event.xconfigure.height = _w->atts.height;
    event.xconfigure.above = None;
    Tk_QueueWindowEvent(&event, TCL_QUEUE_TAIL);
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkGenerateExpose --
 *
 *	Generate an <Expose> event for part of a window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A window event is added to the event queue.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkGenerateExpose(Display *display, Window w, int x, int y,
		    int width, int height)
{
    XEvent event;

    event.type = Expose;
    event.xexpose.serial = display->request;
    event.xexpose.send_event = False;
    event.xexpose.display = display;
    event.xexpose.window = w;
    event.xexpose.x = x;
    event.xexpose.y = y;
    event.xexpose.width = width;
    event.xexpose.height = height;
    event.xexpose.count = 0;
    Tk_QueueWindowEvent(&event, TCL_QUEUE_TAIL);
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkRootCoords --
 *
 *	Determines the screen coordinates of the top-left corner
 *	of a window, whether it is mapped or not.
 *
 * Results:
 *	*x and *y contain the offsets from the top-left of the
 *	root window to the top-left corner of the given window.
 *
 * Side effects:
 *
 *----------------------------------------------------------------------
 */

void
SdlTkRootCoords(_Window *_w, int *x, int *y)
{
    int xOff, yOff;

    xOff = _w->atts.x;
    yOff = _w->atts.y;
    while (_w->parent != NULL) {
	_w = _w->parent;
	xOff += _w->atts.x;
	yOff += _w->atts.y;
    }
    if (x) {
	*x = xOff;
    }
    if (y) {
	*y = yOff;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkToplevelForWindow --
 *
 *	Return the coordinates for a window in its toplevel.
 *
 * Results:
 *	Returns the toplevel window containing the given window.
 *	*x and *y (which may be NULL) contain the offsets from
 *	the top-left corner of the toplevel to the top-left
 *	corner of the given window.
 *
 * Side effects:
 *
 *----------------------------------------------------------------------
 */

_Window *
SdlTkToplevelForWindow(_Window *_w, int *x, int *y)
{
    int xOff, yOff;

    if (IS_ROOT(_w)) {
	return NULL;
    }

    /* Won't usually ask for this (no drawing takes place in wrapper) */
    if (PARENT_IS_ROOT(_w)) {
        if (x) {
	    *x = 0;
	}
	if (y) {
	    *y = 0;
	}
	return _w;
    }

    xOff = _w->atts.x;
    yOff = _w->atts.y;
    while (!PARENT_IS_ROOT(_w->parent)) {
	_w = _w->parent;
	xOff += _w->atts.x;
	yOff += _w->atts.y;
    }
    if (x) {
	*x = xOff;
    }
    if (y) {
	*y = yOff;
    }
    return _w->parent;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkWrapperForWindow --
 *
 *	Get the Tk wrapper for a window.
 *
 * Results:
 *	Returns the Tk wrapper window that is an ancestor of the given
 *	window, or, if the given window is a decframe, returns its child
 *	(which must be a wrapper).
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

_Window *
SdlTkWrapperForWindow(_Window *_w)
{
    if (IS_ROOT(_w)) {
	return NULL;
    }
    while (!PARENT_IS_ROOT(_w)) {
	_w = _w->parent;
    }
    if (_w->dec != NULL) {
	_w = _w->child;
    }
    return _w;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkTopVisibleWrapper --
 *
 *	Determine the top mapped Tk wrapper.
 *
 * Results:
 *	Returns the mapped wrapper highest in the stack, or NULL if
 *	there are no mapped wrapper windows.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

_Window *
SdlTkTopVisibleWrapper(void)
{
    _Window *child = ((_Window *) SdlTkX_screen->root)->child;

    while (child != NULL) {
	if (child->atts.map_state != IsUnmapped) {
	    if (child->dec != NULL) {
		child = child->child; /* the wrapper */
	    }
	    break;
	}
	child = child->next;
    }
    return child;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkGetDrawableSurface --
 *
 *	Determines which SDL_Surface should be used to draw into an
 *	X Drawable.
 *
 * Results:
 *	If the given Drawable is a pixmap, the result is the pixmap's
 *	own surface.
 *
 *	If the Drawable is a double-buffered toplevel or the descendant
 *	of one, the result is the toplevel's own surface. 
 *
 *	Otherwise the result is the screen's surface.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

SDL_Surface *
SdlTkGetDrawableSurface(Drawable d, int *x, int *y, int *format)
{
    _Pixmap *_p = (_Pixmap *) d;
    _Window *_w = (_Window *) d;

    if (_p->type == DT_PIXMAP) {
        if (x) {
	    *x = 0;
	}
	if (y) {
	    *y = 0;
	}
	if (format != NULL) {
	    *format = _p->format;
	}
	return _p->sdl;
    }

    if (IS_ROOT(_w)) {
        if (x) {
	    *x = 0;
	}
	if (y) {
	    *y = 0;
	}
	if (format != NULL) {
	    *format = SdlTkX_root->format;
	}
	return SdlTk_sdlsurf;
    }

    SdlTkRootCoords(_w, x, y);
    if (format != NULL) {
	*format = SdlTkX_root->format;
    }
    return SdlTk_sdlsurf;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkPointToWindow --
 *
 *	Determines which window contains the given x,y location.
 *
 * Results:
 *	Returns the deepest descendant containing the given
 *	location; returns the given window if no descendant
 *	contains the location.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

_Window *
SdlTkPointToWindow(_Window *_w, int x, int y, Bool mapped)
{
    _Window *child;

    for (child = _w->child;
	child != NULL;
	child = child->next) {
	if (x >= child->atts.x &&
	    x < child->atts.x + child->parentWidth &&
	    y >= child->atts.y &&
	    y < child->atts.y + child->parentHeight) {
	    if (!mapped || (child->atts.map_state != IsUnmapped)) {
		x -= child->atts.x;
		y -= child->atts.y;
		return SdlTkPointToWindow(child, x, y, mapped);
	    }
	}
    }
    return _w;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkRemoveFromParent --
 *
 *	Remove a window from its parent's list of children.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The window is removed from the parent's list of children.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkRemoveFromParent(_Window *_w)
{
    _Window *child = _w->parent->child, *prev = NULL;

    /* Find previous */
    while (child != NULL) {
        if (child == _w) {
	    break;
	}
	prev = child;
	child = child->next;
    }
    if (child == NULL) {
	Tcl_Panic("SdlTkRemoveFromParent: can't find %p\n", _w);
    }

    if (prev == NULL) {
	_w->parent->child = _w->next;
    } else {
	prev->next = _w->next;
    }
    _w->parent = NULL;
    _w->next = NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkAddToParent --
 *
 *	Add a window to another's list of children.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The window is inserted into the parent's list of children.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkAddToParent(_Window *_w, _Window *parent, _Window *sibling)
{
    _Window *child = parent->child, *prev = NULL;

    _w->parent = parent;

    /* Make only child */
    if (child == NULL) {
	parent->child = _w;
	return;
    }

    /* Make last child */
    if (sibling == NULL) {
        while (child->next != NULL) {
	    child = child->next;
	}
	child->next = _w;
	return;
    }

    /* Make first child */
    if (child == sibling) {
	_w->next = sibling;
	parent->child = _w;
	return;
    }

    /* Find previous to sibling */
    while (child != NULL) {
        if (child == sibling) {
	    break;
	}
	prev = child;
	child = child->next;
    }
    if (child == NULL) {
	Tcl_Panic("SdlTkAddToParent: can't find sibling");
    }

    prev->next = _w;
    _w->next = sibling;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkRestackWindow --
 *
 *	Put a window above or below a sibling. This is the main
 *	window-restacking function.
 *
 * Results:
 *
 * Side effects:
 *	Stacking order may change, visible regions may change, <Expose>
 *	events may be added to the event queue.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkRestackWindow(_Window *_w, _Window *sibling, int stack_mode)
{
    _Window *parent = _w->parent;
    int oldPos = 0, newPos = 0;
    _Window *child, *oldNext = _w->next;;

    for (child = parent->child; child != _w; child = child->next)
	oldPos++;

    SdlTkRemoveFromParent(_w);

    if (sibling == NULL) {
	switch (stack_mode) {
	    case Above:
		sibling = parent->child;
		break;
	    case Below:
		sibling = _w;
		while (sibling->next != NULL) {
		    sibling = sibling->next;
		}
		if (sibling == NULL || sibling == _w) {
		    return;
		}
		break;
	}
    } else {
	switch (stack_mode) {
	    case Below:
		sibling = sibling->next;
		break;
	}
    }
    SdlTkAddToParent(_w, parent, sibling);

    /* When restacking a child window, the parent's visible region is
     * never affected */

    for (child = parent->child; child != _w; child = child->next) {
	newPos++;
    }

    if (oldPos > newPos) {
        /* Raised */
	SdlTkVisRgnChanged(_w, VRC_CHANGED, 0, 0);
    } else if (oldPos < newPos) {
        /* Lowered */
	SdlTkVisRgnChanged(oldNext, VRC_CHANGED, 0, 0);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkRestackTransients --
 *
 *	Restack any transient toplevels of the given toplevel so they
 *	are kept in front of it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Stacking order may change, visible regions may change, <Expose>
 *	events may be added to the event queue.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkRestackTransients(_Window *_w)
{
    _Window *sibling;

    _w = SdlTkToplevelForWindow(_w, NULL, NULL);

again:
    sibling = _w->next;
    while (sibling != NULL) {
	if (SdlTkWrapperForWindow(sibling)->master ==
	    SdlTkWrapperForWindow(_w)) {
	    SdlTkRestackWindow(sibling, _w, Above);
	    SdlTkRestackTransients(sibling);
	    goto again;
	}
	sibling = sibling->next;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkBringToFrontIfNeeded --
 *
 *	Raises the toplevel for the given window above any toplevels
 *	higher in the stacking order which are not transients (or
 *	transients of transients) of the toplevel.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Stacking order may change, visible regions may change, <Expose>
 *	events may be added to the event queue.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkBringToFrontIfNeeded(_Window *_w)
{
    _Window *sibling, *master;

    _w = SdlTkToplevelForWindow(_w, NULL, NULL);
    if (_w == NULL) {
	return;
    }

    sibling = _w->parent->child;
    while (sibling != _w && SdlTkIsTransientOf(sibling, _w))
	sibling = sibling->next;

    if (sibling != _w) {
	SdlTkRestackWindow(_w, sibling, Above);
	SdlTkRestackTransients(_w);
    }

    master = SdlTkWrapperForWindow(_w)->master;
    if (master != NULL) {
	SdlTkBringToFrontIfNeeded(master);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkIsTransientOf --
 *
 *	Determine if a toplevel is a transient (or transient of a
 *	transient) of another toplevel.
 *
 * Results:
 *	Returns TRUE if the window is a transient, FALSE otherwise.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
SdlTkIsTransientOf(_Window *_w, _Window *other)
{
    _Window *master = SdlTkWrapperForWindow(_w)->master;

    other = SdlTkWrapperForWindow(other);
    while (master != NULL) {
        if (master == other) {
	    return 1;
	}
	master = master->master;
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkLostFocusWindow --
 *
 *	Called when the X wrapper which had the focus is unmapped.
 *	Sets the focus to the topmost visible wrapper window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The focus changes.
 *
 *----------------------------------------------------------------------
 */

void
SdlTkLostFocusWindow(void)
{
    _Window *focus;

    focus = SdlTkTopVisibleWrapper();

    XSetInputFocus(SdlTkX_display, (Window) focus, RevertToParent, CurrentTime);
}

/*
 *----------------------------------------------------------------------
 *
 * SdlTkGrabCheck --
 *
 *	Determine if a pointer events should be allowed in a window.
 *
 * Results:
 *	Returns TRUE if the Tk toplevel for the window has a grab on
 *	the pointer, or if the Tk toplevel is not excluded from any
 *	local or global grab that is in progress.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
SdlTkGrabCheck(_Window *_w, int *othergrab)
{
    TkMainInfo *mainInfo;
    int state = 0;
    _w = SdlTkWrapperForWindow(_w);

    *othergrab = 0;
    /* Get the actual Tk toplevel inside the wrapper */
    if (_w->child != NULL) {
        if (_w->child->next != NULL) {
	    _w = _w->child->next; /* skip menubar */
	} else {
	    _w = _w->child;
	}
    }

    if (tk_capture_window != NULL) {
	return tk_capture_window == _w->tkwin;
    }

    if (_w->tkwin != NULL) {
        state = TkGrabState(_w->tkwin);
        if (state == TK_GRAB_EXCLUDED) {
	    return 0;
	}
	if (state != TK_GRAB_NONE) {
	    return 1;
	}
	/* check console vs. main window */
	mainInfo = TkGetMainInfoList();
	while (mainInfo != NULL) {
	    if (mainInfo != _w->tkwin->mainPtr) {
		if (mainInfo->winPtr != NULL) {
		    TkDisplay *dispPtr = mainInfo->winPtr->dispPtr;

		    if (dispPtr->grabWinPtr != NULL) {
			*othergrab = 1;
			return 0;
		    }
		}
	    }
	    mainInfo = mainInfo->nextPtr;
	}
	return 1;
    }

    return 0;
}

void
SdlTkDirtyAll(Window w)
{
    _Window *_w = (_Window *) w;

    SdlTkDirtyArea(w, 0, 0, _w->parentWidth, _w->parentHeight);
}

void
SdlTkDirtyArea(Window w, int x, int y, int width, int height)
{
    _Window *_w = (_Window *) w;
    int xOff, yOff;
    _Window *top = SdlTkToplevelForWindow(_w, &xOff, &yOff);
    XRectangle rect;
    Region rgn = SdlTkRgnPoolGet();

    rect.x = x;
    rect.y = y;
    rect.width = width;
    rect.height = height;
    XUnionRectWithRegion(&rect, rgn, rgn);
    XIntersectRegion(_w->visRgn, rgn, rgn);
    XOffsetRegion(rgn, xOff, yOff);

    XUnionRegion(rgn, top->dirtyRgn, top->dirtyRgn);
    SdlTkRgnPoolFree(rgn);
}

void
SdlTkDirtyRegion(Window w, Region rgn)
{
    _Window *_w = (_Window *) w;
    int xOff, yOff;
    _Window *top = SdlTkToplevelForWindow(_w, &xOff, &yOff);
    Region r = SdlTkRgnPoolGet();

    XIntersectRegion(_w->visRgn, rgn, r);
    XOffsetRegion(r, xOff, yOff);
    XUnionRegion(top->dirtyRgn, r, top->dirtyRgn);
    SdlTkRgnPoolFree(r);
}

#ifdef SDL_GESTURES_ARE_WORKING

struct grwops {
    SDL_RWops rwops;
    Tcl_Interp *interp;
    Tcl_Channel channel;
};

static Sint64
RW_Size(SDL_RWops *rwops)
{
    struct grwops *rws = (struct grwops *) rwops;
    Sint64 pos, oldpos;

    oldpos = Tcl_Tell(rws->channel);
    pos = Tcl_Seek(rws->channel, 0, SEEK_END);
    Tcl_Seek(rws->channel, oldpos, SEEK_SET);
    return pos;
}

static Sint64
RW_Seek(SDL_RWops *rwops, Sint64 offset, int whence)
{
    struct grwops *rws = (struct grwops *) rwops;

    switch (whence) {
	case RW_SEEK_CUR:
	    whence = SEEK_CUR;
	    break;
	case RW_SEEK_END:
	    whence = SEEK_END;
	    break;
	case RW_SEEK_SET:
	    whence = SEEK_SET;
	    break;
	default:
	    return -1;
    }
    return Tcl_Seek(rws->channel, offset, whence);
}

static size_t
RW_Read(SDL_RWops *rwops, void *buf, size_t size, size_t num)
{
    struct grwops *rws = (struct grwops *) rwops;
    int ret;

    if (size == 0) {
	return 0;
    }
    ret = Tcl_Read(rws->channel, (char *) buf, size * num);
    if (ret == -1) {
	return 0;
    }
    return ret / size;
}

static size_t
RW_Write(SDL_RWops *rwops, const void *buf, size_t size, size_t num)
{
    struct grwops *rws = (struct grwops *) rwops;
    int ret;

    if (size == 0) {
	return 0;
    }
    ret = Tcl_Write(rws->channel, (CONST char *) buf, size * num);
    if (ret == -1) {
	return 0;
    }
    return ret / size;
}

static int
RW_Close(SDL_RWops *rwops)
{
    struct grwops *rws = (struct grwops *) rwops;

    Tcl_Close(rws->interp, rws->channel);
    ckfree((char *) rws);
    return 0;
}

static SDL_RWops *
RW_Open(Tcl_Interp *interp, CONST char *filename, int iswr)
{
    struct grwops *rws;
    Tcl_Channel channel;

    channel = Tcl_OpenFileChannel(interp, filename, iswr ? "w" : "r", 0644);
    if (channel == NULL) {
	return NULL;
    }
    if (Tcl_SetChannelOption(interp, channel, "-encoding", "binary") !=
	TCL_OK ||
	Tcl_SetChannelOption(interp, channel, "-translation", "binary") !=
	TCL_OK) {
	Tcl_Close(interp, channel);
	return NULL;
    }
    rws = (struct grwops *) ckalloc(sizeof (struct grwops));
    memset(rws, 0, sizeof (struct grwops));
    rws->rwops.size = RW_Size;
    rws->rwops.seek = RW_Seek;
    rws->rwops.read = RW_Read;
    rws->rwops.write = RW_Write;
    rws->rwops.close = RW_Close;
    rws->interp = interp;
    rws->channel = channel;
    return &rws->rwops;
}

#endif

int
Tk_SdlTkObjCmd(ClientData clientData, Tcl_Interp *interp, int objc,
	       Tcl_Obj *const objv[])
{
    Tk_Window tkwin = (Tk_Window) clientData;
    static const char *optionStrings[] = {
	"expose", "paintvisrgn",
	"textinput", "accelerometer", "powerinfo",
#ifdef SDL_GESTURES_ARE_WORKING
	"gesture",
#endif
	NULL
    };
    enum options {
	CMD_EXPOSE, CMD_PAINT_VISRGN, CMD_TEXTINPUT, CMD_ACCELEROMETER,
	CMD_POWERINFO,
#ifdef SDL_GESTURES_ARE_WORKING
	CMD_GESTURE,
#endif
    };
#ifdef SDL_GESTURES_ARE_WORKING
    static const char *goptStrings[] = {
	"record", "save", "load", NULL
    };
    enum gopts {
	CMD_GESTURE_RECORD, CMD_GESTURE_SAVE, CMD_GESTURE_LOAD,
    };
#endif
    int index;

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "option ?arg?");
	return TCL_ERROR;
    }
    if (Tcl_GetIndexFromObj(interp, objv[1], optionStrings, "option", 0,
			    &index) != TCL_OK) {
	return TCL_ERROR;
    }
    switch (index) {
	/* sdltk expose ?win? */
	case CMD_EXPOSE: {
	    int x, y;
	    _Window *_w;
	    Region rgn;
	    XRectangle rect;

	    if (objc > 3) {
		Tcl_WrongNumArgs(interp, 2, objv, "?window?");
		return TCL_ERROR;
	    }

	    if (objc == 3) {
		Tk_Window tkwin2;

		if (TkGetWindowFromObj(interp, tkwin, objv[2],
				       &tkwin2) != TCL_OK) {
		    return TCL_ERROR;
		}
		_w = (_Window *) ((TkWindow *) tkwin2)->window;
	    } else {
		(void) SDL_GetMouseState(&x, &y);
		_w = SdlTkPointToWindow((_Window *) SdlTkX_screen->root,
					x, y, True);
	    }
	    rgn = SdlTkRgnPoolGet();
	    rect.x = rect.y = 0;
	    rect.width = _w->parentWidth;
	    rect.height = _w->parentHeight;
	    XUnionRectWithRegion(&rect, rgn, rgn);
	    XIntersectRegion(_w->visRgn, rgn, rgn);
	    if (IS_ROOT(_w)) {
		XUnionRegion(rgn, screen_dirty_region, screen_dirty_region);
		SdlTkScreenChanged();
	    } else if (_w->tkwin != NULL) {
		SdlTkGfxExposeRegion((Window) _w, rgn);
	    }
	    SdlTkRgnPoolFree(rgn);
	    break;
	}

	/* sdltk paintvisrgn ?win? */
	case CMD_PAINT_VISRGN: {
	    int x, y;
	    _Window *_w;
	    Region r;

	    if (objc > 3) {
		Tcl_WrongNumArgs(interp, 2, objv, "?window?");
		return TCL_ERROR;
	    }
	    if (objc == 3) {
		Tk_Window tkwin2;

		if (TkGetWindowFromObj(interp, tkwin, objv[2],
				       &tkwin2) != TCL_OK) {
		    return TCL_ERROR;
		}
		_w = (_Window *) ((TkWindow *) tkwin2)->window;
	    } else {
		(void) SDL_GetMouseState(&x, &y);
		_w = SdlTkPointToWindow((_Window *) SdlTkX_screen->root,
					x, y, True);
	    }

	    r = SdlTkGetVisibleRegion(_w);
	    SdlTkGfxFillRegion((Drawable) _w, r, 0x0000FF88);
	    SDL_UpdateWindowSurface(SdlTk_sdlscreen);
	    break;
	}

	case CMD_TEXTINPUT: {
	    int flag = 0;

	    if (objc > 5) {
tiWrongArgs:
		Tcl_WrongNumArgs(interp, 2, objv, "?onoff? ?x y?");
		return TCL_ERROR;
	    }
	    if (objc == 3 || objc == 5) {
		if (Tcl_GetBooleanFromObj(interp, objv[2], &flag) != TCL_OK) {
		    return TCL_ERROR;
		}
		if (!SDL_HasScreenKeyboardSupport()) {
		    return TCL_OK;
		}
		if (flag) {
		    if (objc == 5) {
			int x, y;
			SDL_Rect r;

			if (Tcl_GetIntFromObj(interp, objv[3], &x) != TCL_OK ||
			    Tcl_GetIntFromObj(interp, objv[4], &y) != TCL_OK) {
			    return TCL_ERROR;
			}
			x -= 32;
			if (x < 0) {
			    x = 0;
			}
			y -= 32;
			if (y < 0) {
			    y = 0;
			}
			r.x = x;
			r.y = y;
			r.w = 256;
			r.h = 128;
			SDL_SetTextInputRect(&r);
		    }
		    SDL_StartTextInput();
		} else {
		    SDL_StopTextInput();
		}
		return TCL_OK;
	    } else if (objc == 4) {
		goto tiWrongArgs;
	    }
	    if (SDL_HasScreenKeyboardSupport()) {
		flag = SDL_IsScreenKeyboardShown(SdlTk_sdlscreen);
	    }
	    Tcl_SetBooleanObj(Tcl_GetObjResult(interp), flag);
	    break;
	}

	case CMD_ACCELEROMETER: {
	    int flag = 0;

	    if (objc > 3) {
		Tcl_WrongNumArgs(interp, 2, objv, "?onoff?");
		return TCL_ERROR;
	    }
	    if (objc == 3) {
		if (Tcl_GetBooleanFromObj(interp, objv[2], &flag) != TCL_OK) {
		    return TCL_ERROR;
		}
#ifdef ANDROID
		SDL_EventState(SDL_JOYAXISMOTION,
			       flag ? SDL_ENABLE : SDL_DISABLE);
#endif
		break;
	    }
#ifdef ANDROID
	    flag = SDL_EventState(SDL_JOYAXISMOTION, SDL_QUERY);
#endif
	    Tcl_SetBooleanObj(Tcl_GetObjResult(interp), flag);
	    break;
	}

	case CMD_POWERINFO: {
	    SDL_PowerState pst;
	    int secs, pct;
	    char buf[32];

	    if (objc != 2) {
		Tcl_WrongNumArgs(interp, 2, objv, "");
		return TCL_ERROR;
	    }
	    pst = SDL_GetPowerInfo(&secs, &pct);
	    Tcl_AppendElement(interp, "state");
	    switch (pst) {
	    case SDL_POWERSTATE_ON_BATTERY:
		Tcl_AppendElement(interp, "onbattery");
		break; 
	    case SDL_POWERSTATE_NO_BATTERY:
		Tcl_AppendElement(interp, "nobattery");
		break; 
	    case SDL_POWERSTATE_CHARGING:
		Tcl_AppendElement(interp, "charging");
		break; 
	    case SDL_POWERSTATE_CHARGED:
		Tcl_AppendElement(interp, "charged");
		break; 
	    default:
		Tcl_AppendElement(interp, "unknown");
		break; 
	    }
	    Tcl_AppendElement(interp, "seconds");
	    sprintf(buf, "%d", secs);
	    Tcl_AppendElement(interp, buf);
	    Tcl_AppendElement(interp, "percent");
	    sprintf(buf, "%d", pct);
	    Tcl_AppendElement(interp, buf);
	    break;
	}

#ifdef SDL_GESTURES_ARE_WORKING
	case CMD_GESTURE: {
	    SDL_RWops *rw;

	    if (Tcl_GetIndexFromObj(interp, objv[2], goptStrings, "option", 0,
				    &index) != TCL_OK) {
		return TCL_ERROR;
	    }
	    switch (index) {
		case CMD_GESTURE_RECORD:
		    if (objc > 3) {
			Tcl_WrongNumArgs(interp, 3, objv, NULL);
			return TCL_ERROR;
		    }
		    SDL_RecordGesture(-1);
		    break;
		case CMD_GESTURE_SAVE:
		    if (objc != 4) {
			Tcl_WrongNumArgs(interp, 3, objv, "filename");
			return TCL_ERROR;
		    }
		    rw = RW_Open(interp, Tcl_GetString(objv[3]), 1);
		    if (rw == NULL) {
			return TCL_ERROR;
		    }
		    SDL_SaveAllDollarTemplates(rw);
		    break;
		case CMD_GESTURE_LOAD:
		    if (objc != 4) {
			Tcl_WrongNumArgs(interp, 3, objv, "filename");
			return TCL_ERROR;
		    }
		    rw = RW_Open(interp, Tcl_GetString(objv[3]), 0);
		    if (rw == NULL) {
			return TCL_ERROR;
		    }
		    SDL_LoadDollarTemplates(-1, rw);
		    break;
	    }
	    break;
	}
#endif
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TkpGetMS --
 *
 *	Return a relative time in milliseconds.  It doesn't matter
 *	when the epoch was.
 *
 * Results:
 *	Number of milliseconds.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

unsigned long
TkpGetMS(void)
{
    Tcl_Time now;
    TclpGetTime(&now);
    return now.usec / 1000;
}

/* NOTE: If this changes, update XAllocColor */
unsigned long
TkpGetPixel(XColor *color)
{
    Uint8 r = (color->red / 65535.0) * 255.0;
    Uint8 g = (color->green / 65535.0) * 255.0;
    Uint8 b = (color->blue / 65535.0) * 255.0;

    /* All SDL_gfx xxxColor() routines expect RGBA format */
    return SDL_MapRGB(SdlTk_sdlsurf->format, r, g, b);
}

/*
 *----------------------------------------------------------------------
 *
 * TkpSetCapture --
 *
 *	This function captures the mouse so that all future events
 *	will be reported to this window, even if the mouse is outside
 *	the window.  If the specified window is NULL, then the mouse
 *	is released. 
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Sets the capture flag and captures the mouse.
 *
 *----------------------------------------------------------------------
 */

void
TkpSetCapture(TkWindow *winPtr)
{
    tk_capture_window = winPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * TkpSetCursor --
 *
 *	Set the global cursor.  If the cursor is None, then use the
 *	default Tk cursor.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Changes the mouse cursor.
 *
 *----------------------------------------------------------------------
 */

void
TkpSetCursor(TkpCursor cursor)
{
}

void
SdlTkClearPointer(_Window *_w)
{
    if (_w != NULL && _w->tkwin != NULL) {
	if (tk_pointergrab_window == _w->tkwin) {
	    tk_pointergrab_window = NULL;
	}
	if (tk_capture_window == _w->tkwin) {
	    tk_capture_window = NULL;
	}
	if (tk_mouse_window == _w) {
	    tk_mouse_window = NULL;
	}
    }
}

