#include <tcl.h>
#include <tclRfcomm.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#ifdef ANDROID
#include <jni.h>
#else
#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>
#endif

#ifdef ANDROID

EXTERN JNIEnv *		Tkborg_JNIEnv(void);

struct sockaddr_rc {
    struct {
	unsigned char b[6];
    } rc_bdaddr;
    unsigned char rc_channel;
    unsigned char rc_filler;
    struct {
	unsigned char b[16];
    } rc_uuid;
    char rc_name[256];
};

static const unsigned char spp_uuid[16] = {
    0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
    0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb
};

#endif

typedef struct RfcommState {
    Tcl_Channel channel;	/* Channel associated with this file. */
    int fd;			/* The socket itself. */
#define ASYNC_SOCKET	(1<<0)	/* Asynchronous socket. */
#define ASYNC_CONNECT	(1<<1)	/* Async connect in progress. */
    int flags;			/* ORed combination of ASYNC_* bits. */
    Tcl_TcpAcceptProc *acceptProc;
				/* Proc to call on accept. */
    ClientData acceptProcData;	/* The data for the accept proc. */
#ifdef ANDROID
    jobject jbt;
    int fd2;
    struct sockaddr_rc a_peer;
    struct sockaddr_rc a_sock;
#endif
} RfcommState;

typedef struct AcceptCallback {
    char *script;			/* Script to invoke. */
    Tcl_Interp *interp;			/* Interpreter in which to run it. */
} AcceptCallback;

#ifdef ANDROID

static char *
FormatUUID(struct sockaddr_rc *addr, char *buf)
{
    sprintf(buf, "%02X%02X%02X%02X-%02X%02X-%02X%02X-%02X%02X-"
	    "%02X%02X%02X%02X%02X%02X",
	    addr->rc_uuid.b[0], addr->rc_uuid.b[1],
	    addr->rc_uuid.b[2], addr->rc_uuid.b[3],
	    addr->rc_uuid.b[4], addr->rc_uuid.b[5],
	    addr->rc_uuid.b[6], addr->rc_uuid.b[7],
	    addr->rc_uuid.b[8], addr->rc_uuid.b[9],
	    addr->rc_uuid.b[10], addr->rc_uuid.b[11],
	    addr->rc_uuid.b[12], addr->rc_uuid.b[13],
	    addr->rc_uuid.b[14], addr->rc_uuid.b[15]);
    return buf;
}

static int
GetUUID(struct sockaddr_rc *addr, const char *str)
{
    int i, k, val;
    char buf[16];

    memset(&addr->rc_uuid, 0, sizeof (addr->rc_uuid));
    i = 0;
    k = 0;
    while (*str != '\0') {
	if (strchr("0123456789ABCDEFabcdef", *str) != NULL) {
	    buf[k++] = *str;
	    if (k > 1) {
		buf[k] = '\0';
		k = 0;
		sscanf(buf, "%x", &val);
		addr->rc_uuid.b[i++] = val;
		if (i > sizeof (addr->rc_uuid.b)) {
		    return 1;
		}
	    }
	}
	++str;
    }
    return 0;
}

#endif

static char *
FormatBTAddr(struct sockaddr_rc *addr, char *buf)
{
    sprintf(buf, "%02X:%02X:%02X:%02X:%02X:%02X",
	    addr->rc_bdaddr.b[5], addr->rc_bdaddr.b[4],
	    addr->rc_bdaddr.b[3], addr->rc_bdaddr.b[2],
	    addr->rc_bdaddr.b[1], addr->rc_bdaddr.b[0]);
    return buf;
}

static int
GetBTAddr(struct sockaddr_rc *addr, const char *str)
{
    int i, bda[6];

    memset(bda, 0, sizeof (bda));
    if (sscanf(str, "%x:%x:%x:%x:%x:%x", bda + 5, bda + 4,
	       bda + 3, bda + 2, bda + 1, bda + 0) < 6) {
	return 0;
    }
    for (i = 0; i < 6; i++) {
	addr->rc_bdaddr.b[i] = bda[i];
    }
    return 1;
}

static int
WaitForFile(int fd, int mask, int timeout)
{
    Tcl_Time abortTime = {0, 0}, now;
    struct timeval blockTime, *timeoutPtr;
    int numFound, result = 0;
    fd_set readableMask;
    fd_set writableMask;
    fd_set exceptionalMask;

    if (timeout > 0) {
	Tcl_GetTime(&now);
	abortTime.sec = now.sec + timeout/1000;
	abortTime.usec = now.usec + (timeout%1000)*1000;
	if (abortTime.usec >= 1000000) {
	    abortTime.usec -= 1000000;
	    abortTime.sec += 1;
	}
	timeoutPtr = &blockTime;
    } else if (timeout == 0) {
	timeoutPtr = &blockTime;
	blockTime.tv_sec = 0;
	blockTime.tv_usec = 0;
    } else {
	timeoutPtr = NULL;
    }

    FD_ZERO(&readableMask);
    FD_ZERO(&writableMask);
    FD_ZERO(&exceptionalMask);

    while (1) {
	if (timeout > 0) {
	    blockTime.tv_sec = abortTime.sec - now.sec;
	    blockTime.tv_usec = abortTime.usec - now.usec;
	    if (blockTime.tv_usec < 0) {
		blockTime.tv_sec -= 1;
		blockTime.tv_usec += 1000000;
	    }
	    if (blockTime.tv_sec < 0) {
		blockTime.tv_sec = 0;
		blockTime.tv_usec = 0;
	    }
	}

	if (mask & TCL_READABLE)  {
	    FD_SET(fd, &readableMask);
	}
	if (mask & TCL_WRITABLE)  {
	    FD_SET(fd, &writableMask);
	}
	if (mask & TCL_EXCEPTION) {
	    FD_SET(fd, &exceptionalMask);
	}

	numFound = select(fd + 1, &readableMask, &writableMask,
		&exceptionalMask, timeoutPtr);
	if (numFound == 1) {
	    if (FD_ISSET(fd, &readableMask))   {
		result |= TCL_READABLE;
	    }
	    if (FD_ISSET(fd, &writableMask))  {
		result |= TCL_WRITABLE;
	    }
	    if (FD_ISSET(fd, &exceptionalMask)) { 
		result |= TCL_EXCEPTION;
	    }
	    result &= mask;
	    if (result) {
		break;
	    }
	}
	if (timeout == 0) {
	    break;
	}
	if (timeout < 0) {
	    continue;
	}

	Tcl_GetTime(&now);
	if ((abortTime.sec < now.sec)
		|| ((abortTime.sec == now.sec)
		&& (abortTime.usec <= now.usec))) {
	    break;
	}
    }
    return result;
}

static int
WaitForConnect(RfcommState *statePtr, int *errorCodePtr)
{
    int timeOut, state, flags;

    if (statePtr->flags & ASYNC_CONNECT) {
	if (statePtr->flags & ASYNC_SOCKET) {
	    timeOut = 0;
	} else {
	    timeOut = -1;
	}
	errno = 0;
	state = WaitForFile(statePtr->fd,
			    TCL_WRITABLE | TCL_EXCEPTION, timeOut);
	if (!(statePtr->flags & ASYNC_SOCKET)) {
	    flags = fcntl(statePtr->fd, F_GETFL);
	    flags &= (~(O_NONBLOCK));
	    (void) fcntl(statePtr->fd, F_SETFL, flags);
	}
	if (state & TCL_EXCEPTION) {
	    return -1;
	}
	if (state & TCL_WRITABLE) {
	    statePtr->flags &= (~(ASYNC_CONNECT));
	} else if (timeOut == 0) {
	    *errorCodePtr = errno = EWOULDBLOCK;
	    return -1;
	}
    }
    return 0;
}

static int
RfcommInputProc(ClientData instanceData, char *buf, int bufSize,
		int *errorCodePtr)
{
    RfcommState *statePtr = (RfcommState *) instanceData;
    int bytesRead, state;

    *errorCodePtr = 0;
    state = WaitForConnect(statePtr, errorCodePtr);
    if (state != 0) {
	return -1;
    }
    bytesRead = recv(statePtr->fd, buf, (size_t) bufSize, MSG_NOSIGNAL);
    if (bytesRead > -1) {
	return bytesRead;
    }
    if (errno == ECONNRESET) {
	return 0;
    }
    *errorCodePtr = errno;
    return -1;
}

static int
RfcommOutputProc(ClientData instanceData, CONST char *buf, int toWrite,
		 int *errorCodePtr)
{
    RfcommState *statePtr = (RfcommState *) instanceData;
    int written;
    int state;				/* Of waiting for connection. */

    *errorCodePtr = 0;
    state = WaitForConnect(statePtr, errorCodePtr);
    if (state != 0) {
	return -1;
    }
    written = send(statePtr->fd, buf, (size_t) toWrite, MSG_NOSIGNAL);
    if (written > -1) {
	return written;
    }
    *errorCodePtr = errno;
    return -1;
}

static int
RfcommCloseProc(ClientData instanceData, Tcl_Interp *interp)
{
    RfcommState *statePtr = (RfcommState *) instanceData;
    int errorCode = 0;
#ifdef ANDROID
    JNIEnv *env = Tkborg_JNIEnv();
#endif

    Tcl_DeleteFileHandler(statePtr->fd);
    if (close(statePtr->fd) < 0) {
	errorCode = errno;
    }
#ifdef ANDROID
    if (statePtr->jbt != NULL) {
	jmethodID mid;

	mid = (*env)->GetMethodID(env,
				  (*env)->GetObjectClass(env, statePtr->jbt),
				  "close", "()V");
	if (mid != NULL) {
	    (*env)->CallVoidMethod(env, statePtr->jbt, mid);
	}
	(*env)->DeleteGlobalRef(env, statePtr->jbt);
	statePtr->jbt = NULL;
    }
    if (statePtr->fd2 >= 0) {
	shutdown(statePtr->fd2, 2);
	close(statePtr->fd2);
    }
#endif
    ckfree((char *) statePtr);
    return errorCode;
}

static int
RfcommGetOptionProc(ClientData instanceData, Tcl_Interp *interp,
		    CONST char *optionName, Tcl_DString *dsPtr)
{
    RfcommState *statePtr = (RfcommState *) instanceData;
    struct sockaddr_rc sockname;
    struct sockaddr_rc peername;
    socklen_t size = sizeof (struct sockaddr_rc);
    size_t len = 0;
    char buf[TCL_INTEGER_SPACE * 32];

    if (optionName != (char *) NULL) {
	len = strlen(optionName);
    }

    if ((len > 1) && (optionName[1] == 'e') &&
	    (strncmp(optionName, "-error", len) == 0)) {
	socklen_t optlen = sizeof (int);
	int err, ret;

	ret = getsockopt(statePtr->fd, SOL_SOCKET, SO_ERROR,
		(char *)&err, &optlen);
	if (ret < 0) {
	    err = errno;
	}
	if (err != 0) {
	    Tcl_DStringAppend(dsPtr, Tcl_ErrnoMsg(err), -1);
	}
	return TCL_OK;
    }

    if ((len == 0) ||
	    ((len > 1) && (optionName[1] == 'p') &&
		    (strncmp(optionName, "-peername", len) == 0))) {
#ifdef ANDROID
	peername = statePtr->a_peer;
#else
	if (getpeername(statePtr->fd, (struct sockaddr *) &peername,
		&size) >= 0)
#endif
	{
	    if (len == 0) {
		Tcl_DStringAppendElement(dsPtr, "-peername");
		Tcl_DStringStartSublist(dsPtr);
	    }
	    FormatBTAddr(&peername, buf);
	    Tcl_DStringAppendElement(dsPtr, buf);
	    Tcl_DStringAppendElement(dsPtr, buf);
	    TclFormatInt(buf, peername.rc_channel);
	    Tcl_DStringAppendElement(dsPtr, buf);
	    if (len == 0) {
		Tcl_DStringEndSublist(dsPtr);
	    } else {
		return TCL_OK;
	    }
#ifndef ANDROID
	} else {
	    if (len) {
		if (interp) {
		    Tcl_AppendResult(interp, "can't get peername: ",
			    Tcl_PosixError(interp), (char *) NULL);
		}
		return TCL_ERROR;
	    }
#endif
	}
    }

    if ((len == 0) ||
	    ((len > 1) && (optionName[1] == 's') &&
	    (strncmp(optionName, "-sockname", len) == 0))) {
#ifdef ANDROID
	sockname = statePtr->a_sock;
#else
	if (getsockname(statePtr->fd, (struct sockaddr *) &sockname,
		&size) >= 0)
#endif
	{
	    if (len == 0) {
		Tcl_DStringAppendElement(dsPtr, "-sockname");
		Tcl_DStringStartSublist(dsPtr);
	    }
	    FormatBTAddr(&sockname, buf);
	    Tcl_DStringAppendElement(dsPtr, buf);
	    Tcl_DStringAppendElement(dsPtr, buf);
	    TclFormatInt(buf, peername.rc_channel);
	    Tcl_DStringAppendElement(dsPtr, buf);
	    if (len == 0) {
		Tcl_DStringEndSublist(dsPtr);
	    } else {
		return TCL_OK;
	    }
#ifndef ANDROID
	} else {
	    if (interp) {
		Tcl_AppendResult(interp, "can't get sockname: ",
			Tcl_PosixError(interp), (char *) NULL);
	    }
	    return TCL_ERROR;
#endif
	}
    }

#ifdef ANDROID
    if ((len == 0) ||
	    ((len > 1) && (optionName[1] == 'u') &&
	    (strncmp(optionName, "-uuid", len) == 0))) {
	sockname = statePtr->a_sock;
	if (len == 0) {
	    Tcl_DStringAppendElement(dsPtr, "-uuid");
	    Tcl_DStringStartSublist(dsPtr);
	}
	FormatUUID(&sockname, buf);
	Tcl_DStringAppendElement(dsPtr, buf);
	if (len == 0) {
	    Tcl_DStringEndSublist(dsPtr);
	} else {
	    return TCL_OK;
	}
    }
#endif

    if (len > 0) {
	return Tcl_BadChannelOption(interp, optionName,
#ifdef ANDROID
				    "uuid "
#endif
				    "peername sockname");
    }

    return TCL_OK;
}

static void
RfcommWatchProc(ClientData instanceData, int mask)
{
    RfcommState *statePtr = (RfcommState *) instanceData;

    if (!statePtr->acceptProc) {
	if (mask) {
	    Tcl_CreateFileHandler(statePtr->fd, mask,
		    (Tcl_FileProc *) Tcl_NotifyChannel,
		    (ClientData) statePtr->channel);
	} else {
	    Tcl_DeleteFileHandler(statePtr->fd);
	}
    }
}

static int
RfcommGetHandleProc(ClientData instanceData, int direction,
		    ClientData *handlePtr)
{
    RfcommState *statePtr = (RfcommState *) instanceData;

    *handlePtr = (ClientData)statePtr->fd;
    return TCL_OK;
}

static int
RfcommBlockModeProc(ClientData instanceData, int mode)
{
    RfcommState *statePtr = (RfcommState *) instanceData;
    int setting;

    setting = fcntl(statePtr->fd, F_GETFL);
    if (mode == TCL_MODE_BLOCKING) {
	statePtr->flags &= (~(ASYNC_SOCKET));
	setting &= (~(O_NONBLOCK));
    } else {
	statePtr->flags |= ASYNC_SOCKET;
	setting |= O_NONBLOCK;
    }
    if (fcntl(statePtr->fd, F_SETFL, setting) < 0) {
	return errno;
    }
    return 0;
}

static Tcl_ChannelType rfcommChannelType = {
    "rfcomm",			/* Type name. */
    TCL_CHANNEL_VERSION_4,	/* v4 channel */
    RfcommCloseProc,		/* Close proc. */
    RfcommInputProc,		/* Input proc. */
    RfcommOutputProc,		/* Output proc. */
    NULL,			/* Seek proc. */
    NULL,			/* Set option proc. */
    RfcommGetOptionProc,	/* Get option proc. */
    RfcommWatchProc,		/* Initialize notifier. */
    RfcommGetHandleProc,	/* Get OS handles out of channel. */
    NULL,			/* close2proc. */
    RfcommBlockModeProc,	/* Set blocking or non-blocking mode.*/
    NULL,			/* flush proc. */
    NULL,			/* handler proc. */
    NULL,			/* wide seek proc. */
    NULL,			/* thread action proc. */
};

#ifndef ANDROID
static int
CreateSocketAddress(struct sockaddr_rc *sockaddrPtr, CONST char *host,
		    int port)
{
    memset(sockaddrPtr, 0, sizeof (struct sockaddr_rc));
    sockaddrPtr->rc_family = AF_BLUETOOTH;
    sockaddrPtr->rc_channel = port;
    if (host == NULL) {
	memset(&sockaddrPtr->rc_bdaddr, 0xff, sizeof (sockaddrPtr->rc_bdaddr));
	return 1;
    }
    return GetBTAddr(sockaddrPtr, host);
}
#endif

static RfcommState *
CreateSocket(Tcl_Interp *interp, int port, CONST char *host,
	     int server, CONST char *myaddr, int myport, int async)
{
    int status, sock, asyncConnect, curState, origState;
    struct sockaddr_rc sockaddr;
    struct sockaddr_rc mysockaddr;
    RfcommState *statePtr;
#ifdef ANDROID
    int argc;
    CONST char **argv;
    int fd[2];
    char buf[256];
    JNIEnv *env = Tkborg_JNIEnv();
    jfieldID fid;
    jmethodID mid;
    jclass cls = NULL;
    jobject jfd = NULL, jbt;
    jstring btaddr = NULL, btuuid = NULL;
    jthrowable exc;
#endif

    sock = -1;
    origState = 0;
#ifdef ANDROID
    memset(&sockaddr, 0, sizeof (sockaddr));
    memset(&mysockaddr, 0, sizeof (mysockaddr));
    if ((host == NULL) ||
	(Tcl_SplitList(interp, host, &argc, &argv) != TCL_OK)) {
	errno = EINVAL;
	goto addressError;
    }
    if (argc < 1) {
	ckfree((char *) argv);
	errno = EINVAL;
	goto addressError;
    }
    if (server) {
	/* UUID + name (optional) */
	if (!GetUUID(&sockaddr, argv[0])) {
	    ckfree((char *) argv);
	    errno = EINVAL;
	    goto addressError;
	}
	btuuid = (*env)->NewStringUTF(env, FormatUUID(&sockaddr, buf));
	if (argc > 1) {
	    strncpy(sockaddr.rc_name, argv[1], sizeof (sockaddr.rc_name));
	    sockaddr.rc_name[sizeof (sockaddr.rc_name) - 1] = '\0';
	}
	btaddr = (*env)->NewStringUTF(env, sockaddr.rc_name);
    } else {
	/* BT address + UUID (optional) */
	if (!GetBTAddr(&sockaddr, argv[0])) {
	    ckfree((char *) argv);
	    errno = EINVAL;
	    goto addressError;
	}
	if (argc > 1) {
	    if (!GetUUID(&sockaddr, argv[1])) {
		ckfree((char *) argv);
		errno = EINVAL;
		goto addressError;
	    }
	} else {
	    memcpy(&sockaddr.rc_uuid, spp_uuid, sizeof (sockaddr.rc_uuid));
	}
	btaddr = (*env)->NewStringUTF(env, FormatBTAddr(&sockaddr, buf));
	btuuid = (*env)->NewStringUTF(env, FormatUUID(&sockaddr, buf));
    }
    ckfree((char *) argv);
#else
    if (!CreateSocketAddress(&sockaddr, host, port)) {
	goto addressError;
    }
    if ((myaddr != NULL || myport != 0) &&
	    !CreateSocketAddress(&mysockaddr, myaddr, myport)) {
	goto addressError;
    }
#endif

#ifdef ANDROID
    if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd) < 0) {
	goto addressError;
    }
    sock = fd[1];
    cls = (*env)->FindClass(env, "java/io/FileDescriptor");
    if (cls == NULL) {
	close(fd[0]);
	errno = ENOTSUP;
	goto addressError;
    }
    fid = (*env)->GetFieldID(env, cls, "descriptor", "I");
    if (fid == NULL) {
	close(fd[0]);
	errno = ENOTSUP;
	goto addressError;
    }
    mid = (*env)->GetMethodID(env, cls, "<init>", "()V");
    if (fid == NULL) {
	(*env)->DeleteLocalRef(env, cls);
	close(fd[0]);
	errno = ENOTSUP;
	goto addressError;
    }
    jfd = (*env)->NewObject(env, cls, mid);
    if (jfd == NULL) {
	(*env)->DeleteLocalRef(env, cls);
	close(fd[0]);
	errno = ENOTSUP;
	goto addressError;
    }
    (*env)->SetIntField(env, jfd, fid, fd[0]);
    (*env)->DeleteLocalRef(env, cls);
    cls = NULL;
    if (server) {
	cls = (*env)->FindClass(env, "tk/tcl/wish/BTServer");
    } else {
	cls = (*env)->FindClass(env, "tk/tcl/wish/BTClient");
    }
    if (cls == NULL) {
	errno = ENOTSUP;
	goto addressError;
    }
    mid = (*env)->GetMethodID(env, cls, "<init>",
				  "(Ljava/io/FileDescriptor;Ljava/lang/String;Ljava/lang/String;)V");
    if (mid == NULL) {
	errno = ENOTSUP;
	goto addressError;
    }
    jbt = (*env)->NewObject(env, cls, mid, jfd, btaddr, btuuid);
    if (jbt == NULL) {
	errno = ENOTSUP;
	goto addressError;
    }
    exc = (*env)->ExceptionOccurred(env);
    if (exc != NULL) {
	(*env)->DeleteLocalRef(env, jbt);
	(*env)->DeleteLocalRef(env, exc);
	(*env)->ExceptionClear(env);
	errno = ENOTCONN;
	goto addressError;
    }
    (*env)->DeleteLocalRef(env, jfd);
    (*env)->DeleteLocalRef(env, cls);
#else
    sock = socket(PF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
#endif
    if (sock < 0) {
	goto addressError;
    }

    fcntl(sock, F_SETFD, FD_CLOEXEC);
#ifdef ANDROID
    fcntl(fd[0], F_SETFD, FD_CLOEXEC);
#endif

    asyncConnect = 0;
    status = 0;
#ifndef ANDROID
    if (server) {
	status = 1;
	(void) setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *) &status,
		sizeof (status));
	status = bind(sock, (struct sockaddr *) &sockaddr,
		sizeof (struct sockaddr));
	if (status != -1) {
	    status = listen(sock, SOMAXCONN);
	}
    } else {
	if (myaddr != NULL || myport != 0) { 
	    curState = 1;
	    (void) setsockopt(sock, SOL_SOCKET, SO_REUSEADDR,
		    (char *) &curState, sizeof (curState));
	    status = bind(sock, (struct sockaddr *) &mysockaddr,
		    sizeof (struct sockaddr));
	    if (status < 0) {
		goto bindError;
	    }
	}
	if (async) {
	    origState = fcntl(sock, F_GETFL);
	    curState = origState | O_NONBLOCK;
	    status = fcntl(sock, F_SETFL, curState);
	} else {
	    status = 0;
	}
	if (status > -1) {
	    status = connect(sock, (struct sockaddr *) &sockaddr,
		    sizeof (sockaddr));
	    if (status < 0) {
		if (errno == EINPROGRESS) {
		    asyncConnect = 1;
		    status = 0;
		}
	    } else {
		if (async) {
		    origState = fcntl(sock, F_GETFL);
		    curState = origState & ~(O_NONBLOCK);
		    status = fcntl(sock, F_SETFL, curState);
		}
	    }
	}
    }

bindError:
    if (status < 0) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, "couldn't open socket: ",
		    Tcl_PosixError(interp), (char *) NULL);
	}
	if (sock != -1) {
	    close(sock);
	}
	return NULL;
    }
#endif

    statePtr = (RfcommState *) ckalloc((unsigned) sizeof (RfcommState));
    statePtr->flags = 0;
    if (asyncConnect) {
	statePtr->flags = ASYNC_CONNECT;
    }
    statePtr->fd = sock;
#ifdef ANDROID
    statePtr->fd2 = fd[0];
    statePtr->jbt = (*env)->NewGlobalRef(env, jbt);
    (*env)->DeleteLocalRef(env, jbt);
    if (server) {
	memset(&statePtr->a_peer, 0, sizeof (statePtr->a_peer));
	origState = fcntl(sock, F_GETFL);
	curState = origState | O_NONBLOCK;
	status = fcntl(sock, F_SETFL, curState);
    } else {
	statePtr->a_peer = sockaddr;
    }
    statePtr->a_sock = sockaddr;
#endif
    return statePtr;

addressError:
#ifdef ANDROID
    if (cls != NULL) {
	(*env)->DeleteLocalRef(env, cls);
    }
    if (jfd != NULL) {
	(*env)->DeleteLocalRef(env, jfd);
	close(fd[0]);
    }
    if (btaddr != NULL) {
	(*env)->DeleteLocalRef(env, btaddr);
    }
    if (btuuid != NULL) {
	(*env)->DeleteLocalRef(env, btuuid);
    }
#endif
    if (sock != -1) {
	close(sock);
    }
    if (interp != NULL) {
	Tcl_AppendResult(interp, "couldn't open socket: ",
		Tcl_PosixError(interp), (char *) NULL);
    }
    return NULL;
}

static void
RfcommAccept(ClientData data, int mask)
{
    RfcommState *sockState, *newSockState;
    int newsock;
    struct sockaddr_rc addr;
    socklen_t len;
    char channelName[16 + TCL_INTEGER_SPACE];
#ifdef ANDROID
    const char *str;
    int fd[2];
    JNIEnv *env = Tkborg_JNIEnv();
    jfieldID fid;
    jmethodID mid;
    jclass cls = NULL;
    jobject jfd = NULL;
    jobject jbt = NULL;
    jthrowable exc;
#endif

    sockState = (RfcommState *) data;

#ifdef ANDROID
    while (1) {
	int n = recv(sockState->fd, channelName, sizeof (channelName),
		     MSG_NOSIGNAL);

	if (n == 0) {
	    return;
	}
	if (n < 0) {
	    if ((errno == EWOULDBLOCK) || (errno == EAGAIN)) {
		break;
	    }
	    return;
	}
    }
    if ((sockState->jbt == NULL) ||
	(socketpair(AF_UNIX, SOCK_STREAM, 0, fd) < 0)) {
	return;
    }
    cls = (*env)->FindClass(env, "java/io/FileDescriptor");
    if (cls == NULL) {
	close(fd[0]);
	close(fd[1]);
	return;
    }
    fid = (*env)->GetFieldID(env, cls, "descriptor", "I");
    if (fid == NULL) {
	(*env)->DeleteLocalRef(env, cls);
	close(fd[0]);
	close(fd[1]);
	return;
    }
    mid = (*env)->GetMethodID(env, cls, "<init>", "()V");
    if (fid == NULL) {
	(*env)->DeleteLocalRef(env, cls);
	close(fd[0]);
	close(fd[1]);
	return;
    }
    jfd = (*env)->NewObject(env, cls, mid);
    if (jfd == NULL) {
	(*env)->DeleteLocalRef(env, cls);
	close(fd[0]);
	close(fd[1]);
	return;
    }
    (*env)->SetIntField(env, jfd, fid, fd[0]);
    (*env)->DeleteLocalRef(env, cls);
    cls = NULL;
    mid = (*env)->GetMethodID(env,
			      (*env)->GetObjectClass(env, sockState->jbt),
			      "accept",
			      "(Ljava/io/FileDescriptor;)Ltk/tcl/wish/BTClient;");
    if (mid == NULL) {
	(*env)->DeleteLocalRef(env, jfd);
	close(fd[1]);
	return;
    }
    jbt = (*env)->CallObjectMethod(env, sockState->jbt, mid, jfd);
    if (jbt == NULL) {
	(*env)->DeleteLocalRef(env, jfd);
	close(fd[1]);
	return;
    }
    exc = (*env)->ExceptionOccurred(env);
    if (exc != NULL) {
	(*env)->DeleteLocalRef(env, jfd);
	(*env)->DeleteLocalRef(env, jbt);
	(*env)->DeleteLocalRef(env, exc);
	(*env)->ExceptionClear(env);
	close(fd[1]);
	return;
    }
    newsock = fd[1];
#else
    len = sizeof (struct sockaddr_rc);
    newsock = accept(sockState->fd, (struct sockaddr *) &addr, &len);
    if (newsock < 0) {
	return;
    }
#endif

    (void) fcntl(newsock, F_SETFD, FD_CLOEXEC);

    newSockState = (RfcommState *) ckalloc((unsigned) sizeof (RfcommState));

    newSockState->flags = 0;
    newSockState->fd = newsock;
    newSockState->acceptProc = NULL;
    newSockState->acceptProcData = NULL;
#ifdef ANDROID
    newSockState->fd2 = fd[0];
    newSockState->jbt = (*env)->NewGlobalRef(env, jbt);
    newSockState->a_sock = sockState->a_sock;
    memset(&addr, 0, sizeof (addr));
    mid = (*env)->GetMethodID(env,
			      (*env)->GetObjectClass(env, jbt),
			      "peername", "()Ljava/lang/String;");
    if (mid != NULL) {
	jstring btaddr = (*env)->CallObjectMethod(env, jbt, mid);

	if (btaddr != NULL) {
	    str = (*env)->GetStringUTFChars(env, btaddr, 0);
	    GetBTAddr(&addr, str);
	    (*env)->ReleaseStringUTFChars(env, btaddr, str);
	    (*env)->DeleteLocalRef(env, btaddr);
	}
    }
    (*env)->DeleteLocalRef(env, jbt);
    newSockState->a_peer = addr;
    memset(&newSockState->a_sock, 0, sizeof (newSockState->a_sock));
#endif

    sprintf(channelName, "rfcomm%d", newsock);
    newSockState->channel = Tcl_CreateChannel(&rfcommChannelType,
	    channelName, (ClientData) newSockState,
	    (TCL_READABLE | TCL_WRITABLE));

    Tcl_SetChannelOption(NULL, newSockState->channel, "-translation",
	    "auto lf");

    if (sockState->acceptProc != NULL) {
	char buf[TCL_INTEGER_SPACE * 16];

	FormatBTAddr(&addr, buf);
	(*sockState->acceptProc)(sockState->acceptProcData,
				 newSockState->channel, buf, addr.rc_channel);
    }
}

static Tcl_Channel
OpenRfcommClient(Tcl_Interp *interp, int port, CONST char *host,
		 CONST char *myaddr, int myport, int async)
{
    RfcommState *statePtr;
    char channelName[16 + TCL_INTEGER_SPACE];

    statePtr = CreateSocket(interp, port, host, 0, myaddr, myport, async);
    if (statePtr == NULL) {
	return NULL;
    }

    statePtr->acceptProc = NULL;
    statePtr->acceptProcData = (ClientData) NULL;

    sprintf(channelName, "rfcomm%d", statePtr->fd);

    statePtr->channel = Tcl_CreateChannel(&rfcommChannelType, channelName,
	    (ClientData) statePtr, (TCL_READABLE | TCL_WRITABLE));
    if (Tcl_SetChannelOption(interp, statePtr->channel, "-translation",
	    "auto lf") == TCL_ERROR) {
	Tcl_Close((Tcl_Interp *) NULL, statePtr->channel);
	return NULL;
    }
    return statePtr->channel;
}

static Tcl_Channel
OpenRfcommServer(Tcl_Interp *interp, int port, CONST char *myHost,
		 Tcl_TcpAcceptProc *acceptProc, ClientData acceptProcData)
{
    RfcommState *statePtr;
    char channelName[16 + TCL_INTEGER_SPACE];

    statePtr = CreateSocket(interp, port, myHost, 1, NULL, 0, 0);
    if (statePtr == NULL) {
	return NULL;
    }

    statePtr->acceptProc = acceptProc;
    statePtr->acceptProcData = acceptProcData;

    Tcl_CreateFileHandler(statePtr->fd, TCL_READABLE, RfcommAccept,
	    (ClientData) statePtr);
    sprintf(channelName, "rfcomm%d", statePtr->fd);
    statePtr->channel = Tcl_CreateChannel(&rfcommChannelType, channelName,
	    (ClientData) statePtr, 0);
    return statePtr->channel;
}

static void
RfcommAcceptCallbacksDeleteProc(ClientData clientData, Tcl_Interp *interp)
{
    Tcl_HashTable *hTblPtr;
    Tcl_HashEntry *hPtr;
    Tcl_HashSearch hSearch;
    AcceptCallback *acceptCallbackPtr;

    hTblPtr = (Tcl_HashTable *) clientData;
    for (hPtr = Tcl_FirstHashEntry(hTblPtr, &hSearch);
	     hPtr != (Tcl_HashEntry *) NULL;
	     hPtr = Tcl_NextHashEntry(&hSearch)) {
	acceptCallbackPtr = (AcceptCallback *) Tcl_GetHashValue(hPtr);
	acceptCallbackPtr->interp = (Tcl_Interp *) NULL;
    }
    Tcl_DeleteHashTable(hTblPtr);
    ckfree((char *) hTblPtr);
}

static void
RegisterRfcommServerInterpCleanup(Tcl_Interp *interp,
				  AcceptCallback *acceptCallbackPtr)
{
    Tcl_HashTable *hTblPtr;
    Tcl_HashEntry *hPtr;
    int new;

    hTblPtr = (Tcl_HashTable *) Tcl_GetAssocData(interp,
	    "rfcommAcceptCallbacks", NULL);
    if (hTblPtr == (Tcl_HashTable *) NULL) {
	hTblPtr = (Tcl_HashTable *) ckalloc((unsigned) sizeof (Tcl_HashTable));
	Tcl_InitHashTable(hTblPtr, TCL_ONE_WORD_KEYS);
	(void) Tcl_SetAssocData(interp, "rfcommAcceptCallbacks",
		RfcommAcceptCallbacksDeleteProc, (ClientData) hTblPtr);
    }
    hPtr = Tcl_CreateHashEntry(hTblPtr, (char *) acceptCallbackPtr, &new);
    if (!new) {
	panic("RegisterRfcommServerCleanup: damaged accept record table");
    }
    Tcl_SetHashValue(hPtr, (ClientData) acceptCallbackPtr);
}

static void
UnregisterRfcommServerInterpCleanupProc(Tcl_Interp *interp,
					AcceptCallback *acceptCallbackPtr)
{
    Tcl_HashTable *hTblPtr;
    Tcl_HashEntry *hPtr;

    hTblPtr = (Tcl_HashTable *) Tcl_GetAssocData(interp,
	    "rfcommAcceptCallbacks", NULL);
    if (hTblPtr == (Tcl_HashTable *) NULL) {
	return;
    }
    hPtr = Tcl_FindHashEntry(hTblPtr, (char *) acceptCallbackPtr);
    if (hPtr == (Tcl_HashEntry *) NULL) {
	return;
    }
    Tcl_DeleteHashEntry(hPtr);
}

static void
AcceptCallbackProc(ClientData callbackData, Tcl_Channel chan,
		   char *address, int port)
{
    AcceptCallback *acceptCallbackPtr;
    Tcl_Interp *interp;
    char *script;
    char portBuf[TCL_INTEGER_SPACE];
    int result;

    acceptCallbackPtr = (AcceptCallback *) callbackData;
    
    if (acceptCallbackPtr->interp != (Tcl_Interp *) NULL) {

        script = acceptCallbackPtr->script;
        interp = acceptCallbackPtr->interp;
        
        Tcl_Preserve((ClientData) script);
        Tcl_Preserve((ClientData) interp);

	TclFormatInt(portBuf, port);
        Tcl_RegisterChannel(interp, chan);

        Tcl_RegisterChannel((Tcl_Interp *) NULL,  chan);
        
        result = Tcl_VarEval(interp, script, " ", Tcl_GetChannelName(chan),
                " ", address, " ", portBuf, (char *) NULL);
        if (result != TCL_OK) {
            Tcl_BackgroundError(interp);
	    Tcl_UnregisterChannel(interp, chan);
        }

        Tcl_UnregisterChannel((Tcl_Interp *) NULL, chan);
        
        Tcl_Release((ClientData) interp);
        Tcl_Release((ClientData) script);
    } else {
        Tcl_Close((Tcl_Interp *) NULL, chan);
    }
}

static void
RfcommServerCloseProc(ClientData callbackData)
{
    AcceptCallback *acceptCallbackPtr;

    acceptCallbackPtr = (AcceptCallback *) callbackData;
    if (acceptCallbackPtr->interp != (Tcl_Interp *) NULL) {
	UnregisterRfcommServerInterpCleanupProc(acceptCallbackPtr->interp,
		acceptCallbackPtr);
    }
    Tcl_EventuallyFree((ClientData) acceptCallbackPtr->script, TCL_DYNAMIC);
    ckfree((char *) acceptCallbackPtr);
}

static int
RfcommObjCmd(ClientData clientData, Tcl_Interp *interp, int objc,
	     Tcl_Obj *CONST objv[])
{
    static CONST char *socketOptions[] = {
	"-async", "-myaddr", "-myport", "-server", (char *) NULL
    };
    enum socketOptions {
	SKT_ASYNC,      SKT_MYADDR,      SKT_MYPORT,      SKT_SERVER  
    };
    int optionIndex, a, server, port;
    char *arg, *copyScript, *host, *script;
    char *myaddr = NULL;
    int myport = 0;
    int async = 0;
    Tcl_Channel chan;
    AcceptCallback *acceptCallbackPtr;
    
    server = 0;
    script = NULL;

    for (a = 1; a < objc; a++) {
	arg = Tcl_GetString(objv[a]);
	if (arg[0] != '-') {
	    break;
	}
	if (Tcl_GetIndexFromObj(interp, objv[a], socketOptions,
		"option", TCL_EXACT, &optionIndex) != TCL_OK) {
	    return TCL_ERROR;
	}
	switch ((enum socketOptions) optionIndex) {
	    case SKT_ASYNC: {
		if (server == 1) {
		    Tcl_AppendResult(interp,
			    "cannot set -async option for server sockets",
			    (char *) NULL);
		    return TCL_ERROR;
		}
		async = 1;		
		break;
	    }
	    case SKT_MYADDR: {
		a++;
		if (a >= objc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -myaddr option",
			    (char *) NULL);
		    return TCL_ERROR;
		}
		myaddr = Tcl_GetString(objv[a]);
		break;
	    }
	    case SKT_MYPORT: {
		a++;
		if (a >= objc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -myport option",
			    (char *) NULL);
		    return TCL_ERROR;
		}
		if (Tcl_GetIntFromObj(interp, objv[a], &myport) != TCL_OK) {
		    return TCL_ERROR;
		}
		break;
	    }
	    case SKT_SERVER: {
		if (async == 1) {
		    Tcl_AppendResult(interp,
			    "cannot set -async option for server sockets",
			    (char *) NULL);
		    return TCL_ERROR;
		}
		server = 1;
		a++;
		if (a >= objc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -server option",
			    (char *) NULL);
		    return TCL_ERROR;
		}
		script = Tcl_GetString(objv[a]);
		break;
	    }
	    default: {
		panic("Tcl_SocketObjCmd: bad option index to SocketOptions");
	    }
	}
    }
    if (server) {
	host = myaddr;
	if (myport != 0) {
	    Tcl_AppendResult(interp, "Option -myport is not valid for servers",
		    NULL);
	    return TCL_ERROR;
	}
    } else if (a < objc) {
	host = Tcl_GetString(objv[a]);
	a++;
    } else {
wrongNumArgs:
	Tcl_AppendResult(interp, "wrong # args: should be either:\n",
		Tcl_GetString(objv[0]),
		" ?-myaddr addr? ?-myport myport? ?-async? host port\n",
		Tcl_GetString(objv[0]),
		" -server command ?-myaddr addr? port",
		(char *) NULL);
	return TCL_ERROR;
    }

    if (a == objc - 1) {
	if (Tcl_GetIntFromObj(interp, objv[a], &port) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	goto wrongNumArgs;
    }

    if (server) {
	acceptCallbackPtr = (AcceptCallback *) ckalloc((unsigned)
		sizeof (AcceptCallback));
	copyScript = ckalloc((unsigned) strlen(script) + 1);
	strcpy(copyScript, script);
	acceptCallbackPtr->script = copyScript;
	acceptCallbackPtr->interp = interp;
	chan = OpenRfcommServer(interp, port, host, AcceptCallbackProc,
		(ClientData) acceptCallbackPtr);
	if (chan == (Tcl_Channel) NULL) {
	    ckfree(copyScript);
	    ckfree((char *) acceptCallbackPtr);
	    return TCL_ERROR;
	}

	RegisterRfcommServerInterpCleanup(interp, acceptCallbackPtr);
	
	Tcl_CreateCloseHandler(chan, RfcommServerCloseProc,
		(ClientData) acceptCallbackPtr);
    } else {
	chan = OpenRfcommClient(interp, port, host, myaddr, myport, async);
	if (chan == (Tcl_Channel) NULL) {
	    return TCL_ERROR;
	}
    }
    Tcl_RegisterChannel(interp, chan);            
    Tcl_AppendResult(interp, Tcl_GetChannelName(chan), (char *) NULL);
    
    return TCL_OK;
}

#ifdef ANDROID
void
Java_tk_tcl_wish_BTClient_shutdown(JNIEnv * env, jclass cls, jobject jfd)
{
    jfieldID fid;
    jint fd;

    if (jfd != NULL) {
	fid = (*env)->GetFieldID(env, (*env)->GetObjectClass(env, jfd),
				 "descriptor", "I");
	if (fid != NULL) {
	    fd = (*env)->GetIntField(env, jfd, fid);
	    if (fd >= 0) {
		shutdown(fd, 2);
	    }
	}
    }
}
#endif

int
Tcl_RfcommInit(Tcl_Interp *interp)
{
    if (Tcl_CreateObjCommand(interp, "rfcomm", RfcommObjCmd,
	    (ClientData) 0, (Tcl_CmdDeleteProc *) NULL) == NULL) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * fill-column: 78
 * tab-width: 8
 * End:
 */
