/*
 * tclBorg.c --
 *
 *      This file contains the implementation of the "borg" Tcl built-in
 *      command. This command deals with various subsystems of the Android
 *	world.
 *
 * Copyright (c) 2013 Christian Werner <chw@ch-werner.de>
 *
 * See the file "license.terms" for information on usage and redistribution of
 * this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include <pthread.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <jni.h>
#include <tcl.h>
#include <tclInt.h>
#include <SDL.h>

typedef struct Callback {
    struct Callback *next;
    int id;
    Tcl_Interp *interp;
    Tcl_DString cmd;
} Callback;

static JavaVM *jvm;
static jclass activity;
static pthread_mutex_t cbmut = PTHREAD_MUTEX_INITIALIZER;
static struct {
    Callback *first, *last;
    int pfd[2];
} cbq;
static Tcl_HashTable cbh;
static int startupFileSet = 0;
static char *startupFile = NULL;

static JNIEnv *
GetJNIEnv(void)
{
    JNIEnv *env;
    int status = (*jvm)->AttachCurrentThread(jvm, &env, NULL);

    if (status < 0) {
	return NULL;
    }
    return env;
}

jint
JNI_OnLoad(JavaVM *vm, void *reserved)
{
    JNIEnv *env;

    jvm = vm;
    if ((*jvm)->GetEnv(jvm, (void **) &env, JNI_VERSION_1_4) != JNI_OK) {
	return JNI_ERR;
    }
    if (pipe(cbq.pfd) < 0) {
	return JNI_ERR;
    }
    fcntl(cbq.pfd[0], F_SETFL, O_NONBLOCK);
    fcntl(cbq.pfd[1], F_SETFL, O_NONBLOCK);
    cbq.first = cbq.last = NULL;
    Tcl_InitHashTable(&cbh, TCL_ONE_WORD_KEYS);
    return JNI_VERSION_1_4;
}

void
Java_tk_tcl_wish_AndroWish_nativeInit(JNIEnv * env, jclass cls, jint gles,
				      jstring lang)
{
    activity = (jclass) ((*env)->NewGlobalRef(env, cls));
    if (gles < 2) {
	SDL_SetHint("SDL_RENDER_DRIVER", "opengles");
    }
    if (lang) {
	const char *str = (*env)->GetStringUTFChars(env, lang, 0);
	int len;

	len = strlen(str);
	if ((len > 0) && (str[0] != '_')) {
	    char *elang = malloc(len + 10);

	    strcpy(elang, str);
	    strcat(elang, ".UTF-8");
	    setenv("LANG", elang, 1);
	}
	(*env)->ReleaseStringUTFChars(env, lang, str);
	(*env)->DeleteLocalRef(env, lang);
    }
}

void
Java_tk_tcl_wish_AndroWish_nativeSetStartupFile(JNIEnv * env, jclass cls,
						jstring file)
{
    if (!startupFileSet) {
	startupFileSet = 1;
	if (file) {
	    const char *str = (*env)->GetStringUTFChars(env, file, 0);
	    int len, i;

	    len = strlen(str);
	    startupFile = malloc(len + 1);
	    if ((len > 5) && (strncmp(str, "file:", 5) == 0)) {
		len = 5;
		while (str[len] && (str[len] == '/')) {
		    ++len;
		}
		if (len > 5) {
		    --len;
		}	
		strcpy(startupFile, str + len);
		len = strlen(startupFile);
	    } else {
		strcpy(startupFile, str);
	    }
	    (*env)->ReleaseStringUTFChars(env, file, str);
	    (*env)->DeleteLocalRef(env, file);
	    if ((len > 8) && (strncmp(startupFile, "/assets/", 8) == 0)) {
		/* accept */
	    } else if (access(startupFile, R_OK) != 0) {
		if ((len > 6) && (strncmp(startupFile, "zipfs:", 6) == 0)) {
		    str = startupFile + 6;
		    len = 0;
		    while (str[len] && (str[len] == '/')) {
			++len;
		    }
		    if (len > 0) {
			--len;
		    }
		    str += len;
		    len = strlen(str);
		    if ((len > 8) && (strncmp(str, "/assets/", 8) == 0)) {
			/* accept */
			return;
		    } else if (access(str, R_OK) == 0) {
			/* accept */
			return;
		    }
		}
		free(startupFile);
		startupFile = NULL;
	    }
	}
    }
}

static int
StartActivity(Tcl_Interp *interp, const char *action, const char *uristr,
	      const char *type, int cargc, const char **cargv,
	      int argc, const char **argv, const char *cbcmd)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;
    int i, id, idret, isNew;
    Tcl_HashEntry *hPtr;
    Callback *cb;
    jobjectArray cats, args, types;
    jstring jaction, juristr, jtype;
    static int idcount = 0;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    mid = (*env)->GetStaticMethodID(env, activity,
				    "runActivity",
				    "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;[Ljava/lang/String;[Ljava/lang/String;[Ljava/lang/String;I)I");
    cats = (*env)->NewObjectArray(env, cargc,
				  (*env)->FindClass(env, "java/lang/String"),
				  NULL);
    for (i = 0; i < cargc; i++) {
	(*env)->SetObjectArrayElement(env, cats, i, 
				      (*env)->NewStringUTF(env, cargv[i]));
    }
    args = (*env)->NewObjectArray(env, argc,
				  (*env)->FindClass(env, "java/lang/String"),
				  NULL);
    types = (*env)->NewObjectArray(env, argc,
				   (*env)->FindClass(env, "java/lang/String"),
				   NULL);
    for (i = 0; i < argc; i++) {
	if ((i % 2) == 0) {
	    CONST char **kargv = NULL;
	    int kargc;

	    if (Tcl_SplitList(interp, argv[i], &kargc, &kargv) != TCL_OK) {
		(*env)->DeleteLocalRef(env, args);
		(*env)->DeleteLocalRef(env, cats);
		(*env)->DeleteLocalRef(env, types);
		return TCL_ERROR;
	    }
	    if (kargc > 1) {
		(*env)->SetObjectArrayElement(env, types, i, 
					      (*env)->NewStringUTF(env, kargv[0]));
		(*env)->SetObjectArrayElement(env, args, i, 
					      (*env)->NewStringUTF(env, kargv[1]));
	    } else {
		(*env)->SetObjectArrayElement(env, args, i, 
					      (*env)->NewStringUTF(env, argv[i]));
	    }
	    ckfree((char *) kargv);
	} else {
	    (*env)->SetObjectArrayElement(env, args, i, 
					  (*env)->NewStringUTF(env, argv[i]));
	}
    }
    if (cbcmd != NULL) {
	pthread_mutex_lock(&cbmut);
	id = ++idcount;
	if (idcount < 0) {
	    idcount = 0;
	    id = ++idcount;
	}
	pthread_mutex_unlock(&cbmut);
    } else {
	id = -1;
    }
    jaction = (*env)->NewStringUTF(env, action);
    juristr = (*env)->NewStringUTF(env, (uristr == NULL) ? "" : uristr);
    jtype = (*env)->NewStringUTF(env, (type == NULL) ? "" : type);
    idret = (*env)->CallStaticIntMethod(env, activity, mid, jaction, juristr,
					jtype, cats, types, args, id);
    (*env)->DeleteLocalRef(env, jaction);
    (*env)->DeleteLocalRef(env, juristr);
    (*env)->DeleteLocalRef(env, jtype);
    (*env)->DeleteLocalRef(env, args);
    (*env)->DeleteLocalRef(env, cats);
    (*env)->DeleteLocalRef(env, types);
    if (idret != id) {
	Tcl_AppendResult(interp, "activity failed", (char *) NULL);
	return TCL_ERROR;
    }
    if (id >= 0) {
	Callback *cb;

	cb = (Callback *) ckalloc(sizeof (Callback));
	cb->next = NULL;
	cb->id = id;
	Tcl_DStringInit(&cb->cmd);
	Tcl_DStringAppend(&cb->cmd, cbcmd, -1);
	cb->interp = interp;
	pthread_mutex_lock(&cbmut);
	hPtr = Tcl_CreateHashEntry(&cbh, (ClientData) id, &isNew);
	if (!isNew) {
	    panic("id for activity reused");
	}
	Tcl_SetHashValue(hPtr, (char *) cb);
	pthread_mutex_unlock(&cbmut);
	Tcl_SetObjResult(interp, Tcl_NewIntObj(id));
    }
    return TCL_OK;
}

static int
CancelActivity(Tcl_Interp *interp, int id)
{
    Tcl_HashEntry *hPtr;
    Callback *cb = NULL;

    pthread_mutex_lock(&cbmut);
    hPtr = Tcl_FindHashEntry(&cbh, (ClientData) id);
    if (hPtr != NULL) {
	cb = (Callback  *) Tcl_GetHashValue(hPtr);
	Tcl_DeleteHashEntry(hPtr);
    } else {
	Callback *cbPrev = NULL;

	cb = cbq.first;
	while (cb != NULL) {
	    if (cb->id == id) {
		break;
	    }
	    cbPrev = cb;
	    cb = cb->next;
	}
	if (cb != NULL) {
	    if (cbPrev == NULL) {
		cbq.first = cb->next;
	    } else {
		cbPrev->next = cb->next;
	    }
	    if (cbq.last == cb) {
		cbq.last = cbq.first;
	    }
	    Tcl_DStringFree(&cb->cmd);
	    ckfree((char *) cb);
	}
    }
    pthread_mutex_lock(&cbmut);
    Tcl_SetObjResult(interp, Tcl_NewIntObj(cb != NULL));
    return TCL_OK;
}

static int
QueryIntents(Tcl_Interp *interp, int which, const char *action,
	     const char *uristr, const char *type)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;
    jobjectArray jlist;
    jstring jaction;
    jstring juristr = NULL;
    jstring jtype = NULL;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    mid = (*env)->GetStaticMethodID(env, activity,
				    "queryIntents",
				    "(ILjava/lang/String;Ljava/lang/String;Ljava/lang/String;)[Ljava/lang/String;");
    jaction = (*env)->NewStringUTF(env, action);
    if (uristr != NULL) {
	juristr = (*env)->NewStringUTF(env, uristr);
    }
    if (type != NULL) {
	jtype = (*env)->NewStringUTF(env, type);
    }
    jlist = (jobjectArray) (*env)->CallStaticObjectMethod(env, activity, mid,
							  which, jaction,
							  juristr, jtype);
    (*env)->DeleteLocalRef(env, jaction);
    if (juristr != NULL) {
	(*env)->DeleteLocalRef(env, juristr);
    }
    if (jtype != NULL) {
	(*env)->DeleteLocalRef(env, jtype);
    }
    if (jlist != NULL) {
	jsize i, len = (*env)->GetArrayLength(env, jlist);
	jstring jelem;
	const char *str;

	for (i = 0; i < len; i++) {
	    jelem = (*env)->GetObjectArrayElement(env, jlist, i);
	    if (jelem != NULL) {
		str = (*env)->GetStringUTFChars(env, jelem, 0);
		Tcl_AppendElement(interp, str);
		(*env)->ReleaseStringUTFChars(env, jelem, str);
		(*env)->DeleteLocalRef(env, jelem);
	    }
	}
	(*env)->DeleteLocalRef(env, jlist);
    }
    return TCL_OK;
}

static int
Vibrate(Tcl_Interp *interp, int dur)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    mid = (*env)->GetStaticMethodID(env, activity,
				    "vibrate", "(I)V");
    (*env)->CallStaticVoidMethod(env, activity, mid, (jint) dur);
    return TCL_OK;
}

static int
Beep(Tcl_Interp *interp)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    mid = (*env)->GetStaticMethodID(env, activity,
				    "beep", "()V");
    (*env)->CallStaticVoidMethod(env, activity, mid);
    return TCL_OK;
}

static int
Speak(Tcl_Interp *interp, int op, const char *text)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;
    jstring jtext = NULL;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    mid = (*env)->GetStaticMethodID(env, activity,
				    "speak", "(ILjava/lang/String;)I");
    if (text != NULL) {
	jtext = (*env)->NewStringUTF(env, text);
    }
    (*env)->CallStaticVoidMethod(env, activity, mid, op, jtext);
    if (jtext != NULL) {
	(*env)->DeleteLocalRef(env, jtext);
    }
    return TCL_OK;
}

static int
QueryConsts(Tcl_Interp *interp, const char *clsname)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;
    jobjectArray jlist;
    jstring jclsname;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    mid = (*env)->GetStaticMethodID(env, activity,
				    "queryConsts",
				    "(Ljava/lang/String;)[Ljava/lang/String;");
    jclsname = (*env)->NewStringUTF(env, clsname);
    jlist = (jobjectArray) (*env)->CallStaticObjectMethod(env, activity, mid,
							  jclsname);
    (*env)->DeleteLocalRef(env, jclsname);
    if (jlist != NULL) {
	jsize i, len = (*env)->GetArrayLength(env, jlist);
	jstring jelem;
	const char *str;

	for (i = 0; i < len; i++) {
	    jelem = (*env)->GetObjectArrayElement(env, jlist, i);
	    if (jelem != NULL) {
		str = (*env)->GetStringUTFChars(env, jelem, 0);
		Tcl_AppendElement(interp, str);
		(*env)->ReleaseStringUTFChars(env, jelem, str);
		(*env)->DeleteLocalRef(env, jelem);
	    } else {
		Tcl_AppendElement(interp, "");
	    }
	}
	(*env)->DeleteLocalRef(env, jlist);
    }
    return TCL_OK;
}

static int
Location(Tcl_Interp *interp, int op, int val0, int val1)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;
    jstring jloc;
    int result = TCL_OK;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    switch (op) {
    case 0:
	mid = (*env)->GetStaticMethodID(env, activity,
					"startLocating", "(II)V");
	(*env)->CallStaticVoidMethod(env, activity, mid, val0, val1);
	break;
    case 1:
	mid = (*env)->GetStaticMethodID(env, activity,
					"stopLocating", "()V");
	(*env)->CallStaticVoidMethod(env, activity, mid);
	break;
    case 2:
	mid = (*env)->GetStaticMethodID(env, activity,
					"getLocation", "()Ljava/lang/String;");
	jloc = (jstring) (*env)->CallStaticObjectMethod(env, activity, mid);
	if (jloc != NULL) {
	    const char *str = (*env)->GetStringUTFChars(env, jloc, 0);

	    Tcl_AppendResult(interp, str, (char *) NULL);
	    (*env)->ReleaseStringUTFChars(env, jloc, str);
	    (*env)->DeleteLocalRef(env, jloc);
	}
	break;
    default:
	Tcl_AppendResult(interp, "unknown location operation", (char *) NULL);
	result = TCL_ERROR;
	break;
    }
    return result;
}

static int
NetworkInfo(Tcl_Interp *interp)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;
    jstring jnetinfo;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    mid = (*env)->GetStaticMethodID(env, activity,
				    "getNetworkInfo", "()Ljava/lang/String;");
    jnetinfo = (jstring) (*env)->CallStaticObjectMethod(env, activity, mid);
    if (jnetinfo != NULL) {
	const char *str = (*env)->GetStringUTFChars(env, jnetinfo, 0);

	Tcl_AppendResult(interp, str, (char *) NULL);
	(*env)->ReleaseStringUTFChars(env, jnetinfo, str);
	(*env)->DeleteLocalRef(env, jnetinfo);
    }
    return TCL_OK;
}

static int
Shortcut(Tcl_Interp *interp, int op, const char *name, const char *arg,
	 const char *icon)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;
    jstring jname, jarg = NULL, jicon = NULL;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    mid = (*env)->GetStaticMethodID(env, activity,
				    "shortcut",
				    "(ILjava/lang/String;Ljava/lang/String;Ljava/lang/String;)V");
    jname = (*env)->NewStringUTF(env, name);
    if (arg != NULL) {
	jarg = (*env)->NewStringUTF(env, arg);
    }
    if (icon != NULL) {
	jicon = (*env)->NewStringUTF(env, icon);
    }
    (*env)->CallStaticVoidMethod(env, activity, mid, op, jname, jarg, jicon);
    (*env)->DeleteLocalRef(env, jname);
    if (icon != NULL) {
	(*env)->DeleteLocalRef(env, jicon);
    }
    if (arg != NULL) {
	(*env)->DeleteLocalRef(env, jarg);
    }
    return TCL_OK;
}

static int
Notify(Tcl_Interp *interp, int op, int id, const char *title, const char *text)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;
    jstring jtitle = NULL, jtext = NULL;
    jthrowable exc;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    mid = (*env)->GetStaticMethodID(env, activity,
				    "notify", "(IILjava/lang/String;Ljava/lang/String;)V");
    if (title != NULL) {
        jtitle = (*env)->NewStringUTF(env, title);
    }
    if (text != NULL) {
	jtext = (*env)->NewStringUTF(env, text);
    }
    (*env)->CallStaticVoidMethod(env, activity, mid, op, id, jtitle, jtext);
    if (title != NULL) {
	(*env)->DeleteLocalRef(env, jtitle);
    }
    if (text != NULL) {
	(*env)->DeleteLocalRef(env, jtext);
    }
    exc = (*env)->ExceptionOccurred(env);
    if (exc) {
	(*env)->DeleteLocalRef(env, exc);
	(*env)->ExceptionClear(env);
    }
    return TCL_OK;
}

static int
Bluetooth(Tcl_Interp *interp, int op, const char *arg)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;
    jstring jarg = NULL;
    jobjectArray jlist;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    mid = (*env)->GetStaticMethodID(env, activity, "bluetooth",
				    "(ILjava/lang/String;)[Ljava/lang/String;");
    if (arg != NULL) {
        jarg = (*env)->NewStringUTF(env, arg);
    }
    jlist = (*env)->CallStaticObjectMethod(env, activity, mid, op, jarg);
    if (arg != NULL) {
	(*env)->DeleteLocalRef(env, jarg);
    }
    if (jlist != NULL) {
	jsize i, len = (*env)->GetArrayLength(env, jlist);
	jstring jelem;
	const char *str;

	for (i = 0; i < len; i++) {
	    jelem = (*env)->GetObjectArrayElement(env, jlist, i);
	    if (jelem != NULL) {
		str = (*env)->GetStringUTFChars(env, jelem, 0);
		Tcl_AppendElement(interp, str);
		(*env)->ReleaseStringUTFChars(env, jelem, str);
		(*env)->DeleteLocalRef(env, jelem);
	    }
	}
	(*env)->DeleteLocalRef(env, jlist);
    }
    return TCL_OK;
}

static int
Toast(Tcl_Interp *interp, const char *text, int dur)
{
    JNIEnv *env = GetJNIEnv();
    jmethodID mid;
    jstring jtext = NULL;
    jthrowable exc;

    if (env == NULL) {
	Tcl_AppendResult(interp, "no JNIEnv", (char *) NULL);
	return TCL_ERROR;
    }
    mid = (*env)->GetStaticMethodID(env, activity,
				    "toast", "(Ljava/lang/String;I)V");
    if (text != NULL) {
	jtext = (*env)->NewStringUTF(env, text);
    }
    (*env)->CallStaticVoidMethod(env, activity, mid, jtext, dur);
    if (text != NULL) {
	(*env)->DeleteLocalRef(env, jtext);
    }
    return TCL_OK;
}

void
Java_tk_tcl_wish_AndroWish_nativeIntentCallback(JNIEnv * env, jclass cls,
						jint id, jint ret,
						jstring action,
						jstring uristr,
						jstring type,
						jobjectArray cats,
						jobjectArray args)
{
    Tcl_HashEntry *hPtr;
    jstring jstr;
    const char *str;
    Callback *cb = NULL;
    int i, max = 0;

    pthread_mutex_lock(&cbmut);
    hPtr = Tcl_FindHashEntry(&cbh, (ClientData) id);
    if (hPtr != NULL) {
	cb = (Callback *) Tcl_GetHashValue(hPtr);
	Tcl_DeleteHashEntry(hPtr);
    }
    pthread_mutex_unlock(&cbmut);
    if (cb != NULL) {
	char buffer[32];

	sprintf(buffer, "%d", ret);
	Tcl_DStringAppendElement(&cb->cmd, buffer);
	str = NULL;
	if (action != NULL) {
	    str = (*env)->GetStringUTFChars(env, action, 0);
	}
	Tcl_DStringAppendElement(&cb->cmd, (str == NULL) ? "" : str);
	if (action != NULL) {
	    (*env)->ReleaseStringUTFChars(env, action, str);
	}
	str = NULL;
	if (uristr != NULL) {
	    str = (*env)->GetStringUTFChars(env, uristr, 0);
	}
	Tcl_DStringAppendElement(&cb->cmd, (str == NULL) ? "" : str);
	if (uristr != NULL) {
	    (*env)->ReleaseStringUTFChars(env, uristr, str);
	}
	str = NULL;
	if (type != NULL) {
	    str = (*env)->GetStringUTFChars(env, type, 0);
	}
	Tcl_DStringAppendElement(&cb->cmd, (str == NULL) ? "" : str);
	if (type != NULL) {
	    (*env)->ReleaseStringUTFChars(env, type, str);
	}
	Tcl_DStringStartSublist(&cb->cmd);
	if (cats != NULL) {
	    max = (*env)->GetArrayLength(env, cats);
	}
	for (i = 0; i < max; i++) {
	    jstr = (*env)->GetObjectArrayElement(env, cats, i);
	    str = (*env)->GetStringUTFChars(env, jstr, 0);
	    Tcl_DStringAppendElement(&cb->cmd, (str == NULL) ? "" : str);
	    (*env)->ReleaseStringUTFChars(env, jstr, str);
	    (*env)->DeleteLocalRef(env, jstr);
	}
	Tcl_DStringEndSublist(&cb->cmd);
	Tcl_DStringStartSublist(&cb->cmd);
	if (args != NULL) {
	    max = (*env)->GetArrayLength(env, args);
	}
	for (i = 0; i < max; i++) {
	    jstr = (*env)->GetObjectArrayElement(env, args, i);
	    str = (*env)->GetStringUTFChars(env, jstr, 0);
	    Tcl_DStringAppendElement(&cb->cmd, (str == NULL) ? "" : str);
	    (*env)->ReleaseStringUTFChars(env, jstr, str);
	    (*env)->DeleteLocalRef(env, jstr);
	}
	Tcl_DStringEndSublist(&cb->cmd);
	pthread_mutex_lock(&cbmut);
	if (cbq.last != NULL) {
	    cbq.last->next = cb;
	    cbq.last = cb;
	} else {
	    cbq.first = cbq.last = cb;
	    write(cbq.pfd[1], "A", 1);
	}
	pthread_mutex_unlock(&cbmut);
    }
}

int
Java_tk_tcl_wish_AndroWish_nativeTriggerLocation(JNIEnv * env, jclass cls)
{
    int ret;

    pthread_mutex_lock(&cbmut);
    ret = write(cbq.pfd[1], "L", 1);
    if (ret < 0) {
	if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR)) {
	    ret = 0;
	}
    }
    pthread_mutex_unlock(&cbmut);
    return ret;
}

int
Java_tk_tcl_wish_AndroWish_nativeTriggerNetworkInfo(JNIEnv * env, jclass cls)
{
    int ret;

    pthread_mutex_lock(&cbmut);
    ret = write(cbq.pfd[1], "N", 1);
    if (ret < 0) {
	if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR)) {
	    ret = 0;
	}
    }
    pthread_mutex_unlock(&cbmut);
    return ret;
}

int
Java_tk_tcl_wish_AndroWish_nativeTriggerBluetooth(JNIEnv * env, jclass cls)
{
    int ret;

    pthread_mutex_lock(&cbmut);
    ret = write(cbq.pfd[1], "B", 1);
    if (ret < 0) {
	if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR)) {
	    ret = 0;
	}
    }
    pthread_mutex_unlock(&cbmut);
    return ret;
}

static void
FileHandler(ClientData clientData, int mask)
{
    char buffer[64];
    int n, doloc = 0, donet = 0, dobt = 0;
    Callback *cb = NULL;

    if (mask & TCL_READABLE) {
	for (;;) {
	    n = read(cbq.pfd[0], buffer, sizeof (buffer));
	    if (n == 0) {
		close(cbq.pfd[0]);
		goto done;
	    }
	    if (n < 0) {
		if ((errno != EAGAIN) && (errno != EWOULDBLOCK) &&
		    (errno != EINTR)) {
		    close(cbq.pfd[0]);
		}
		goto done;
	    }
	    if (n > 0) {
		if (memchr(buffer, 'L', n) != NULL) {
		    doloc++;
		}
		if (memchr(buffer, 'N', n) != NULL) {
		    donet++;
		}
		if (memchr(buffer, 'B', n) != NULL) {
		    dobt++;
		}
	    }
	}
done:
	pthread_mutex_lock(&cbmut);
	cb = cbq.first;
	cbq.first = cbq.last = NULL;
	pthread_mutex_unlock(&cbmut);
    }
    while (cb != NULL) {
	Callback *cbNext = cb->next;
	Tcl_Interp *interp = cb->interp;
	int ret;

	cb->next = NULL;
	Tcl_Preserve((ClientData) interp);
	ret = Tcl_GlobalEval(interp, Tcl_DStringValue(&cb->cmd));
	if (ret != TCL_OK) {
	    Tcl_BackgroundError(interp);
	}
	Tcl_DStringFree(&cb->cmd);
	ckfree((char *) cb);
	Tcl_Release((ClientData) interp);
	cb = cbNext;
    }
    if (doloc) {
	SDL_Event event;

	event.type = SDL_USEREVENT;
	event.user.windowID = 0;
	event.user.code = 0;
	event.user.data1 = "LocationUpdate";
	event.user.data2 = NULL;
	SDL_PushEvent(&event);
    }
    if (donet) {
	SDL_Event event;

	event.type = SDL_USEREVENT;
	event.user.windowID = 0;
	event.user.code = 0;
	event.user.data1 = "NetworkInfo";
	event.user.data2 = NULL;
	SDL_PushEvent(&event);
    }
    if (dobt) {
	SDL_Event event;

	event.type = SDL_USEREVENT;
	event.user.windowID = 0;
	event.user.code = 0;
	event.user.data1 = "Bluetooth";
	event.user.data2 = NULL;
	SDL_PushEvent(&event);
    }
}

static int
BorgCmdDeleted(ClientData clientData)
{
    Callback *cb = NULL;

    Tcl_DeleteFileHandler(cbq.pfd[0]);
    pthread_mutex_lock(&cbmut);
    close(cbq.pfd[0]);
    cb = cbq.first;
    cbq.first = cbq.last = NULL;
    pthread_mutex_unlock(&cbmut);
    while (cb != NULL) {
	Callback *cbNext = cb->next;

	cb->next = NULL;
	Tcl_DStringFree(&cb->cmd);
	ckfree((char *) cb);
	cb = cbNext;
    }
}

static int
BorgCmd(ClientData clientData, Tcl_Interp *interp,
	int argc, CONST char **argv)
{
    int ret, length;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # arguments: should be \"", argv[0],
			 " option ... \"", (char *) NULL);
	return TCL_ERROR;
    }
    length = strlen(argv[1]);
    if ((length > 1) && (strncmp(argv[1], "activity", length) == 0)) {
	int cargc = 0, largc = 0;
	CONST char **cargv = NULL, **largv = NULL;

	if ((argc < 5) || (argc > 8)) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
			     argv[0], " activity action uri type",
			     " ?cats args callback?\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (argc > 5) {
	    if (Tcl_SplitList(interp, argv[5], &cargc, &cargv) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
	if (argc > 6) {
	    if (Tcl_SplitList(interp, argv[6], &largc, &largv) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
	ret = StartActivity(interp, argv[2], argv[3], argv[4],
			    cargc, cargv, largc, largv,
			    (argc > 7) ? argv[7] : NULL);
	if (cargv != NULL) {
	    ckfree((char *) cargv);
	}
	if (largv != NULL) {
	    ckfree((char *) largv);
	}
	return ret;
    } else if ((length > 1) && (strncmp(argv[1], "cancel", length) == 0)) {
	int id;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		    argv[0], " cancel id\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (Tcl_GetInt(interp, argv[2], &id) != TCL_OK) {
	    return TCL_ERROR;
	}
	return CancelActivity(interp, id);
    } else if ((length > 1) &&
	       (strncmp(argv[1], "queryactivities", length) == 0)) {
	if ((argc < 3) || (argc > 5)) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
			     argv[0], " queryactivities action ?uristr type?",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	ret = QueryIntents(interp, 0, argv[2], (argc > 3) ? argv[3] : NULL,
			   (argc > 4) ? argv[4] : NULL);
	return ret;
    } else if ((length > 1) &&
	       (strncmp(argv[1], "queryservices", length) == 0)) {
	if ((argc < 3) || (argc > 5)) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
			     argv[0], " queryservices action ?uristr type?",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	ret = QueryIntents(interp, 1, argv[2], (argc > 3) ? argv[3] : NULL,
			   (argc > 4) ? argv[4] : NULL);
	return ret;
    } else if ((length > 1) &&
	       (strncmp(argv[1], "querybroadcastreceivers", length) == 0)) {
	if ((argc < 3) || (argc > 5)) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
			     argv[0], " querybroadcastreceivers ",
			     "action ?uristr type?", (char *) NULL);
	    return TCL_ERROR;
	}
	ret = QueryIntents(interp, 2, argv[2], (argc > 3) ? argv[3] : NULL,
			   (argc > 4) ? argv[4] : NULL);
	return ret;
    } else if ((length > 1) &&
	       (strncmp(argv[1], "queryconsts", length) == 0)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
			     argv[0], " queryconsts classname", (char *) NULL);
	    return TCL_ERROR;
	}
	ret = QueryConsts(interp, argv[2]);
	return ret;
    } else if ((length > 1) && (strncmp(argv[1], "vibrate", length) == 0)) {
	int dur;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		    argv[0], " vibrate duration\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (Tcl_GetInt(interp, argv[2], &dur) != TCL_OK) {
	    return TCL_ERROR;
	}
	return Vibrate(interp, dur);
    } else if ((length > 1) && (strncmp(argv[1], "beep", length) == 0)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		    argv[0], " beep\"", (char *) NULL);
	    return TCL_ERROR;
	}
	return Beep(interp);
    } else if ((length > 1) && (strncmp(argv[1], "speak", length) == 0)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		    argv[0], " speak text\"", (char *) NULL);
	    return TCL_ERROR;
	}
	return Speak(interp, 1, argv[2]);
    } else if ((length > 1) && (strncmp(argv[1], "stopspeak", length) == 0)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		    argv[0], " stopspeak\"", (char *) NULL);
	    return TCL_ERROR;
	}
	return Speak(interp, 0, NULL);
    } else if ((length > 1) && (strncmp(argv[1], "isspeaking", length) == 0)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		    argv[0], " isspeaking\"", (char *) NULL);
	    return TCL_ERROR;
	}
	ret = Speak(interp, 2, NULL);
	Tcl_SetObjResult(interp, Tcl_NewIntObj(ret));
	return TCL_OK;
    } else if ((length > 1) && (strncmp(argv[1], "endspeak", length) == 0)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		    argv[0], " endspeak\"", (char *) NULL);
	    return TCL_ERROR;
	}
	return Speak(interp, -1, NULL);
    } else if ((length > 1) && (strncmp(argv[1], "location", length) == 0)) {
	if (argc < 3) {
badLocation:
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		    argv[0], " location start|stop|get ...\"", (char *) NULL);
	    return TCL_ERROR;
	}
	length = strlen(argv[2]);
	if ((length > 2) && (strncmp(argv[2], "start", length) == 0)) {
	    int minrate = 60000;
	    int mindist = 30;

	    if (argc > 5) {
		Tcl_AppendResult(interp, "wrong # arguments: should be \"",
				 argv[0],
				 " location start ?minrate mindist?\"",
				 (char *) NULL);
		return TCL_ERROR;
	    }
	    if ((argc > 3) &&
		(Tcl_GetInt(interp, argv[3], &minrate) != TCL_OK)) {
		return TCL_ERROR;
	    }
	    if ((argc > 4) &&
		(Tcl_GetInt(interp, argv[4], &mindist) != TCL_OK)) {
		return TCL_ERROR;
	    }
	    return Location(interp, 0, minrate, mindist);
	} else if ((length > 2) &&
		   (strncmp(argv[2], "stop", length) == 0)) {
	    if (argc != 3) {
		Tcl_AppendResult(interp, "wrong # arguments: should be \"",
				 argv[0], " location stop\"", (char *) NULL);
		return TCL_ERROR;
	    }
	    return Location(interp, 1, 0, 0);
	} else if ((length > 1) && (strncmp(argv[2], "get", length) == 0)) {
	    if (argc != 3) {
		Tcl_AppendResult(interp, "wrong # arguments: should be \"",
				 argv[0], " location get\"", (char *) NULL);
		return TCL_ERROR;
	    }
	    return Location(interp, 2, 0, 0);
	}
	goto badLocation;
    } else if ((length > 1) &&
	       (strncmp(argv[1], "networkinfo", length) == 0)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		    argv[0], " networkinfo\"", (char *) NULL);
	    return TCL_ERROR;
	}
	return NetworkInfo(interp);
    } else if ((length > 1) && (strncmp(argv[1], "shortcut", length) == 0)) {
	int op;

	if ((argc < 4) || (argc > 6)) {
scWrongArgs:
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		    argv[0], " shortcut add|delete name ?arg ?icon??\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	length = strlen(argv[2]);
	if ((length > 0) && (strncmp(argv[2], "add", length) ==  0)) {
	    op = 0;
	} else if ((length > 0) && (strncmp(argv[2], "delete", length) == 0)) {
	    op = 1;
	} else {
	    goto scWrongArgs;
	}
	return Shortcut(interp, op, argv[3], (argc > 4) ? argv[4] : NULL,
			(argc > 5) ? argv[5] : NULL);
    } else if ((length > 1) &&
	       (strncmp(argv[1], "notification", length) == 0)) {
        int op, id = 0;
	CONST char *title = NULL, *text = NULL;

	if ((argc < 3) || (argc > 6)) {
nmWrongArgs:
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		argv[0], " notification add|delete ?id ?title text??\"",
		(char *) NULL);
	    return TCL_ERROR;
	}
	length = strlen(argv[2]);
	if ((length > 0) && (strncmp(argv[2], "add", length) ==  0)) {
	    if (argc != 6) {
	        goto nmWrongArgs;
	    }
	    if (Tcl_GetInt(interp, argv[3], &id) != TCL_OK) {
	        return TCL_ERROR;
	    }
	    op = 0;
	    title = argv[4];
	    text = argv[5];
	} else if ((length > 0) && (strncmp(argv[2], "delete", length) == 0)) {
	    if (argc == 3) {
	        op = 2;	/* all */
	    } else if (argc == 4) {
	        if (Tcl_GetInt(interp, argv[3], &id) != TCL_OK) {
		    return TCL_ERROR;
		}
		op = 1;
	    } else {
	        goto nmWrongArgs;
	    }
	} else {
	    goto nmWrongArgs;
	}
	return Notify(interp, op, id, title, text);
    } else if ((length > 1) && (strncmp(argv[1], "bluetooth", length) == 0)) {
        int op;
	CONST char *arg = NULL;

	if ((argc < 3) || (argc > 4)) {
btWrongArgs:
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		argv[0], " bluetooth devices|state|scanmode|myaddress|",
		"remoteaddress ?addr?",	(char *) NULL);
	    return TCL_ERROR;
	}
	length = strlen(argv[2]);
	if ((length > 0) && (strncmp(argv[2], "devices", length) ==  0)) {
	    if (argc != 3) {
	        goto btWrongArgs;
	    }
	    op = 0;
	} else if ((length > 1) && (strncmp(argv[2], "state", length) ==  0)) {
	    if (argc != 3) {
	        goto btWrongArgs;
	    }
	    op = 1;
	} else if ((length > 1) &&
		   (strncmp(argv[2], "scanmode", length) ==  0)) {
	    if (argc != 3) {
	        goto btWrongArgs;
	    }
	    op = 2;
	} else if ((length > 0) &&
		   (strncmp(argv[2], "myaddress", length) ==  0)) {
	    if (argc != 3) {
	        goto btWrongArgs;
	    }
	    op = 3;
	} else if ((length > 0) &&
		   (strncmp(argv[2], "remoteaddress", length) ==  0)) {
	    if (argc != 4) {
	        goto btWrongArgs;
	    }
	    op = 4;
	    arg = argv[3];
	} else {
	    goto btWrongArgs;
	}
	return Bluetooth(interp, op, arg);
    } else if ((length > 1) &&
	       (strncmp(argv[1], "toast", length) == 0)) {
        int dur = 0;
	CONST *text = NULL;

	if ((argc < 3) || (argc > 4)) {
	    Tcl_AppendResult(interp, "wrong # arguments: should be \"",
		argv[0], " toast text ?longdur?\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if ((argc > 3) && Tcl_GetBoolean(interp, argv[3], &dur) != TCL_OK) {
	    return TCL_ERROR;
	}
	return Toast(interp, argv[2], dur);	
    }
    Tcl_AppendResult(interp, "bad option \"", argv[1],
		     "\": must be activity, cancel, queryactivities, ",
		     "queryservices, querybroadcastreceivers, queryconsts, ",
		     "vibrate, beep, speak, stopspeak, isspeaking, "
		     "endspeak, location, networkinfo, shortcut, ",
		     "notification, bluetooth, or toast", (char *) NULL);
    return TCL_ERROR;
}

JNIEnv *
Tkborg_JNIEnv(void)
{
    return GetJNIEnv();
}

int
Tkborg_Init(Tcl_Interp *interp)
{
    if (startupFile != NULL) {
	Tcl_SetVar(interp, "tcl_interactive", "0", TCL_GLOBAL_ONLY);
	Tcl_SetStartupScript(Tcl_NewStringObj(startupFile, -1), NULL);
	free(startupFile);
    }
    Tcl_CreateFileHandler(cbq.pfd[0], TCL_READABLE, FileHandler,
			  (ClientData) NULL);
    Tcl_CreateCommand(interp, "borg", BorgCmd, (ClientData) NULL,
		      (Tcl_CmdDeleteProc *) BorgCmdDeleted);
    return TCL_OK;
}
    
/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * fill-column: 78
 * tab-width: 8
 * End:
 */
