# These routines are used when WiKit is called for use with Tk

# Data structures ...
#
# pageWin    - name of text widget where we display the page contents.
# currMode   - No 'mode', but the string to search for in page titles.
#              Connected to the entry, traced ensures invocation of an
#              incremental search. Also holds the page title when using
#              the internal editor to manipulate a page's contents.
#
# pageStack  - History of pages visited to get to the current page.
#              Used by "Back" button.
# searchPage - Results generated by the incremental search.

package require Tk

if {![catch {package require gbutton}]} {
  package provide Wikit::Gui 1.0
  package require autoscroll

  namespace eval Wikit {
    variable searchPage "" ;# avoid error when viewing before searching

    proc Expand_Tk {str} {
      variable D
      
      set result [StreamToTk [TextToStream $str] ::Wikit::InfoProc]
      
      foreach {a b c} [lindex $result 1] {
	set tag $a$b
	
	$D tag bind $tag <Any-Enter> "$D tag configure $tag -foreground red"
	$D tag bind $tag <Any-Leave> "$D tag configure $tag -foreground blue"
	
	if {$a == "u" || $a == "x"} {
          # better not render in underline, since we don't support
          # browsing in local mode
	  # $D tag configure $tag -font wikit_underline
	}
	
	if {$a == "g"} {
	  set id [LookupPage $c]
	  $D tag bind $tag <ButtonPress> "Wikit::ShowPage $id"
	}
      }
      
      return [lindex $result 0]
    }

    # Show a page containing a list of links to all pages
    # referring to the page with the given id.
    proc ShowRefsPage {id} {
        variable D
        variable readonly
        variable top
        variable topwin

        # Retrieve the title and list of references for the page with this id
        pagevars $id name refs

        # wm title $topwin "References to $name - WiKit"

        gButton::modify Edit -state disabled
        $D configure -state normal -font wiki_default
        $D delete 1.0 end
        $D insert end "References to $name" title
        
        # Generate a wiki formatted list of links.  One link for each reference
        set contents "\n"
        foreach r $refs {
            pagevars $r name format
            append contents "   * \[$name]\n"
            }
        eval $D insert end [Expand_Tk $contents]
        $D configure -state disabled
        focus $top.n.enter
    }

    # Internal helper. Central functionality to render wiki pages in the
    # tk interface. Also associates functionality with the buttons in the
    # toolbar.
    #
    # Argument is the numerical id of the page as stored in the underlying
    # database. If no argument is given the history is used to step back
    # to the last page visited by the user.

    proc ShowPage {{id ""}} {
      variable D
      variable urlSeq
      variable currMode
      variable searchPage
      variable readonly
      variable top
      variable toppage
      set urlSeq 0

      if {!$readonly} {
        if {![string equal [gButton::cget Back -text] Back]} {
          # only update these if changed after edit
          gButton::modify Back -text Back -command "Wikit::ShowPage"
          gButton::modify Forward -text Forward -command "Wikit::ShowPage -1"
          gButton::modify Home -text Home -command "Wikit::ShowPage $toppage"
	  gButton::modify Edit -text Edit -command "Wikit::EditPage"
          bind $D <ButtonRelease-1> {}
        }
        gButton::modify Edit -state normal
      }

      # Record new page in transient viewing history (or retrieve it
      # from the history) and also import the relevant information from
      # the database into this context (the title of the page, its
      # format, and its contents). If any page different from the search
      # page is displayed we reset the entry to prevent further
      # incremental searching from interfering with the display, until
      # the user starts typing again.

      set id [History $id]
      pagevars $id name format refs
      
      if {$id == 2} {
	set page $searchPage
      } else {
	set page [GetPage $id]
      }

      $top.n.mode configure -text Search:
      if {$id != 2} {
	set currMode ""
      }
      
      $D configure -state normal -font wikit_default
      $D delete 1.0 end

      # Setup for title backlinks
      switch [llength $refs] {
        0 {set titleTags title}
        x1 {
            # note I've disabled this - since jumping to the referenced page
            # introduces an inconsistent behaviour and that can cause
            # confusion for users  - stevel
            set titleTags [list title backlink]
            $D tag bind backlink <ButtonPress> "Wikit::ShowPage [lindex $refs 0]"
        }
        default {
            set titleTags [list title backlink]
            $D tag bind backlink <ButtonPress> "Wikit::ShowRefsPage $id"
        }
      }
      

      # First we dynamically generate a script containing commands to
      # update the page window, then we execute the script, thus
      # rendering the page.

      set cmd "$D insert end {$name} {$titleTags} "
      
      if {$format == 0 && $id != 2} {
	append cmd [list \n\n$page fixed] ;# because Expand_Tk inserts \n's
      } else {
	append cmd [Expand_Tk $page]
      }
      eval $cmd
      $D configure -state disabled

      focus $top.n.enter
      if {$id == $toppage} {
          gButton::modify Home -state disabled
      } else {
          gButton::modify Home -state normal
      }
      if {$id == 3} {
          gButton::modify Help -state disabled
      } else {
          gButton::modify Help -state normal
      }
    }

    proc EditPage {} {
      variable D 
      variable currMode 
      variable pageStack
      variable top
      set id [lindex $pageStack end]

      pagevars $id name page
      
      gButton::modify Back -text Cancel -command "Wikit::ShowPage $id" \
      				-state normal
      gButton::modify Forward -text Save -command "
		SavePage $id \[$D get 1.0 end\] local \[$top.n.enter get\]
		Wikit::ShowPage $id" -state normal
      gButton::modify Home -text Copy -state disabled -command Wikit::Copy
      gButton::modify Edit -text Paste -state normal -command Wikit::Paste
      gButton::modify Help -state disabled
      bind $D <ButtonRelease-1> Wikit::CopyCheck

      $top.n.mode configure -text "Edit Title:"
      set currMode $name

      $D configure -state normal -font wikit_fixed
      $D delete 1.0 end
      $D insert end $page fixed \n fixed
      
      $D mark set insert 1.0
      
      focus $D
    }

    proc History {page} {
      variable pageStack 
      variable forwStack

      if {$page == ""} { # pop last page and put on forward stack
	set forwStack [linsert $forwStack 0 [lindex $pageStack end]]
	set pageStack [lreplace $pageStack end end]
	set page [lindex $pageStack end]
	gButton::modify Forward -state normal
      } elseif {$page == -1} { # go to first page on forward stack
	set page [lindex $forwStack 0]
	set forwStack [lreplace $forwStack 0 0]
	lappend pageStack $page
      } else { # push specified page
	set forwStack [list]
	if {$page != [lindex $pageStack end]} {
	  lappend pageStack $page
	}
      }
       
      if {[llength $forwStack] == 0} {
	    gButton::modify Forward -state disabled
      }
      set state normal
      if {[llength $pageStack] <= 1} {set state disabled}
      gButton::modify Back -state $state
      
      # puts "history $page: $pageStack : $forwStack"
      return $page
    }

    proc LocalInterface {{win ""} {page 0}}  {
      global tcl_platform
      variable D
      variable currMode
      variable pageStack
      variable forwStack
      variable readonly
      variable top
      variable topwin
      variable toppage
      variable b0
      variable b1
      set pageStack ""
      set forwStack ""
     
      set toppage $page
      set top $win

      # accomodate platform differences - on some Unices the helvetica font
      # appears smaller for a given point size.
      set family helvetica
      set title 16
      set default 9
      set thin 4
      set buttonsize 9
      switch $tcl_platform(os) {
	  SunOS    -
	  Darwin   { set default 11 ; set buttonsize 10}
	  Linux    { set default 10 }
	  Windows* { set family arial }
      }
      catch {
        font create wikit_default -family $family -size $default
        font create wikit_underline -family $family -size $default \
                                    -underline true
        font create wikit_button -family $family -size $buttonsize
        font create wikit_title -family $family -size $title -weight bold
        font create wikit_fixed -family courier -size $default
        font create wikit_thin -family courier -size $thin
        font create wikit_bold -family $family -size $default -weight bold
        font create wikit_italic -family $family -size $default -slant italic
        font create wikit_bolditalic -family $family -size $default \
                                -weight bold -slant italic

        gButton::init -bg white -font wikit_button -disabledfill darkgrey
      }

      if {$top == ""} {
        set topwin "."
      } else {
        if {![winfo exists $top]} { 
            toplevel $top
        }
        set topwin [winfo toplevel $top]
      }
      if {![winfo exists $top.n]} {
        frame $top.n -relief raised -bg white -bd 1
        frame $top.n.f0 -background white
        set b0 [gButton #auto $top.n.f0]
        $b0 new Back 
        $b0 new Forward 
        $b0 new Home 
        label $top.n.mode -width 7 -anchor e -bg white -font wikit_default
        entry $top.n.enter -textvariable Wikit::currMode
        if {!$readonly} {
          $b0 new Edit 
          gButton::modify Edit -command "Wikit::EditPage"
        }
        $b0 size
        pack $top.n.f0 -side left -padx 0 -pady 0
        pack $top.n.mode -side left -padx 4 -pady 4
        pack $top.n.enter -side left -padx 4 -expand 1 -fill x
        frame $top.n.f1 -background white
        set b1 [gButton #auto $top.n.f1]
        $b1 new Help
        $b1 size
        pack $top.n.f1 -side left -padx 0 -pady 0
        scrollbar $top.scroll -orient vertical -command [list $top.details yview]
        text $top.details -yscrollcommand "$top.scroll set" -width 72 \
                    -height 20 -state disabled -wrap word -font wikit_default \
                    -bg white -relief flat -exportselection 1
        grid $top.n -row 0 -column 0 -columnspan 2 -sticky new
        grid $top.details -row 1 -column 0 -sticky news
        grid $top.scroll -row 1 -column 1 -sticky ns
        grid columnconfigure $topwin 0 -weight 1
        grid rowconfigure $topwin 1 -weight 1
        autoscroll::autoscroll $top.scroll
        set D $top.details
        $D tag configure title -font wikit_title -lmargin1 3 -lmargin2 3
        $D tag configure backlink -foreground blue
        $D tag bind backlink <Any-Enter> \
                        "$D tag configure backlink -foreground red"
        $D tag bind backlink <Any-Leave> \
                        "$D tag configure backlink -foreground blue"
        $D tag configure fixed -font wikit_fixed -lmargin1 3 -lmargin2 3
        $D tag configure body  -font wikit_default -lmargin1 3 -lmargin2 3
        $D tag configure url -font wikit_default -foreground blue
        $D tag configure ul -font wikit_default -lmargin1  3 -lmargin2 30 -tabs 30
        $D tag configure ol -font wikit_default -lmargin1  3 -lmargin2 30 -tabs 30
        $D tag configure dt -font wikit_default -lmargin1  3 -lmargin2  3 -tabs 30
        $D tag configure dl -font wikit_default -lmargin1 30 -lmargin2 30 -tabs 30
        $D tag configure i -font wikit_italic
        $D tag configure b -font wikit_bold
        $D tag configure bi -font wikit_bolditalic

	# support for horizontal lines
        $D tag configure thin -font wikit_thin
        $D tag configure hr -relief sunken -borderwidth 1 -wrap none
        bind $D <Configure> {%W tag configure hr -tabs %w}
        gButton::modify Back -command "Wikit::ShowPage"
        gButton::modify Forward -command "Wikit::ShowPage -1"
        gButton::modify Help -text [lindex [GetTitle 3] 0] \
                                -command "Wikit::ShowPage 3"
      }
      gButton::modify Home -command "Wikit::ShowPage $toppage"
      Wikit::ShowPage $page
      
      trace variable currMode w { after cancel Wikit::KeyTracker; after idle Wikit::KeyTracker }
      
      wm title $topwin [GetTitle $page]
      wm protocol $topwin WM_DELETE_WINDOW [list destroy $topwin]
      update
      after idle raise $topwin
      wm minsize $topwin [winfo width $topwin] [winfo height $topwin]
      if {$topwin == "."} {
          tkwait window $topwin
      }
    }

    proc KeyTracker {v op} {
      variable currMode 
      variable searchKey 
      variable searchLong 
      variable searchPage
      variable pageStack
      
      # don't perform a search if:
      # we're not in browse mode
      #   (in edit mode, the entry field is used for the page title)
      # or
      # we're not on page 2 and the search box is empty
      #   (not on page two prevents searching when going back)
      #   (if the search box is not empty, a search key is present)
      # or
      # we're on page 2 and the search box is empty
      #   (return to previous page)
	
      #puts "KeyTracker [gButton::cget Edit -text] [lindex $pageStack end] $currMode"
      
      set txt [gButton::cget Edit -text] 
      if {$txt != "" && $txt != "Edit"} return
      
      if {$currMode == ""} {
	if {[lindex $pageStack end] == 2} Wikit::ShowPage ;# pop last
	return
      }
      
	# keep iterating while the "currMode" search request changes
      while {1} {
	set cMode $currMode
	set sKey $currMode
	set sLong [regexp {^(.*)\*$} $sKey x sKey]
      
	if {$searchKey == $sKey && $searchLong == $sLong} break
	
	set searchKey $sKey
	set searchLong $sLong
	set rows [SearchList] ;# this takes time
	
	after cancel Wikit::KeyTracker
	update
	if {$currMode != $cMode} continue
	
	set searchPage [SearchResults $rows]
	
	after cancel Wikit::KeyTracker
	update
	if {$currMode == $cMode} break
      }
      
      Wikit::ShowPage 2
    }

    proc Copy {} {
      clipboard clear
      if {![catch {set txt [selection get]}]} {
          clipboard append $txt
      }
    }

    proc Paste {} {
     variable D
     if {![catch {set txt [clipboard get]}] && $txt != ""} {
       $D insert insert [clipboard get]
     }
    }
    
    proc CopyCheck {} {
      if {[catch {set txt [selection get]}] || $txt == ""} {
        set state disabled
      } else {
        set state normal
      }
      gButton::modify Home -state $state
    }

  }
}
