[manpage_begin relation n 0.10.2]
[comment {
# This software is copyrighted 2004 - 2011 by G. Andrew Mangogna.
# The following terms apply to all files associated with the software unless
# explicitly disclaimed in individual files.
# 
# The authors hereby grant permission to use, copy, modify, distribute,
# and license this software and its documentation for any purpose, provided
# that existing copyright notices are retained in all copies and that this
# notice is included verbatim in any distributions. No written agreement,
# license, or royalty fee is required for any of the authorized uses.
# Modifications to this software may be copyrighted by their authors and
# need not follow the licensing terms described here, provided that the
# new terms are clearly indicated on the first page of each file where
# they apply.
# 
# IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY FOR
# DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
# OUT OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY DERIVATIVES
# THEREOF, EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE
# IS PROVIDED ON AN "AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE
# NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
# OR MODIFICATIONS.
# 
# GOVERNMENT USE: If you are acquiring this software on behalf of the
# U.S. government, the Government shall have only "Restricted Rights"
# in the software and related documentation as defined in the Federal
# Acquisition Regulations (FARs) in Clause 52.227.19 (c) (2).  If you
# are acquiring the software on behalf of the Department of Defense,
# the software shall be classified as "Commercial Computer Software"
# and the Government shall have only "Restricted Rights" as defined in
# Clause 252.227-7013 (c) (1) of DFARs.  Notwithstanding the foregoing,
# the authors grant the U.S. Government and others acting in its behalf
# permission to use and distribute the software in accordance with the
# terms specified in this license.
# 
#  *++
# MODULE:
# 
# ABSTRACT:
# 
# $RCSfile: relation.man,v $
# $Revision: 1.39 $
# $Date: 2012/02/26 19:09:04 $
#  *--
}]
[moddesc "Relation Operators"]
[titledesc "Operators for Relation Values"]
[copyright "2004 - 2012 by G. Andrew Mangogna"]
[require ral [opt 0.10.2]]
[description]
This manpage describes the [cmd relation] command.
The [cmd relation]
command is part of the Tcl Relational Algebra Library (ral) package.
The [cmd relation] command defines a set of operators on relations that
constitute the formal algebra of relations that is supported by TclRAL.
The algebra is patterned after that described by Date (Date, C.J.,
[emph "An Introduction to Database Systems"], 8th ed, Pearson Education, 2004,
ISBN 0-321-19784-4, chap. 7)

[para]
Formally a relation has a heading and a body.
The heading, like a tuple heading, defines the attribute names and
data types of the components of the relation.
The body consists of tuples, all of which match the heading.
Because relations are fundamentally sets, several things should be noted:
[list_begin enum]
[enum]
There is no left to right ordering of the attributes.
Like tuples, relations have no real left to right ordering.
The implementation may choose to store attributes in any order
and users should not depend upon certain string representations
nor operate on relations with any operators other than those given here.
[enum]
There is no top to bottom ordering of tuples.
Again the implementation is free to store tuples in any order it chooses.
There are no access operations that depend upon a tuple at some fixed
offset or location.
[enum]
There are no duplicate tuples in the body of a relation value
and there is no concept of a NULL value.
Relations are sets and sets do not have duplicates.
All attributes of all tuples of a relation must contain a valid value of
the data type defined for that attribute.
[list_end]

[para]
Note that all the operators are [emph "read only"] in the sense that
they operate on relation values without modifying them.
All of these operators return either a new relation or scalar value and
do not modify the relation arguments.
Only the [cmd relvar] command can modify, in place,
the relation value that is held in a relation variable.

[section "STRING REPRESENTATION"]
Like any Tcl value, relations have a string representation.
The string representation of a relations is a specially formatted list
consisting of two elements:
[list_begin enum]
[enum]
The relation heading.
[enum]
The relation body.
[list_end]
The relation heading is a list containing an even number of elements
consisting of alternating
attribute names and attribute data types.
It is the same form as a tuple heading.
[para]
The body of a relation consists of a list of tuples.
Each tuple in the body is in a list containing an even number of elements
consisting alternately of an
attribute name and its corresponding attribute value.
[para]
The following is a literal string representation of a relation with
one tuple in its body.
[example_begin]
{DogName string Breed string} {{DogName Fido Breed Poodle}}
[example_end]

[section "SET OPERATIONS"]
[para]
Several set-like operation are supported by TclRAL
([emph e.g.] union and intersection).
These operations work in much the same way as the familiar mathematical
operations on sets.
When applied to relation values, there is a special consideration.
In order to perform set-like operations on relations they must
have the same heading.
This means that the headings must have the same attribute names and
the corresponding attributes must have the same data type.

[section EXAMPLE]
[para]
In the command descriptions that follow,
we will often refer to the example data given below.
We assume that the relation is contained in a Tcl variable with the
same name as the tables given below.
For brevity we assume that the necessary commands have been imported into
the namespace of the example.

[example {
% set DOG {
  {DogName string Breed string} {
      {DogName Fido Breed Poodle}
      {DogName Sam Breed Collie}
      {DogName Spot Breed Terrier}
      {DogName Rover Breed Retriever}
      {DogName Fred Breed Spaniel}
      {DogName Jumper Breed Mutt}
  }
}

% relformat $DOG DOG
+-------+---------+
|DogName|Breed    |
|string |string   |
+-------+---------+
|Fido   |Poodle   |
|Sam    |Collie   |
|Spot   |Terrier  |
|Rover  |Retriever|
|Fred   |Spaniel  |
|Jumper |Mutt     |
+-------+---------+
DOG
---
% set OWNER {
  {OwnerName string Age int City string} {
      {OwnerName Sue Age 24 City Cupertino}
      {OwnerName George Age 35 City Sunnyvale}
      {OwnerName Alice Age 30 City {San Jose}}
      {OwnerName Mike Age 50 City {San Jose}}
      {OwnerName Jim Age 42 City {San Francisco}}
  }
}

% relformat $OWNER OWNER
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
OWNER
-----
% set OWNERSHIP {
  {OwnerName string DogName string Acquired string} {
    {OwnerName Sue DogName Fido Acquired 2001}
    {OwnerName Sue DogName Sam Acquired 2000}
    {OwnerName George DogName Fido Acquired 2001}
    {OwnerName George DogName Sam Acquired 2000}
    {OwnerName Alice DogName Spot Acquired 2001}
    {OwnerName Mike DogName Rover Acquired 2002}
    {OwnerName Jim DogName Fred Acquired 2003}
  }
}

% relformat $OWNERSHIP OWNERSHIP
+---------+-------+--------+
|OwnerName|DogName|Acquired|
|string   |string |string  |
+---------+-------+--------+
|Sue      |Fido   |2001    |
|Sue      |Sam    |2000    |
|George   |Fido   |2001    |
|George   |Sam    |2000    |
|Alice    |Spot   |2001    |
|Mike     |Rover  |2002    |
|Jim      |Fred   |2003    |
+---------+-------+--------+
OWNERSHIP
---------
}]

[section COMMANDS]
[list_begin definitions]

[call [cmd "::ral::relation array"] [arg relationValue] [arg arrayVarName]\
    [arg keyAttr] [arg valueAttr]]

The [cmd array] subcommand converts the relation value
given by [arg relationValue] into an array variable named [arg arrayVarName].
The values of [arg keyAttr] are used as the array indices and the corresponding
values of [arg valueAttr] are the array value.
Note, that if the values of the [arg keyAttr] attribute are not unique then it
is possible that more than one tuple in [arg relationValue] will have the same
value for [arg keyAttr].
In this case the array value from [arg valueAttr]
corresponding to duplicated values of
[arg keyAttr] is arbitrarily chosen by the implementation.
Thus the safe approach is to project the relation value that is to be converted
into an array to be a binary relation or to tag the relation and use the tag
attribute name as [arg keyAttr] thereby insure that no duplicated array indices
can occur.

[example {
% relation array $DOG dogarray DogName Breed
% parray dogarray
dogarray(Fido)   = Poodle
dogarray(Fred)   = Spaniel
dogarray(Jumper) = Mutt
dogarray(Rover)  = Retriever
dogarray(Sam)    = Collie
dogarray(Spot)   = Terrier
}]

[call [cmd "::ral::relation assign"] [arg relationValue]\
    [opt "attrName | attr-var-pair ..."]]

The [cmd assign] subcommand is short hand for
[cmd "tuple assign \[relation tuple [arg relationValue]\] \?...\?"].
This allow moving the values of a relation with a single tuple into
ordinary Tcl variable.

[example {
% relation foreach d $DOG -descending DogName {
    relation assign $d DogName
    puts $DogName
}
Spot
Sam
Rover
Jumper
Fred
Fido
}]

[call [cmd "::ral::relation attributes"] [arg relationValue]]

The [cmd attributes] subcommand returns the names of the attributes of
[arg relationValue] as a list.

[example {
% relation attributes $DOG
DogName Breed
% relation attributes $OWNER
OwnerName Age City
}]

[call [cmd "::ral::relation body"] [arg relationValue]]

The [cmd body] subcommand returns the body of the relation value
given by [arg relationValue].
The body is a list of tuple values, each tuple value being in turn
a list of attribute name / attribute value pairs.
The order of the tuple values in the body list is arbitrary.

[example {
% foreach t [relation body $DOG] {puts $t}
DogName Fido Breed Poodle
DogName Sam Breed Collie
DogName Spot Breed Terrier
DogName Rover Breed Retriever
DogName Fred Breed Spaniel
DogName Jumper Breed Mutt
}]

[call [cmd "::ral::relation cardinality"] [arg relationValue]]

The [cmd cardinality] subcommand returns the number tuples contained
in the body of [arg relationValue].

[example {
% relation cardinality $DOG
6
% relation cardinality $OWNERSHIP
7
}]

[call [cmd "::ral::relation compose"] [arg relationValue1]\
    [arg relationValue2] [opt "-using attrList"] [opt "relationValue3 ..."]]

The [cmd compose] subcommand computes the join of [arg relationValue1]
and [arg relationValue2] but eliminates all of the attributes used in the
join.
The returned relation has a heading the same as the union of the headings
of [arg relationValue1] and [arg relationValue2] minus all of the
join attributes from the two relations.
The returned relation has a body that is the same as that of the
[cmd join] minus any duplicated tuples that result when the join
attributes are eliminated.
As with [cmd join], if the [arg -using] argument are missing, the
join is computed across the attributes in [arg relationValue1] and
[arg relationValue2] that are named the same.
Otherwise the [arg attrList] argument is treated the same as for the
[cmd join] subcommand.

[call [cmd "::ral::relation create"] [arg header]\
        [opt [arg "tuple1 tuple2 ..."]]]

The [cmd create] subcommand returns the relation value described by
the [arg header] and [arg tupleN] arguments.
The [arg header] argument is a list of attribute name / attribute type pairs.
Any tuples given by the [opt [arg tupleN]] arguments are inserted into the
relation value.
Each [arg tupleN] argument is a list of attribute name / attribute value pairs.
This command allows for creating a relation value without having knowledge of
the string representation of a relation value. Compare with the
[cmd "::ral::tuple create"] command.

[example {
% relation create {DogName string Breed string}\
    {DogName Fido Breed Poodle}
{DogName string Breed string} {{DogName Fido Breed Poodle}}
}]

[call [cmd "::ral::relation degree"] [arg relationValue]]

The [cmd degree] subcommand returns the number of attributes in the
heading of [arg relationValue]

[example {
% relation degree $DOG
2
}]

[call [cmd "::ral::relation dict"] [arg relationValue]\
        [arg keyAttr] [arg valueAttr]]

The [cmd dict] subcommand returns a dictionary object whose keys are
the values of the [arg keyAttr] and whose values are the values of the
[arg valueAttr].
Like the [cmd "relation array"] command,
the values of the [arg keyAttr] attribute should be unique
to be assured that the correct corresponding values of the [arg valueAttr]
attribute are in the resulting dictionary.

[example {
% dict for {name breed} [relation dict $::DOG DogName Breed] {
  puts "$name ==> $breed"
  }
Fred ==> Spaniel
Sam ==> Collie
Rover ==> Retriever
Spot ==> Terrier
Fido ==> Poodle
Jumper ==> Mutt
}]

[call [cmd "::ral::relation divide"] [arg dividend] [arg divisor]\
    [arg mediator]]

The [cmd divide] subcommand implements the relational divide operation.
The headings of [arg dividend] and [arg divisor] must be disjoint and
the heading of [arg mediator] must be the union of the [arg dividend]
and [arg divisor] headings.
The returned result is a new relation that has the same heading as
[arg dividend] and contains all the tuples from [arg dividend] whose
corresponding tuples in [arg mediator] include all the tuples in
[arg divisor].
Stated another way,
the result of [cmd divide] subcommand is the maximal set of tuples
from [arg dividend] whose Cartesian product with [arg divisor] is
completely contained in [arg mediator].
Using the data from our ongoing example, then the following example shows
how division can be used to find all the Dogs owned by both Sue and George.

[example {
% set dividend [relation project $::DOG DogName]
{DogName string} {\ 
{DogName Fido}\ 
{DogName Sam}\ 
{DogName Spot}\ 
{DogName Rover}\ 
{DogName Fred}\ 
{DogName Jumper}}
% puts [relformat $dividend Dividend:]
+-------+
|DogName|
|string |
+-------+
|Fido   |
|Sam    |
|Spot   |
|Rover  |
|Fred   |
|Jumper |
+-------+
Dividend:
---------
% set divisor [relation project [relation restrict $::OWNER t\ 
  {[tuple extract $t OwnerName] eq "Sue" ||\ 
  [tuple extract $t OwnerName] eq "George"}] OwnerName]
{OwnerName string} {{OwnerName Sue} {OwnerName George}}
% puts [relformat $divisor Divisor:]
+---------+
|OwnerName|
|string   |
+---------+
|Sue      |
|George   |
+---------+
Divisor:
--------
% set mediator [relation eliminate $::OWNERSHIP Acquired]
{OwnerName string DogName string} {\ 
{OwnerName Sue DogName Fido}\ 
{OwnerName Sue DogName Sam}\ 
{OwnerName George DogName Fido}\ 
{OwnerName George DogName Sam}\ 
{OwnerName Alice DogName Spot}\ 
{OwnerName Mike DogName Rover}\ 
{OwnerName Jim DogName Fred}}
% puts [relformat $mediator Mediator:]
+---------+-------+
|OwnerName|DogName|
|string   |string |
+---------+-------+
|Sue      |Fido   |
|Sue      |Sam    |
|George   |Fido   |
|George   |Sam    |
|Alice    |Spot   |
|Mike     |Rover  |
|Jim      |Fred   |
+---------+-------+
Mediator:
---------
% set quotient [relation divide $dividend $divisor $mediator]
{DogName string} {{DogName Fido} {DogName Sam}}
% puts [relformat $quotient "All dogs owned by both Sue and George"]
+-------+
|DogName|
|string |
+-------+
|Fido   |
|Sam    |
+-------+
All dogs owned by both Sue and George
-------------------------------------
}]

[call [cmd "::ral::relation dunion"] [arg relationValue1]\
    [arg relationValue2] [opt "relationValue3 ..."]]

The [cmd dunion] subcommand returns the set union of two or more relations.
However, unlike the [cmd "relation union"] command, the [cmd dunion]
command insists that all the relation values be disjoint.
All relations must be of the same type.
The result relation has a heading that is the same as any of the arguments
and has a body consisting of all tuples present in any of the
[arg relationValue] arguments.
However, if any duplicated tuples are found, then an error results.
Since the dunion operation is both associative and commutative,
the order of the [arg relationValue] arguments has no effect the result.

[call [cmd "::ral::relation eliminate"] [arg relationValue]\
    [opt "attribute ..."]]

The [cmd eliminate] subcommand returns a new relation that has a heading
equal to that of the [arg relationValue] minus any attributes whose names
are given in the [arg attribute] arguments.
The body of the new relation is the same as the body of [arg relationValue]
removing any tuples that might be duplicated as a result of removing the
attributes.
The [cmd eliminate] subcommand complements the [cmd project] command
in the sense that [cmd eliminate] specifies which attributes
to discard and [cmd project] specifies which attribute to keep.

[example {
% relformat [relation eliminate $::DOG Breed]
+-------+
|DogName|
|string |
+-------+
|Fido   |
|Sam    |
|Spot   |
|Rover  |
|Fred   |
|Jumper |
+-------+
}]

[call [cmd "::ral::relation emptyof"] [arg relationValue]]
The [cmd emptyof] subcommand returns a new relation that has the same
heading as [arg relationValue] but whose cardinality is zero.

[call [cmd "::ral::relation extend"] [arg relationValue]\
    [arg tupleVariable] [opt "attr1 type1 expr1 attr2 type2 expr2 ..."]]

The [cmd extend] subcommand returns a new relation which has the same
heading as [arg relationValue] with zero or more additional attributes.
The first additional attribute is given by [arg attr1] which has
type [arg type1] and its value is
set to the result returned by passing [arg expr1] to the [cmd expr] command.
Subsequent attributes are treated similarly.
As each tuple in the body of [arg relationValue] is considered, its
value is set into the variable whose name is given by the [arg tupleVariable]
argument.
This variable is accessible to the extending expressions so that the
current tuple values of [arg relationValue] are available for computing
the values of the new attributes.

[example {
% relformat [relation extend $::OWNER o AgeInMonths int {[tuple extract $o Age] * 12}]
+---------+---+-------------+-----------+
|OwnerName|Age|City         |AgeInMonths|
|string   |int|string       |int        |
+---------+---+-------------+-----------+
|Sue      |24 |Cupertino    |288        |
|George   |35 |Sunnyvale    |420        |
|Alice    |30 |San Jose     |360        |
|Mike     |50 |San Jose     |600        |
|Jim      |42 |San Francisco|504        |
+---------+---+-------------+-----------+
}]

[call [cmd "::ral::relation extract"] [arg relationValue]\
    [arg attrName] [opt [arg "attrName2 ..."]]]

The [cmd extract] subcommand is short hand for
[cmd "tuple extract \[relation tuple [arg relationValue]\] [arg attrName] \?...\?"].
This command obtains one or more attribute values from a relation that
is of cardinality one.

[call [cmd "::ral::relation foreach"] [arg relationVariable]\
    [arg relationValue] [opt "-ascending | -descending"] [opt attr-list]\
    [arg script]]

The [cmd foreach] subcommand provides a means to iterate through the
body of a relation.
For each tuple in the body of [arg relationValue],
the [arg relationVariable] variable is successively set to a relation
value that contains a single tuple from [arg relationValue] and then
[arg script] is executed.
The order in which the tuples are considered is unspecified unless
a list of sorting attributes is specified by the [arg attr-list] argument.
In this case,
the tuples are visited in the [arg -ascending] order of
the values of the sorting attibutes if the direction option
of [arg -ascending] is supplied or if no direction option is given.
Tuples can be visited in descending order of the sorting attributes if
the [arg -descending] option is given.

[example {
% relation foreach d $DOG -descending DogName {
    puts [tuple extract [relation tuple $d] DogName]
}
Spot
Sam
Rover
Jumper
Fred
Fido
}]

[call [cmd "::ral::relation fromdict"] [arg dictValue]\
    [arg keyattr] [arg keytype] [arg valattr] [arg valtype]]

The [cmd fromdict] subcommand returns a relation value of degree two
that contains all the keys and values from [arg dictValue].
The heading of the returned relation will have two attributes
names [arg keyattr] and [arg valattr] of the corresponding types,
[arg keytype] and [arg valtype].
All the key / value pairs from the dictionary are inserted as tuples in the
returned relation value.
This command provides a simpler means to move data from core Tcl types
into relation values.
Compare with [cmd "relation dict"] which extracts relation attribute values
into Tcl dictionaries.

[example {
% set d [dict create Fido Poodle Sam Collie Spot Terrier]
% relformat [relation fromdict $d DogName string Breed string]
+-------+-------+
|DogName|Breed  |
|string |string |
+-------+-------+
|Fido   |Poodle |
|Sam    |Collie |
|Spot   |Terrier|
+-------+-------+
}]

[call [cmd "::ral::relation fromlist"] [arg listValue]\
    [arg attrname] [arg attrtype]]

The [cmd fromlist] subcommand returns a relation value of degree one
that contains the values in the list given by [arg listValue].
The heading of the returned relation contains a single attribute named,
[arg attrname], whose type is given by, [arg attrtype].
Since relations are sets and sets do not have duplicated values,
any duplicated values in [arg listValue] are silently excluded from the
returned relation value.
Comparing the cardinality of the returned relation with the length of
[arg listValue] can be used to determine of any elements of [arg listValue]
were excluded.
This command provides a simpler means to convert lists into a relation value.
Compare with [cmd "relation list"] which extracts relation attribute values
into Tcl lists.

[call [cmd "::ral::relation group"] [arg relationValue] [arg newattribute]\
    [opt "attr1 attr2 ..."]]

The [cmd group] subcommand creates a new relation from [arg relationValue]
that contains an attribute which is of type [emph Relation].
The [arg newattribute] argument gives the name of the relation valued
attribute and
the [arg attrN] arguments give the names of attributes in
[arg relationValue] that are to be assembled into a relation value.
The returned relation has a heading that is the same as [arg relationValue]
minus the attributes given in the [arg attrN] arguments plus the
new relation attribute given by [arg newattribute].
The resulting body has tuples for each unique set of values for the
remaining ungrouped attributes in the original relation with the
corresponding part of the tuple placed in the new relation valued attribute.
See also the [cmd ungroup] subcommand for the complementary operation.

[example {
% relformat [relation group $::OWNERSHIP DogAcquisition DogName Acquired]
+---------+------------------+
|OwnerName|DogAcquisition    |
|string   |Relation          |
+---------+------------------+
|Sue      |+-------+--------+|
|         ||DogName|Acquired||
|         ||string |string  ||
|         |+-------+--------+|
|         ||Fido   |2001    ||
|         ||Sam    |2000    ||
|         |+-------+--------+|
|George   |+-------+--------+|
|         ||DogName|Acquired||
|         ||string |string  ||
|         |+-------+--------+|
|         ||Fido   |2001    ||
|         ||Sam    |2000    ||
|         |+-------+--------+|
|Alice    |+-------+--------+|
|         ||DogName|Acquired||
|         ||string |string  ||
|         |+-------+--------+|
|         ||Spot   |2001    ||
|         |+-------+--------+|
|Mike     |+-------+--------+|
|         ||DogName|Acquired||
|         ||string |string  ||
|         |+-------+--------+|
|         ||Rover  |2002    ||
|         |+-------+--------+|
|Jim      |+-------+--------+|
|         ||DogName|Acquired||
|         ||string |string  ||
|         |+-------+--------+|
|         ||Fred   |2003    ||
|         |+-------+--------+|
+---------+------------------+
}]

[call [cmd "::ral::relation heading"] [arg relationValue]]

The [cmd heading] subcommand returns the relation heading of the
[arg relationValue].
The heading is a list consisting attributes and corresponding data types.

[example {
% relation heading $DOG
DogName string Breed string
}]

[call [cmd "::ral::relation insert"] [arg relationValue]\
        [opt [arg "name-value-list1 name-value-list2 ..."]]]
The [cmd insert] command returns a new relation value whose heading
is the same as [arg relationValue] and whose body includes all the
tuples of [arg relationValue] plus all the tuples given as the
[arg name-value-listN] arguments.
If inserting any of the [arg name-value-listN] arguments would result
in a duplicate tuple,
then an error is thrown.

[call [cmd "::ral::relation intersect"] [arg relationValue1]\
    [arg relationValue2] [opt "relationValue3 ..."]]

The [cmd intersect] subcommand returns the set intersection of two or
more relations.
All relations have equal headings.
The result relation has a heading that is the same as any of the arguments
and has a body consisting of all tuples present in all of the
[arg relationValue] arguments.
Since the intersection operation is both associative and commutative,
the order of the [arg relationValue] arguments has no effect the result.

[call [cmd "::ral::relation is"] [arg relationValue1] [arg compareop]\
    [arg relationValue2]]

The [cmd is] subcommand returns a boolean value based on performing a
comparison operation between two relation values.
Allowed values of [arg compareop] are:
[list_begin definitions]
[def [cmd "equal | =="]]
Returns 1 if [arg relationValue1] is equal to [arg relationValue2].

[def [cmd "notequal | !="]]
Returns 1 if [arg relationValue1] is not equal to [arg relationValue2].

[def [cmd "propersubsetof | <"]]
Returns 1 if [arg relationValue1] is a proper subset of [arg relationValue2].

[def [cmd "subsetof | <="]]
Returns 1 if [arg relationValue1] is a subset of [arg relationValue2].

[def [cmd "propersupersetof | >"]]
Returns 1 if [arg relationValue1] is a proper superset of [arg relationValue2].

[def [cmd "supersetof | >="]]
Returns 1 if [arg relationValue1] is a superset of [arg relationValue2].

[list_end]

Both [arg relationValue1] and [arg relationValue2] must be of the same
type to be compared.

[example {
% set sDogs [relation restrictwith $DOG {[string match S* $DogName]}]
{DogName string Breed string} {{DogName Sam Breed Collie} {DogName Spot Breed Terrier}}
% relation is $DOG == $sDogs
0
% relation is $DOG != $sDogs
1
% relation is $DOG supersetof $sDogs
1
% relation is $DOG > $sDogs
1
% relation is $DOG propersubsetof $sDogs
0
% relation is $DOG <= $sDogs
0
}]

[call [cmd "::ral::relation isempty"] [arg relationValue]]

The [cmd isempty] subcommand returns the boolean value "1" if the
body of [arg relationValue] does not contain any tuples and "0" otherwise.
The [cmd isempty] subcommand is a convenient short hand for the the command
[cmd "expr \{\[::ral::relation cardinality [arg relationValue]] == 0\}"].

[call [cmd "::ral::relation isnotempty"] [arg relationValue]]

The [cmd isnotempty] subcommand returns the boolean value "1" if the
body of [arg relationValue] contains any tuples and "0" otherwise.
The [cmd isnotempty] subcommand is a convenient short hand for the the command
[cmd "expr \{\[::ral::relation cardinality [arg relationValue]] != 0\}"].

[call [cmd "::ral::relation issametype"] [arg relationValue1]\
        [arg relationValue2]]
The [cmd issametype] subcommand returns the boolean value "1" if the
headings of [arg relationValue1] is the same as that of [arg relationValue2]
and "0" otherwise.
Since the relation headings do not have any intrinsic ordering,
using string comparison of relation headings to determine equality
is not reliable and therefore the [cmd issametype] command should be used
to determine type equality.

[call [cmd "::ral::relation join"] [arg relationValue1] [arg relationValue2]\
    [opt "-using attrList"] [opt "relationValue3 ..."] [opt "-using attrList2"]]

The [cmd join] subcommand performs the [emph natural] join of
[arg relationValue1] with [arg relationValue2].
The join is performed across the given [arg attrList].
If no [arg attrList] are given, then the join is performed across
attributes with the same name in both relations
(because of the [cmd rename] subcommand it is always possible to arrange
for the relations to have common names for the join attributes).
If present, the [arg attrList] is a list of pairs of attribute names.
The even numbered elements
give the name of a join attribute in [arg relationValue1] and the
odd numbered elements give the name of the corresponding
join attribute in [arg relationValue2].
The heading of the resulting relation consists of all the attibutes of
[arg relationValue1] plus all the attributes of [arg relationValue2]
minus any of the join attributes from [arg relationValue2].
The body of the resulting relation consists all tuples composed from the
tuples in both [arg relationValue1] and [arg relationValue2] where the values of
the join attributes in [arg relationValue1] equal those of [arg relationValue2].
Several relations can be joined by optionally listing them as additional
[arg relationValueN] arguments.
Additional [opt -using] arguments may be given for each pair of
[arg relationValue] arguments to specify the attributes across which the
join is to be performed.
The [opt -using] options are a convenient short hand for having to
[cmd rename] attributes to achieve common attribute names.
The join operation is both associative and commutative and so the order
of the [arg relationValueN] does not affect the result.

[example {
% relformat [relation join $OWNERSHIP $DOG]
+---------+-------+--------+---------+
|OwnerName|DogName|Acquired|Breed    |
|string   |string |string  |string   |
+---------+-------+--------+---------+
|Sue      |Fido   |2001    |Poodle   |
|Sue      |Sam    |2000    |Collie   |
|George   |Fido   |2001    |Poodle   |
|George   |Sam    |2000    |Collie   |
|Alice    |Spot   |2001    |Terrier  |
|Mike     |Rover  |2002    |Retriever|
|Jim      |Fred   |2003    |Spaniel  |
+---------+-------+--------+---------+
}]

[call [cmd "::ral::relation list"] [arg relationValue] [opt [arg attrName]]\
    [opt "[opt -ascending|-descending] sortAttrList"]]

The [cmd list] subcommand returns the values of a single attribute
from all the tuples of [arg relationValue] as a proper Tcl list.
If [arg attrName] is not given, then
[arg RelationValue] must be of degree one or an error is returned.
Otherwise the attribute values of [arg attrName] are returned.
Since lists are ordered and tuples in a relation values are not,
an optional [arg sortAttrList] may be specified.
In this case,
the returned list is in the same order as the corresponding values
of the attributes given in [arg sortAttrList].
By default the sorting is [arg -ascending] unless otherwise specified.
If no sorting arguments are given, the returned order is arbitrary
and unspecified.
Note that the list returned from a relation of degree one is necessarily a set
and may be used with the [package struct::set] package.
In this case if the values of the [arg attrName] attribute are not
unique then, in general, the returned list is not a set and may include
duplicated values.

[example {
% relation list [relation project $OWNERSHIP Acquired]]
2001 2000 2002 2003

% relation list $OWNERSHIP Acquired -ascending {OwnerName DogName}
2001 2001 2000 2003 2002 2001 2000
}]

[call [cmd "::ral::relation minus"] [arg relationValue1] [arg relationValue2]]

The [cmd minus] subcommand returns the set difference between two relations.
The [arg relationValue] arguments must be of the same type and that is the
type of the result relation.
The body of the result consists of those tuples present in
[arg relationValue1] but not present in [arg relationValue2].
Note that the order of the [arg relationValue1] and [arg relationValue2]
arguments is significant to the result,
[emph i.e.] [cmd minus] is [emph not] a commutative operation.

[call [cmd "::ral::relation project"] [arg relationValue]\
    [opt "attr1 attr2 ..."]]

The [cmd project] subcommand returns a relation whose heading consists of
only those attributes listed in the [arg attrN] arguments.
The body of the result consists of tuples the corresponding tuples from
[arg relationValue], removing any duplicates created by considering only
a subset of the attributes.

[example {
% relformat [relation project $OWNER City]
+-------------+
|City         |
|string       |
+-------------+
|Cupertino    |
|Sunnyvale    |
|San Jose     |
|San Francisco|
+-------------+
}]

[call [cmd "::ral::relation rank"] [arg relationValue]\
    [opt [arg "-ascending | -descending"]] [arg rankAttr] [arg newAttr]]

The [cmd rank] subcommand returns a new relation whose heading is the same
as [arg relationValue] extending by an attribute named [arg newAttr].
The type of [arg newAttr] will be [type int] and its value will be set
to the number of tuples in [arg relationValue] where the value of
[arg rankAttr] is less than or equal to ([opt -descending]) or
greater than or equal to ([opt -ascending]) that of the given tuple.
The default ranking is [arg -ascending].
The type of [arg rankAttr] must be [type int], [type double], or [type string].

The [cmd rank] command is useful when it is desirable to limit the
number of tuples in the result.
For example, to find the names and ages of the two oldest owners:

[example {
% set ro [relation rank $OWNER -descending Age AgeRank]
{OwnerName string Age int City string AgeRank int} {\ 
{OwnerName Sue Age 24 City Cupertino AgeRank 5}\ 
{OwnerName George Age 35 City Sunnyvale AgeRank 3}\ 
{OwnerName Alice Age 30 City {San Jose} AgeRank 4}\ 
{OwnerName Mike Age 50 City {San Jose} AgeRank 1}\ 
{OwnerName Jim Age 42 City {San Francisco} AgeRank 2}}
% relformat [relation project [relation restrictwith $ro {$AgeRank <= 2}] OwnerName Age]
+---------+---+
|OwnerName|Age|
|string   |int|
+---------+---+
|Mike     |50 |
|Jim      |42 |
+---------+---+
}]

[call [cmd "::ral::relation rename"] [arg relationValue]\
    [opt "oldname newname ..."]]

The [cmd rename] subcommand returns a relation whose heading has each
[arg oldname] attribute changed to [arg newname] and whose body is the
same as [arg relationValue].
An arbitrary number of [arg "oldname / newname"] pairs may be given.
Renaming is processed from left to right in the command arguments
and it is not an error to change the name of any given attribute multiple times.
However, [arg oldname] must always be the name of an attribute at the
time the rename takes place.
The [cmd rename] subcommand is useful for manipulating the attribute names
of a relation to prevent naming conflicts in certain operations ([emph e.g.]
[cmd times] and [cmd join]).

[example {
% relformat [relation rename $OWNER City Town]
+---------+---+-------------+
|OwnerName|Age|Town         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
% relformat [relation rename $OWNER City Town City Village]
unknown attribute name, "City"
}]

[call [cmd "::ral::relation restrict"] [arg relationValue] [arg tupleVariable]\
    [arg expression]]

The [cmd restrict] subcommand returns a new relation that is a
subset (possibly improper) of [arg relationValue].
Each tuple in the body of [arg relationValue] is successively assigned
to [arg tupleVariable] and [arg expression] is evaluated.
The resulting relation has a heading that is the same as [arg relationValue]
and a body consisting of all tuples where [arg expression] evaluated to true.

[example {
% relformat [relation restrict $DOG d {[string match S* [tuple extract $d DogName]]}]
+-------+-------+
|DogName|Breed  |
|string |string |
+-------+-------+
|Sam    |Collie |
|Spot   |Terrier|
+-------+-------+
}]


[call [cmd "::ral::relation restrictwith"] [arg relationValue]\
    [arg expression]]

The [cmd restrictwith] subcommand behaves like the [cmd restrict] subcommand
except that the values of the tuple attributes are
placed in Tcl variables that have
the same name as the attributes.
Each tuple in the body of [arg relationValue] is successively considered
and each attribute of the tuple is assigned to a Tcl variable that is the
same name as the attribute and [arg expression] is evaluated.
The resulting relation has a heading that is the same as [arg relationValue]
and a body consisting of all tuples where [arg expression] evaluated to true.
The [cmd restrictwith] subcommand is roughly equal to performing a
[cmd "tuple assign"] command on each tuple in [arg relationValue] before
evaluating [arg expression].
The attribute variables are created if needed and will overwrite the values
of any variables that are named the same.
Also, the variables are deleted after the command completes.
Also note that [cmd restrictwith] will not function correctly if nested
with a more complex expression that also contains a [cmd restrictwith]
command for a [arg relationValue] with the same heading. In this case
the invocation of one [cmd restrictwith] may very well overwrite the
variables of a separate invocation.
With that restriction in mind,
the [cmd restrictwith] subcommand is often more convenient than
performing all the required [cmd "tuple extract"] commands that are
otherwise required.

[example {
% relformat [relation restrictwith $DOG {[string match S* $DogName]}]
+-------+-------+
|DogName|Breed  |
|string |string |
+-------+-------+
|Sam    |Collie |
|Spot   |Terrier|
+-------+-------+
% puts $DogName
can't read "DogName": no such variable
}]

[call [cmd "::ral::relation semijoin"] [arg relationValue1]\
    [arg relationValue2] [opt "-using attrList"] [opt "relationValue3 ..."]]

The [cmd semijoin] subcommand computes the join of [arg relationValue1]
and [arg relationValue2] but eliminates all of the attributes of
[arg relationValue1]
(or alternatively speaking, projecting all attributes of [arg relationValue2]).
The returned relation has a heading the same as [arg relationValue2]
and a body consisting of those tuples in [arg relationValue2] that
would have been included in the natural join with [arg relationValue1].
As with [cmd join], if the [arg -using] argument are missing, the
join is computed across the attributes in [arg relationValue1] and
[arg relationValue2] that are named the same.
Otherwise the [arg attrList] argument is treated the same as for the
[cmd join] subcommand.
Also like the [cmd join] subcommand,
additional [arg relationValue] arguments may be given and the
result is computed in left to right order.
This implies that the type of the result is always the type of the right
most [arg relationValue].
[emph N.B.] the sense of this command is inverted from previous versions of
this library.
For example, to find all the dogs that have some owner:

[example {
% relformat [relation semijoin $OWNERSHIP $DOG]
+-------+---------+
|DogName|Breed    |
|string |string   |
+-------+---------+
|Fido   |Poodle   |
|Sam    |Collie   |
|Spot   |Terrier  |
|Rover  |Retriever|
|Fred   |Spaniel  |
+-------+---------+
}]

[call [cmd "::ral::relation semiminus"] [arg relationValue1]\
    [arg relationValue2] [opt "-using attrList"] [opt "relationValue3 ..."]]

The [cmd semiminus] subcommand computes the difference between
[arg relationValue2] and the semijoin of [arg relationValue1] and
[arg relationValue2].
The returned relation has a heading equal to that of [arg relationValue2]
and a body consisting of those tuples from [arg relationValue2] which would
[emph not] have been included in the natural join of [arg relationValue1]
and [arg relationValue2].
The optional [arg -using] argument is treated in the same way as for
the [cmd join] subcommand.
Also like the [cmd semijoin] subcommand, additional [arg relationValue]
arguments may be given.
[emph N.B.] the sense of this command is inverted from previous versions of
this library.
For example, to find all dogs that have no owner:

[example {
% relformat [relation semiminus $OWNERSHIP $DOG]
+-------+------+
|DogName|Breed |
|string |string|
+-------+------+
|Jumper |Mutt  |
+-------+------+
}]

[call [cmd "::ral::relation summarize"] [arg relationValue] [arg perRelation]\
    [arg relationVarName] [arg attr] [arg type] [arg expression]\
    [opt "[arg attr] [arg type] [arg expression] ..."]]

The [cmd summarize] subcommand allows for computations across sets of tuples.
The [arg perRelation] must be of the same type as a projection of
[arg relationValue].
The returned relation has the same heading as [arg perRelation] extended
by [arg attr].
The [arg attr] gives the name of the new attribute.
The name of the new attribute may not match any existing attributes in
[arg perRelation].
The [arg type] argument is data type of the new attribute.
Several new attributes may be added in one [cmd summarize] command.
The body of the returned relation consists of all the tuples of
[arg perRelation] with values for the new attributes.
Those values are computed by evaluating the [arg summmary] arguments
as an expression.
Before evaluating [arg expression] for a tuple in [arg perRelation],
[arg relationVarName] is set to a relation value consisting of
all the tuples in [arg relationValue] whose attributes match the
corresponding attributes in [arg perRelation].
The [arg expression] expression may then access the value of that relation
using the given [arg relationVarName] variable name.
The [arg expression] must return a result that matches the
type of the new attribute.

In this example we determine the years when dogs were acquired and the
number of dogs acquired in each year.

[example {
% set ac [relation project $OWNERSHIP Acquired]
{Acquired string} {{Acquired 2001} {Acquired 2000} {Acquired 2002} {Acquired 2003}}
% relformat [relation summarize $OWNERSHIP $ac s NumAcquired int {[relation cardinality $s]}]
+--------+-----------+
|Acquired|NumAcquired|
|string  |int        |
+--------+-----------+
|2001    |3          |
|2000    |2          |
|2002    |1          |
|2003    |1          |
+--------+-----------+
}]

The [cmd summarize] command can also be used to perform
more complicated computations. In the case of summarizing over the
[emph dee] relation gives the ability to compute overall totals or
averages.
In this example we compute the average age of the owners.

[example {
proc ravg {rel attr} {
    set sum 0
    relation foreach r $rel {
        tuple assign [relation tuple $r]
        incr sum $Age
    }
    return [expr {$sum / [relation cardinality $rel]}]
}

% set dee {{} {{}}}
{} {{}}
% relformat [relation summarize $OWNER $dee o AverageAge int {[ravg $o Age]}]
+----------+
|AverageAge|
|int       |
+----------+
|36        |
+----------+
}]

[call [cmd "::ral::relation summarizeby"] [arg relationValue] [arg attrList]\
    [arg relationVarName] [arg attr] [arg type] [arg expression]\
    [opt "[arg attr] [arg type] [arg expression] ..."]]

The [cmd summarizeby] subcommand is a more convenient form of
[cmd summarize] where the [emph per] relation is a projection of the
relation value that is to be summarized.
Rather than supplying a [emph per] relation, instead a list of attributes
is given by the [arg attrList] argument and [arg relationValue] is
projected on those attributes and used as the [arg per] relation.
The arguments and results are otherwise the same as for the [cmd summarize]
command.
[example {
% relformat [relation summarizeby $OWNERSHIP Acquired s\ 
    NumAcquired int {[relation cardinality $s]}]
+--------+-----------+
|Acquired|NumAcquired|
|string  |int        |
+--------+-----------+
|2001    |3          |
|2000    |2          |
|2002    |1          |
|2003    |1          |
+--------+-----------+
}]

[call [cmd "::ral::relation table"] [arg heading]\
        [arg [opt "value-list1 value-list2 ..."]]]
The [cmd table] subcommand creates a new relation whose heading
is given by [arg heading] and tuples of the body are given by the
[arg value-listN] arguments.
The [arg heading] is an list containing an even number of elements
consisting of alternating attribute names and attribute data types.
The [arg value-listN] arguments are a list of attribute values.
It is assumed that the attribute values are given in the same order as
the attributes of the [arg heading] argument.
This command is a convenience command that is much like [cmd "relation create"]
but avoids having to give the attribute name for each attribute value
when specifying the tuples of the body.
This allows literal relation values to be specified in a tabular layout.

[example {
set o [relation table {\ 
    OwnerName string   Age int     City string     } {
    Sue                24          Cupertino       } {
    George             35          Sunnyvale       } {
    Alice              30          {San Jose}      }]
}]

[call [cmd "::ral::relation tag"] [arg relationValue] [arg attrName]\
    [opt [arg "-ascending | -descending sort-attr-list"]]\
    [opt [arg "-within attr-list"]]]

The [cmd tag] subcommand creates a new relation which has the same heading
as [arg relationValue] extended by a new attribute named [arg attrName].
The type of [arg attrName] will be [type int] and will have the values
between 0 and the cardinality of [arg relationValue] minus one.
The tuples in [arg relationValue] will be extended in either [arg ascending]
or [arg descending] order of the [arg sort-attr-list].
If no [arg sort-attr-list] argument is given,
then the tagging order is arbitrary.
If the [opt [arg -within]] argument is given then
the values of [arg attrName] attribute will be unique within the subset of
tuples which match the values of [arg attr-list].

The [cmd tag] command is useful when a full ordering needs to be placed
on a relation.
For example, tagging a relation will allow projecting both the tag and
another attribute without losing any values.

[example {
% relformat [relation tag $DOG -ascending DogName DogId]
+-------+---------+-----+
|DogName|Breed    |DogId|
|string |string   |int  |
+-------+---------+-----+
|Fido   |Poodle   |0    |
|Fred   |Spaniel  |1    |
|Jumper |Mutt     |2    |
|Rover  |Retriever|3    |
|Sam    |Collie   |4    |
|Spot   |Terrier  |5    |
+-------+---------+-----+
% relformat [relation tag $OWNERSHIP -ascending OwnerName -within DogName DogId]
+---------+-------+--------+-----+
|OwnerName|DogName|Acquired|DogId|
|string   |string |string  |int  |
+---------+-------+--------+-----+
|Alice    |Spot   |2001    |0    |
|George   |Fido   |2001    |0    |
|George   |Sam    |2000    |0    |
|Jim      |Fred   |2003    |0    |
|Mike     |Rover  |2002    |0    |
|Sue      |Fido   |2001    |1    |
|Sue      |Sam    |2000    |1    |
+---------+-------+--------+-----+
}]


[call [cmd "::ral::relation tclose"] [arg relationValue]]

The [cmd tclose] subcommand returns a new relation value that is the
transitive closure of [arg relationValue].
The [arg relationValue] must be a binary relation and each attribute must
be of the same type.
The returned relation has a heading that is the same as [arg relationValue].
The pairs of attribute values in [arg relationValue] may be considered as
defining the edges of a graph.
The body of the transitive closure will contain all pairs of the two
attributes where there exists some path between the vertices in the implied
graph.
The original [arg relationValue] is necessarily a subset of the transitive
closure relation.

The [cmd tclose] subcommand often useful when there is a hierarchical
relationship among the data values.
Consider the employee / supervisor reporting relationship.
If we are interested in all people who report directly or indirectly
to John, then we have:

[example {
% set REPORTS_TO {
  {Employee string Supervisor string}
  {
    {Employee Sue Supervisor John}
    {Employee Bob Supervisor John}
    {Employee Jane Supervisor Sue}
    {Employee Joe Supervisor Sue}
    {Employee Carl Supervisor Bob}
    {Employee James Supervisor Alice}
  }
}

% relformat [relation project [relation restrictwith\ 
[relation tclose $REPORTS_TO] {$Supervisor eq "John"}] Employee]\ 
"Direct/Indirect Reports to John"
+--------+
|Employee|
|string  |
+--------+
|Sue     |
|Bob     |
|Jane    |
|Joe     |
|Carl    |
+--------+
Direct/Indirect Reports to John
-------------------------------
}]

[call [cmd "::ral::relation times"] [arg relationValue1] [arg relationValue2]\
    [opt "relationValue3 ..."]]

The [cmd times] subcommand returns the extended Cartesian product of two or
more relations.
The heading of the result is the union of the headings of all
the [arg relationValue] arguments.
The body of the result consists of a tuple for all combinations of the tuples
in all the [arg relationValue] arguments.
Since the relational multiplication operation is both
associative and commutative,
the order of the [arg relationValue] arguments has no effect on the result.

[example {
% relformat [relation times $DOG $OWNER]
+-------+---------+---------+---+-------------+
|DogName|Breed    |OwnerName|Age|City         |
|string |string   |string   |int|string       |
+-------+---------+---------+---+-------------+
|Fido   |Poodle   |Sue      |24 |Cupertino    |
|Fido   |Poodle   |George   |35 |Sunnyvale    |
|Fido   |Poodle   |Alice    |30 |San Jose     |
|Fido   |Poodle   |Mike     |50 |San Jose     |
|Fido   |Poodle   |Jim      |42 |San Francisco|
|Sam    |Collie   |Sue      |24 |Cupertino    |
|Sam    |Collie   |George   |35 |Sunnyvale    |
|Sam    |Collie   |Alice    |30 |San Jose     |
|Sam    |Collie   |Mike     |50 |San Jose     |
|Sam    |Collie   |Jim      |42 |San Francisco|
|Spot   |Terrier  |Sue      |24 |Cupertino    |
|Spot   |Terrier  |George   |35 |Sunnyvale    |
|Spot   |Terrier  |Alice    |30 |San Jose     |
|Spot   |Terrier  |Mike     |50 |San Jose     |
|Spot   |Terrier  |Jim      |42 |San Francisco|
|Rover  |Retriever|Sue      |24 |Cupertino    |
|Rover  |Retriever|George   |35 |Sunnyvale    |
|Rover  |Retriever|Alice    |30 |San Jose     |
|Rover  |Retriever|Mike     |50 |San Jose     |
|Rover  |Retriever|Jim      |42 |San Francisco|
|Fred   |Spaniel  |Sue      |24 |Cupertino    |
|Fred   |Spaniel  |George   |35 |Sunnyvale    |
|Fred   |Spaniel  |Alice    |30 |San Jose     |
|Fred   |Spaniel  |Mike     |50 |San Jose     |
|Fred   |Spaniel  |Jim      |42 |San Francisco|
|Jumper |Mutt     |Sue      |24 |Cupertino    |
|Jumper |Mutt     |George   |35 |Sunnyvale    |
|Jumper |Mutt     |Alice    |30 |San Jose     |
|Jumper |Mutt     |Mike     |50 |San Jose     |
|Jumper |Mutt     |Jim      |42 |San Francisco|
+-------+---------+---------+---+-------------+
}]

[call [cmd "::ral::relation tuple"] [arg relationValue]]

The [cmd tuple] subcommand returns the body of [arg relationValue] as a tuple.
The cardinality of [arg relationValue] must be one or an error is returned.

[call [cmd "::ral::relation uinsert"] [arg relationValue]\
        [opt [arg "name-value-list1 name-value-list2 ..."]]]

The [cmd uinsert] subcommand operates in the same fashion as the
[cmd insert] subcommand, except that attempts to insert duplicate
tuples are silently ignored.
Thus the [cmd uinsert] subcommand has union type semantics.

[call [cmd "::ral::relation ungroup"] [arg relationValue] [arg attribute]]

The [cmd ungroup] subcommand maps relations that have relation valued
attributes to relations that have scalar valued attributes.
It is the complementary operation to the [cmd group] command.
The [arg attribute] argument must be the name of a relation valued
attribute of [arg relationValue].
The returned relation has a heading consisting of all the attributes of
[arg relationValue] minus [arg attribute] plus all the attributes of the
[arg attribute] relation.
The body of the returned relation consists of tuples composed of pairing
each tuple from the [arg attribute] attribute with its corresponding
components from [arg relationValue].

[example {
set OwnerContacts {
    {OwnerName string Contact {Relation {Type string Number string} Type}} {
	{OwnerName Sue Contact {
	    {Type Home Number 555-1212}
	    {Type Work Number 555-6635}}
	}
	{OwnerName George Contact {
	    {Type Home Number 555-8810}
	    {Type Work Number 555-1177}
	    {Type Mobile Number 555-3399}}
	}
    }
}
% relformat $OwnerContacts
+---------+-----------------+
|OwnerName|Contact          |
|string   |Relation         |
+---------+-----------------+
|Sue      |+------+--------+|
|         ||Type  |Number  ||
|         ||string|string  ||
|         |+------+--------+|
|         ||Home  |555-1212||
|         ||Work  |555-6635||
|         |+------+--------+|
|George   |+------+--------+|
|         ||Type  |Number  ||
|         ||string|string  ||
|         |+------+--------+|
|         ||Home  |555-8810||
|         ||Work  |555-1177||
|         ||Mobile|555-3399||
|         |+------+--------+|
+---------+-----------------+
% relformat [relation ungroup $OwnerContacts Contact]
+---------+------+--------+
|OwnerName|Type  |Number  |
|string   |string|string  |
+---------+------+--------+
|Sue      |Home  |555-1212|
|Sue      |Work  |555-6635|
|George   |Home  |555-8810|
|George   |Work  |555-1177|
|George   |Mobile|555-3399|
+---------+------+--------+
}]

[call [cmd "::ral::relation union"] [arg relationValue1]\
    [arg relationValue2] [opt "relationValue3 ..."]]

The [cmd union] subcommand returns the set union of two or more relations.
All relations must be of the same type.
The result relation has a heading that is the same as any of the arguments
and has a body consisting of all tuples present in any of the
[arg relationValue] arguments.
Since the union operation is both associative and commutative,
the order of the [arg relationValue] arguments has no effect the result.

[call [cmd "::ral::relation unwrap"] [arg relationValue] [arg attribute]]
The [cmd unwrap] subcommand operates on relation values in a manner similar to
the way that the [cmd "tuple unwrap"] command operates on tuple values.
The [arg attribute] argument is the name of a tuple valued attribute.
The result relation has the same heading as [arg relationValue] minus
the [arg attribute] plus all the attributes contained in the
tuple valued attribute, [arg attribute].
In essense the tuple valued attribute, [arg attribute], is flattened
by one level.

[call [cmd "::ral::relation update"] [arg relationValue] [arg tupleVarName]\
    [arg expr] [arg script]]
The [cmd update] subcommand returns a new relation value that has the
same heading as [arg relationValue] and whose tuple have been modified.
Each tuple in [arg relationValue] is assigned to the Tcl variable
given by [arg tupleVarName] and [arg expr] is evaluated.
If [arg expr] is true then [arg script] is evaluated.
The return value of [arg script] is added to the result relation and it
must be a tuple that has the same heading as [arg relationValue].
If [arg expr] is false then the unmodified tuple from [arg relationValue]
is added to the result.
This command serves the same role for relation values as the
[cmd "relvar update"] command does for relvars.


[call [cmd "::ral::relation wrap"] [arg relationValue] [arg tupleattr]\
    [opt [arg "attr1 attr2 ..."]]]
The [cmd wrap] subcommand operates on relation values in a manner similar
to the way that the [cmd "tuple wrap"] command operates on tuple values.
The result relation has a heading that is the same as [arg relationValue]
plus an attribute named, [arg tupleattr], minus the [arg attrN] attributes.
The values of the [arg attrN] attributes are converted into a tuple
valued attribute that is stored in [arg tupleattr].

[example {
% set r {
    {Name string Street string City string State string Zip string} {
        {Name Arnold Street Main City Sacramento State CA Zip 94444}
        {Name Mary Street Elm City Stockton State CA Zip 95555}
    }
}
% relformat [relation wrap $r Address Street City State Zip]
+------+---------------------------------+
|Name  |Address                          |
|string|Tuple                            |
+------+---------------------------------+
|Arnold|+------+----------+------+------+|
|      ||Street|City      |State |Zip   ||
|      ||string|string    |string|string||
|      |+------+----------+------+------+|
|      ||Main  |Sacramento|CA    |94444 ||
|      |+------+----------+------+------+|
|Mary  |+------+--------+------+------+  |
|      ||Street|City    |State |Zip   |  |
|      ||string|string  |string|string|  |
|      |+------+--------+------+------+  |
|      ||Elm   |Stockton|CA    |95555 |  |
|      |+------+--------+------+------+  |
+------+---------------------------------+
}]

[list_end]

[see_also relvar tuple]

[keywords relation tuple body]

[manpage_end]
