[manpage_begin relvar n 0.10.2]
[comment {
# This software is copyrighted 2004 - 2011 by G. Andrew Mangogna.
# The following terms apply to all files associated with the software unless
# explicitly disclaimed in individual files.
# 
# The authors hereby grant permission to use, copy, modify, distribute,
# and license this software and its documentation for any purpose, provided
# that existing copyright notices are retained in all copies and that this
# notice is included verbatim in any distributions. No written agreement,
# license, or royalty fee is required for any of the authorized uses.
# Modifications to this software may be copyrighted by their authors and
# need not follow the licensing terms described here, provided that the
# new terms are clearly indicated on the first page of each file where
# they apply.
# 
# IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY FOR
# DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
# OUT OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY DERIVATIVES
# THEREOF, EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE
# IS PROVIDED ON AN "AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE
# NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
# OR MODIFICATIONS.
# 
# GOVERNMENT USE: If you are acquiring this software on behalf of the
# U.S. government, the Government shall have only "Restricted Rights"
# in the software and related documentation as defined in the Federal
# Acquisition Regulations (FARs) in Clause 52.227.19 (c) (2).  If you
# are acquiring the software on behalf of the Department of Defense,
# the software shall be classified as "Commercial Computer Software"
# and the Government shall have only "Restricted Rights" as defined in
# Clause 252.227-7013 (c) (1) of DFARs.  Notwithstanding the foregoing,
# the authors grant the U.S. Government and others acting in its behalf
# permission to use and distribute the software in accordance with the
# terms specified in this license.
# 
#  *++
# MODULE:
#   relvar.man -- documentation for the ::ral::relvar command
# ABSTRACT:
#   This file contains the documentation for the ::ral::relvar command.
#   This command is part of TclRAL. This file is in Tcl Markup Language
#   format.
# $RCSfile: relvar.man,v $
# $Revision: 1.36 $
# $Date: 2012/02/26 19:09:04 $
#  *--
}]
[moddesc "Relvar Operators"]
[titledesc "Operators for Relation Variables"]
[copyright "2004 - 2012 by G. Andrew Mangogna"]
[require ral [opt 0.10.2]]
[description]
This manpage describes the [cmd relvar] command.
The [cmd relvar] command is part of the Tcl Relational Algebra Library
(ral) package.

[para]
TclRAL defines a variable space for relation variables.
Relation variables hold relation values and the subcommands of the
[cmd relvar] command generally operate directly on the values storing
the results back into the relation variables.
For this reason, most of the [cmd relvar] subcommands
take the name of a relation variable as an argument.

[para]
In the relational model as it applies to database management systems,
relation variables play the important role of
representing the persistent data base values.
Relation variables are the leaves of the expression tree for a query and
represent the core data from which results may be computed.
That computation is usually expressed as a single, sometimes complex expression.
This allows for a non-procedural way of specifying the computation and
allows optimizers to find the most efficient manner of computing the result.

[para]
In the TclRAL implementation of the relational model,
relation variables serve a similar role although there
is no implicit persistence supplied by the library.
Relation values may be stored in Tcl variables like any other Tcl object.
This does abandon the non-procedural nature of relational expressions
as usually accomplished in the relational model, but is a better fit
to the Tcl way of doing things.

[section "RELVAR NAMES"]

Relvar names follows the familiar Tcl method of naming
ordinary Tcl variables and thus relvars have Tcl namespace-like names.
For example, a relvar named [var Dog] created at the global level
will be named [var ::Dog].
However, relvars names are held separately from Tcl variable names
and despite the similar syntax and appearance,
relvars are a concept only of TclRAL.
Thus it is possible and indeed encouraged to group related relvars
together in a namespace.
If an application uses only one set of relvars,
then the global namespace of relvars is most convenient.
Using several distinct sets of relvars is best accomplished if each
set is placed in their own relvar namespace.

[para]
Since it is possible to store a relation value in an ordinary Tcl variable
and since Tcl variables are convenient to use when formulating relational
expressions ([emph i.e.] it is sometimes more convenient to say [var \$Dog]
rather than [cmd "\[relvar set Dog]"]),
creating a relvar also creates a corresponding Tcl variable by the same name.
The Tcl variable created is traced to prevent accidental writing to it,
since all modifications to a relvar should come via the [cmd relvar] commands.
So, creating a relvar named [var Dog] at the global level creates a
relvar named [var ::Dog] and a read-only Tcl variable also named [var ::Dog].
However, it should be noted that the automatically created Tcl variable is
not protected against [emph unset].
It is allowed to unset the Tcl variable and still access the relation value
stored in the relvar.

[section CONSTRAINTS]

TclRAL supports the notion of [emph constraints] as a means of
insuring integrity among a set of relvars.
There are many types of constraints that might be used to insure integrity,
but TclRAL supports three distinct types of constraints:
identification constraints, referential constraints and procedural constraints.
[list_begin definitions]

[def Identification]
Identification constraints define a set of attributes that must have
unique values.
Each relvar must specify at least one identifier.
An identifier is one or more attributes whose values taken together must
be unique for all tuples stored in the relvar.
Identifiers must also be minimal in the sense that no two identifiers
for a given relvar are allowed to be a subset of one another.

[def Referential]
Referential constraints define how tuples of one relvar refer to tuples
in another relvar.

[def Procedural]
Procedural constraints define a Tcl script that is evaluated to determine
if the constraint is satisfied.

[list_end]

Identification constraints are enforced on a relvar by any operation
that modifies the relation value stored in the relvar.

For referential types of constraints three forms are supported.

[list_begin definitions]
[def Association]
Association constraints define a referential association between two relvars.
In this context,
an association declares that a set of attributes in one relvar refers to
an identifier of another relvar.
Thus for all the tuples in a given relvar,
the given attribute values must match the corresponding attribute values
of the identifier of the associated relvar.
The association also declares the number of times that such matches may
occur and if it is allowed not to match at all.

[def Partition]
Partition constraints define a complete and disjoint
referential partitioning between one relvar,
known as the super set,
and a given set of relvars,
known as the sub sets.
This constraint declares that every tuple of the super set relvar is
referred to by exactly one tuple from among all the tuples in the
sub set relvars.

[def Correlation]
Correlation constraints define a referential association between two
relvars that is mediated by a third relvar.
The third relvar, the correlation relvar, contains attributes that
reference identifiers in the two associated relvars.
It is similar in some ways to two Association Constraints.

[list_end]

Procedural constraints are enforced by evaluating a Tcl script when
a given set of relvars are modified.

At the end of each [cmd ::ral::eval] command or after a
[cmd "::ral::transaction end"] command(see below) or,
if outside of an [cmd eval] command,
at the end of any command that modifies a relvar
that participates in a referential constraint,
all of the referential and procedural constraints defined for the
modified relvars are evaluated.
If any of the constraints fail, the relvar modifications are discarded
and the relvar values are returned to the values they had before
the modifications.
Similarly,
during the creation of a constraint, the constraint is evaluated
on all the relvars associated with the constraint.
If the evaluation fails,
the constraint creation fails.

Like relvars, constraints have names.
Constraints names follow the same pattern as relvar names, i.e.
unresolved names are created in the current namespace.
Thus you are able to keep the relvars and their constraints organized
together in namespace-like groups.

[section COMMANDS]
[list_begin definitions]
[call [cmd "::ral::relvar association"] [arg name]\
    [arg refrngRelvar] [arg refrngAttrList] [arg refToSpec]\
    [arg refToRelvar] [arg refToAttrList] [arg refrngSpec]\
]

The [cmd association] subcommand creates a referential association
between two relvars called [arg name].
The return value of [cmd association] is the empty string.
Associations declare that one relvar has attributes that refer to
an identifier of another relvar.
The referring relvar name is given by the [arg refrngRelvar] argument
and the referring attributes are given by [arg refrngAttrList] argument.
The referred to relvar name is given by [arg refToRelvar] and the
attributes that are referred to by [arg refToAttrList]. The referring
and referred to attribute lists are ordered and referring attributes
refer to the corresponding referred to attribute.
The set of attributes given by [arg refToAttrList] must constitute
an identifier for [arg refToRelvar].
The [arg refToSpec] argument defines
the multiplicity and conditionality of the reference.
It can be one of:

[list_begin definitions]

[def *]
Each tuple in [arg refToRelvar] is referred to zero or more times by
the tuples in [arg refrngRelvar].

[def +]
Each tuple in [arg refToRelvar] is referred to one or more times by
the tuples in [arg refrngRelvar].

[def 1]
Each tuple in [arg refToRelvar] is referred to exactly once by
the tuples in [arg refrngRelvar].

[def ?]
Each tuple in [arg refToRelvar] is referred to zero or one times by
the tuples in [arg refrngRelvar].

[list_end]

Similarly [arg refrngSpec] defines the multiplicity and conditionality
associated with the references to [arg refToRelvar].
Necessarily, [arg refrngSpec] may only be "1" or "?" implying:
[list_begin definitions]

[def 1]
Each tuple in [arg refrngRelvar] refers to exactly one tuple in
[arg refToRelvar].

[def ?]
Each tuple in [arg refrngRelvar] refers to at most one tuple
(and possibly none at all) in [arg refToRelvar].

[list_end]

The symbols for [arg refrngSpec] and [arg refToSpec] were chosen
by analogy to the use of those symbols in regular expression syntax.

[example {
% relvar association A1 OWNERSHIP OwnerName + OWNER OwnerName 1
% relvar association A2 OWNERSHIP DogName * DOG DogName 1
}]

In this example,
A1 declares that for every tuple in OWNERSHIP,
the value of OWNERSHIP.OwnerName matches the value of OWNER.OwnerName
in exactly one tuple of OWNER and that for every tuple in OWNER the
value of OWNER.OwnerName matches the value of OWNERSHIP.OwnerName in
at least one tuple of OWNERSHIP and possibly more.
In the semantics of this example, the concepts of being an OWNER
and that of OWNERSHIP of a DOG are unconditionally associated since
no tuple of OWNER may exist without there being at least one associated
tuple of OWNERSHIP.
Also this constraint allows for OWNERS that may own more than one DOG.

Compare this to the case of A2.
A2 states that every tuple in OWNERSHIP must have a corresponding tuple
in DOG, but that tuples in DOG may not be referenced by any tuple in OWNERSHIP.
OWNERSHIP and DOG are conditionally associated
and in the semantics of this problem, it is possible to have dogs that 
are not owned at all and to have dogs that are owned multiple times.

[call [cmd "::ral::relvar constraint delete"]\
	[opt "[arg name1] [arg name2] ..."]]
The [cmd "constraint delete"] subcommand deletes all the
constraints given by the [arg nameN] arguments.
The return value is the empty string.

[call [cmd "::ral::relvar constraint exists"] [arg name]]
The [cmd "constraint exists"] subcommand returns a boolean value of
true if the relvar named [arg name] exists and false otherwise.
The relvar [arg name] argument is resolved according to the rules
described above.

[call [cmd "::ral::relvar constraint info"] [arg name]]
The [cmd "constraint info"] subcommand returns the detailed information
about the constraint named [arg name].
The information returned is of the same format as the command used
to create the constraint minus the superfluous leading [cmd relvar] command
word.
Additionally,
the relvar names and constraint names are returned in fully qualified form.

[example {
% relvar constraint info A1
association ::A1 ::OWNERSHIP OwnerName + ::OWNER OwnerName 1
% relvar constraint info ::P1
partition ::P1 ::Lamp SerialNo ::TableLamp SerialNo ::FloorLamp SerialNo
% relvar constraint info G
unknown constraint name, "G"
}]

[call [cmd "::ral::relvar constraint names"] [opt [arg pattern]]]

The [cmd "constraint names"] subcommand returns a list of constraint names.
If the optional [arg pattern] is present,
then only those constraint names that match [arg pattern] are returned.
Name matching happens in the same manner as for the [cmd "string match"].
The returned names are fully qualified.

[example {
% relvar constraint names
::A1 ::A2
% relvar constraint names *2
::A2
}]

[call [cmd "::ral::relvar constraint member"] [arg relvarName]]

The [cmd "constraint member"] subcommand returns a list of constraint names
in which [arg relvarName] participates in some fashion.
The returned names are fully qualified.

[example {
% relvar constraint member OWNERSHIP
::A1 ::A2
}]

[call [cmd "::ral::relvar constraint path"] [arg constraintName]]
The [cmd "constraint path"] subcommand returns the fully qualified constraint
name of [arg constraintName].

[call [cmd "::ral::relvar correlation"] [arg [opt -complete]] [arg name]\
[arg correlRelvar]\
[arg correlAttrListA] [arg refToSpecA] [arg refToRelvarA] [arg refToAttrListA]\
[arg correlAttrListB] [arg refToSpecB] [arg refToRelvarB] [arg refToAttrListB]]

The [cmd correlation] subcommand creates a referential correlation
called [arg name].
The referential correlation is between a correlation relvar, [arg correlRelvar],
and two other relvars, [arg refToRelvarA] and [arg refToRelvarB].
The return value of the [cmd correlation] command is the empty string.

Correlations declare that every tuple in [arg correlRelvar] refers to
exactly one tuple of [arg refToRelvarA] and exactly one tuple of
[arg refToRelvarB].
The [arg correlAttrListA] attributes in [arg correlRelvar] refer to an
identifier of [arg refToRelvarA] that is given by the
attributes contained in the [arg refToAttrListA] list.
Similarly,
the [arg correlAttrListB] attributes in [arg correlRelvar] refer to an
identifier of [arg refToRelvarB] that is given by [arg refToAttrListB].
Both [arg refToAttrListA] and [arg refToAttrListB] must constitute
an identifier for their corresponding relvars.
The [arg refToSpecA] and [arg refToSpecB] arguments can have the same values
as for the [cmd "::ral::relvar association"] command and specify
the multiplicity and conditionality of how [arg refToRelvarA] and
[arg refToRelvarB] are referred to by the tuples of [arg correlRelvar].
If the [arg -complete] option is given,
then every tuple of [arg refToRelvarA] must be correlated to every
tuple of [arg refToRelvarB] and [emph "vice versa"].
The [arg -complete] option implies
the cardinality of [arg correlRelvar] must equal the product
of the cardinality of [arg refToRelvarA] and [arg refToRelvarB].
If the [arg -complete] option is missing,
then [arg correlRelvar] is allowed to have a subset of the Cartesian product
of the references.
[para]
Correlation constraints are often used to enforce the referential
constraints for relvars that are related in a many to many fashion.
However, they also arise in situations where the multiplicity is not
many-to-many but when the [arg correlRelvar] contains attributes that
apply to the association between [arg refToRelvarA] and [arg refToRelvarB]
themselves.

[example {
% relvar correlation C1 OWNERSHIP OwnerName + OWNER OwnerName DogName * DOG DogName
}]

[call [cmd "::ral::relvar create"] [arg relvarName] [arg heading]\
    [arg id1] [opt [arg "id2 ..."]]]

The [cmd create] subcommand creates a new relation variable whose name is given
by [arg relvarName] and that will store a relation value whose
heading matches that of the [arg heading] argument.
The [arg heading] argument is a list with an even number of
elements consisting of alternating attribute names and attribute data types
as described for the [cmd ::ral::relation] command.
The tuples of the relation value will be identified by [arg id1] and
any other supplied [arg idN] arguments.
Each [arg idN] argument is a list of attribute names that appear in the
[arg heading].
The subcommand returns the relation value of the newly created relvar,
which necessarily will have cardinality of 0.
If [arg relvarName] name begins with "::", then it is interpreted
as a fully resolved name and will be created in the given namespace.
Otherwise, [arg relvarName] is interpreted as relative the to current namespace.
In either case,
an ordinary Tcl variable is also created by the same fully resolved
name as the relvar.
This implies that any namespaces referenced in [arg relvarName]
must already exist.

[example {
% relvar create OWNER {OwnerName string Age int City string} OwnerName
{OwnerName string Age int City string} {}
% puts [relformat $::OWNER]
+---------+---+------+
|OwnerName|Age|City  |
|string   |int|string|
+---------+---+------+
+---------+---+------+
% relvar create DOG {DogName string Breed string} DogName
{DogName string Breed string} {}
% relvar create OWNERSHIP {OwnerName string DogName string Acquired string}\
{OwnerName DogName}
{OwnerName string DogName string Acquired string} {}
}]

[call [cmd "::ral::relvar delete"] [arg relvarName]\
    [arg tupleVarName] [arg expression]]

The [cmd delete] subcommands deletes tuples from the relation variable
given by the [arg relvarName] argument.
Each tuple in the body of [arg relvarName] is successively
assigned to the tuple variable named [arg tupleVarName] and
[arg expression] is evaluated.
If [arg expression] returns true, then that tuple is deleted, directly
modifying the value contained in [arg relvarName].
The return value of the subcommand is the number of tuples deleted.

[example {
% puts [relformat $::OWNER "Before deleting Sue"]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
Before deleting Sue
-------------------
% relvar delete OWNER o {[tuple extract $o OwnerName] eq "Sue"}
1
% puts [relformat $::OWNER "After deleting Sue"]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
After deleting Sue
------------------
}]

[call [cmd "::ral::relvar deleteone"] [arg relvarName]\
    [opt "attr1 value1 attr2 value2 ..."]]

The [cmd deleteone] subcommand deletes at most one tuple from the
relation variable given by the [arg relvarName] argument.
The [arg "attrN valueN"] arguments must be given in pairs
and are the attribute names and values that determine which tuple
is deleted from the relation value held in [arg relvarName].
The set of attribute names in the [arg "attrN valueN"] arguments
must form one of the identifiers of [arg relvarName].
If [arg relvarName] contains a tuple whose attribute values match those
in given then that tuple is deleted and [arg relvarName]
is modified in place.
Otherwise [arg relvarName] is unchanged.
This subcommand is useful in those contexts where the attribute values of an
identifier are known and evaluating an expression over all the tuples
in [arg relvarName] is superfluous.
The return value of the subcommand is the number of tuples deleted which will
necessarily be either 1 or 0.

[example {
% puts [relformat $::OWNER "Before deleting Mike"]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
Before deleting Mike
--------------------
% relvar deleteone OWNER OwnerName Mike
1
% puts [relformat $::OWNER "After deleting Mike"]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
After deleting Mike
-------------------
% relvar deleteone OWNER OwnerName Alfonse
0
% puts [relformat $::OWNER]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
}]

[call [cmd "::ral::relvar dunion"] [arg relvarName]\
    [arg [opt "relationValue1 ..."]]]
The [cmd dunion] subcommand performs the relation disjoint union of the
relation value contained in [arg relvarName] with the given
[arg relationValueN] arguments and assigns the result back into
[arg relvarName].
This command is shorthand for:
[para]
relvar set <relvarName> [lb]relation dunion [lb]relvar set <relvarName>[rb]
<relationValue1> ...[rb]
[para]
The headings of each [arg relationValueN] must match that of the value
contained in [arg relvarName].
Any duplicated tuples will cause an error to be thrown and thus
this command has insert type semantics.
Since the disjoint union operation is commutative, the order of the
[arg relationValueN] arguments does not affect the result.
The return value of the command is the new relation value stored in
[arg relvarName].

[call [cmd "::ral::relvar eval"] [arg arg] [opt "arg ..."]]

The [cmd eval] subcommand concatenates it arguments together and evaluates
the resulting script as a relvar transaction.
The return value of [cmd eval] is the return value of the last
command executed in the script.
At the end of the script,
the constraints associated with any relvar that was modified
by the script are evaluated.
If either the script generates an error or any of the constraints fail,
then the values of the modified relvars are returned to the values
they held before the [cmd eval] command.
It is an error to attempt to create relvars or constraints in the
script executed by [cmd eval].

[example {
% puts [relformat $::OWNER]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
% relvar eval {
>       relvar insert OWNER {OwnerName Tom Age 22 City Tulsa}
> }
for association ::A1(::OWNERSHIP [+] ==> [1] ::OWNER), in relvar ::OWNER
tuple {OwnerName Tom Age 22 City Tulsa} is not referenced by any tuple

% puts [relformat $::OWNER]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
% relvar eval {
>       relvar insert OWNER {OwnerName Tom Age 22 City Tulsa}
>       relvar insert OWNERSHIP {OwnerName Tom DogName Skippy Acquired 2006}
> }
for association ::A2(::OWNERSHIP [*] ==> [1] ::DOG), in relvar ::OWNERSHIP
tuple {OwnerName Tom DogName Skippy Acquired 2006} references no tuple

% puts [relformat $::OWNER]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
% relvar eval {
>       relvar insert OWNER {OwnerName Tom Age 22 City Tulsa}
>       relvar insert OWNERSHIP {OwnerName Tom DogName Jumper Acquired 2006}
>       return
> }
% puts [relformat $::OWNER]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
|Tom      |22 |Tulsa        |
+---------+---+-------------+
}]

[call [cmd "::ral::relvar exists"] [arg relvarName]]
The [cmd exists] subcommand returns 1 if the relvar given by [arg relvarName]
exists and 0 otherwise.
The relvar name is resolved in the current namespace if it is not
specified as fully qualified.

[call [cmd "::ral::relvar identifiers"] [arg relvarName]]

[call [cmd "::ral::relvar insert"] [arg relvarName]\
    [opt "name-value-list ..."]]

The [cmd insert] subcommand inserts the tuples given by [arg name-value-list]
arguments into the value of the relation stored in [arg relvarName].
The value of [arg relvarName] is modified in place.
The return value is a relation value with the same heading as the
value held in [arg relvarName] and whose body contains the tuples as they
were actually inserted.
This gives the caller access to any attributes that might have been added
or modified by a relvar trace.
It is an error to attempt to insert a duplicate tuple or a tuple whose
heading does not match the heading of the value held in [arg relvarName].
The values of all attributes of the inserted tuples must be specified
either in the command invocation or as supplied by any relvar insert traces
with a valid value representation of the type associated with the attribute.
However, because of relvar tracing, the tuple values given in
[arg name-value-list] may not be the ones actually inserted into the relvar
and relvar traces may extend the given [arg name-value-list] argument.

[example {
% relvar insert DOG {DogName Skippy Breed Dalmation}
    {DogName string Breed string} {{DogName Fido Breed Poodle} \ 
    {DogName Sam Breed Collie} {DogName Spot Breed Terrier} \ 
    {DogName Rover Breed Retriever} {DogName Fred Breed Spaniel} \ 
    {DogName Jumper Breed Mutt} {DogName Skippy Breed Dalmation}}
% relvar insert OWNER {OwnerName Tom Age 22 City Tulsa}
for association ::A1(::OWNERSHIP [+] ==> [1] ::OWNER), in relvar ::OWNER
tuple {OwnerName Tom Age 22 City Tulsa} is not referenced by any tuple

}]

[call [cmd "::ral::relvar intersect"] [arg relvarName]\
    [opt "relationValue1 relationValue2 ..."]]

The [cmd intersect] subcommand performs the relation intersection of the
relation value contained in [arg relvarName] with the given
[arg relationValueN] arguments and assigns the result back into
[arg relvarName].
This command is shorthand for:
[para]
relvar set <relvarName> [lb]relation intersect [lb]relvar set <relvarName>[rb]
<relationValue1> ...[rb]
[para]
The headings of each [arg relationValueN] must match that of the value
contained in [arg relvarName].
Since the intersect operation is commutative, the order of the
[arg relationValueN] arguments does not affect the result.
The return value of the command is the new relation value stored in
[arg relvarName].

[call [cmd "::ral::relvar minus"] [arg relvarName] [arg relationValue]]

The [cmd minus] subcommand performs the relation difference of the
relation value contained in [arg relvarName] with
the [arg relationValue] argument and assigns the result back into
[arg relvarName].
This command is shorthand for:
[para]
relvar set <relvarName> [lb]relation minus [lb]relvar set <relvarName>[rb]
<relationValue1>[rb]
[para]
The headings of [arg relationValue] must match that of the value
contained in [arg relvarName].
Note that this operation is not commutative.
The return value of the command is the new relation value stored in
[arg relvarName].

[call [cmd "::ral::relvar names"] [opt pattern]]

The [cmd names] subcommand returns a list of the currently defined
relation variables.
If the [arg pattern] argument is specified then only those names matching
[arg pattern] are returned.
Name matching is performed as for the [cmd "string match"] command.
The returned names are fully qualified.

[example {
% relvar names
::DOG ::OWNERSHIP ::OWNER
% relvar names ::D*
::DOG
}]

[call [cmd "::ral::relvar partition"] [arg name] [arg super]\
	[arg superAttrList] [arg sub1] [arg sub1AttrList]\
	[opt "sub2 sub2AttrList sub3 sub3AttrList ..."]]

The [cmd partition] subcommand defines a partition constraint named
[arg name].
The relvar named [arg super] must have as one of its identifiers the
attributes named in the [arg superAttrList] argument.
The remaining arguments define the sub set relvars named [arg subN]
and the attributes in those relvars that reference the super set relvar
are given by the [arg subNAttrList] arguments.
A partition constraint declares that every tuple in the [arg subN]
relvars references exactly one tuple in the [arg super] relvar and that
every tuple in the [arg super] relvar is referred to by exactly one
tuple contained in one of the [arg subN] relvars.
This constraint implies that the tuples in the [arg super] relvar are
completely partitioned into the disjoint sub sets given by the the
[arg subN] relvars.

[example {
% relvar create Lamp {SerialNo string ModelNo string Make string} SerialNo
{SerialNo string ModelNo string Make string} {}
% relvar create TableLamp {SerialNo string Shade string} SerialNo
{SerialNo string Shade string} {}
% relvar create FloorLamp {SerialNo string Height int Sockets int} SerialNo
{SerialNo string Height int Sockets int} {}
% relvar partition P1 Lamp SerialNo TableLamp SerialNo FloorLamp SerialNo
% relvar insert Lamp {SerialNo NF100 ModelNo FCN-22 Make Falcon}
for partition ::P1(::Lamp is partitioned [::TableLamp | ::FloorLamp]), in relvar ::Lamp
tuple {SerialNo NF100 ModelNo FCN-22 Make Falcon} is not referred to by any tuple

% relvar eval {
>       relvar insert Lamp {SerialNo NF100 ModelNo FCN-22  Make Falcon}
>       relvar insert TableLamp {SerialNo NF100  Shade Blue}
>       return
>     }
% relvar insert FloorLamp {SerialNo NF100 Height 72 Sockets 3}
for partition ::P1(::Lamp is partitioned [::TableLamp | ::FloorLamp]), in relvar ::Lamp
tuple {SerialNo NF100 ModelNo FCN-22 Make Falcon} is referred to by multiple tuples

% relvar insert FloorLamp {SerialNo NF101 Height 72 Sockets 3}
for partition ::P1(::Lamp is partitioned [::TableLamp | ::FloorLamp]), in relvar ::FloorLamp
tuple {SerialNo NF101 Height 72 Sockets 3} references no tuple

}]

[call [cmd "::ral::relvar path"] [arg relvarName]]
The [cmd path] subcommand returns the fully qualified relvar
name of [arg relvarName].

[call [cmd "::ral::relvar procedural"] [arg constraintName]\
    [arg relvarName] [arg [opt "relvarName2 ..."]] [arg script]]
The [cmd procedural] subcommand defines a procedural constraint.
The name of the constraint is given by the [arg constraintName] argument.
At the end of any transaction (or any command that modifies a relvar
if there is no ongoing transaction),
[arg script] is evaluated if any of the [arg relvarNameN] relvars is
modified.
The [arg script] must return a value that can be intepreted as a boolean.
If [arg script] returns [emph true] then the constraint passes.
If [arg script] returns [emph false] then the constraint is deemed to
have failed an the transaction is rolled back.
Procedural constraints are only evaluated at the end of a transaction
and only if the relvars given in the [cmd procedural] command are
modified during the transaction.
Compare this to the behavior of [cmd "relvar trace"] procedures which
are invoked on a tuple-by-tuple basis as the relvar value is being modified.
[para]
The following example shows a constraint that insures that the
Percentage attribute always totals to 100.
[example {
% relvar create Ingredient {Name string   Percentage int} Name
{Name string Percentage int} {}
% relvar insert Ingredient {Name flour Percentage 100}
{Name string Percentage int} {{Name flour Percentage 100}}
% relvar procedural P2 Ingredient {
> set tot [relation summarizeby [relvar set Ingredient] {}\\
> ig TotalPercent int {rsum($ig, "Percentage")}]
> return [expr {[relation extract $tot TotalPercent] == 100}]
> }
% relvar eval {
> relvar insert Ingredient {Name butter Percentage 25}
> relvar updateone Ingredient ig {Name flour} {
> tuple update $ig Percentage 75}
> }
% relformat $::Ingredient Ingredient
+------+----------+
|Name  |Percentage|
|string|int       |
+------+----------+
|flour |75        |
|butter|25        |
+------+----------+
Ingredient
----------
% relvar insert Ingredient {Name salt Percentage 1}
procedural contraint, "::P2", failed
% relformat $::Ingredient Ingredient
+------+----------+
|Name  |Percentage|
|string|int       |
+------+----------+
|flour |75        |
|butter|25        |
+------+----------+
Ingredient
----------
}]
    

[call [cmd "::ral::relvar restrictone"] [arg relvarName]\
    [arg attr1] [arg value1] [opt [arg "attr2 value2 ..."]]]
The [cmd restrictone] subcommand returns a relation value containing
the tuple whose attributes match those of the [arg attrN] and [arg valueN]
arguments.
The [arg attrN] arguments must form an identifier for the relvar.
The returned relation value will be of cardinality 0 or 1.
The relation value held in [arg relvarName] is not modified.

[call [cmd "::ral::relvar set"] [arg relvarName] [opt relationValue]]

The [cmd set] subcommand replaces the current value held by the relation
variable named [arg relvarName] with the value given by
[arg relationValue].
It is an error to attempt to assign a relation value to a relation variable
that is of a different type than the type of the value
that the variable currently holds.
In other words, it is not possible to change the type of a relvar
by assignment.
The return value of the subcommand is the current value
held by [arg relvarName].
If the [arg relationValue] argument is missing, then no attempt is made to
change the value of [arg relvarName].
This command operates in a manner analogous to the [cmd set] command for
ordinary Tcl variables.

[example {
% relvar create PUPPY {PuppyName string Dame string Sire string} PuppyName
{PuppyName string Dame string Sire string} {}
% relvar set PUPPY {
>   {PuppyName string Dame string Sire string} {
>       {PuppyName Bitsy Dame Spot Sire Jumper}
>   }
>}
{PuppyName string Dame string Sire string} {{PuppyName Bitsy Dame Spot Sire Jumper}}
% relvar set PUPPY $::DOG
headings not equal, "{DogName string Breed string}"
}]

[call [cmd "::ral::relvar trace"] [arg option] [arg type] [opt [arg args]]]
[list_begin definitions]
[def "[cmd {::ral::relvar trace add variable}] [arg relvarName]\
    [arg traceops] [arg cmdPrefix]"]
[def "[cmd {::ral::relvar trace remove variable}] [arg relvarName]\
    [arg traceops] [arg cmdPrefix]"]
[def "[cmd {::ral::relvar trace info variable}] [arg relvarName]"]
[def "[cmd {::ral::relvar trace suspend variable}] [arg relvarName]\
        [arg script]"]
[def "[cmd {::ral::relvar trace add eval}] [arg cmdPrefix]"]
[def "[cmd {::ral::relvar trace remove eval}] [arg cmdPrefix]"]
[def "[cmd {::ral::relvar trace info eval}]"]
[list_end]

The [cmd "trace add variable"] subcommand adds a trace to [arg relvarName].
The [arg traceops] argument is a list of one or more trace operation keywords
from the list:
[list_begin definitions]
[def delete]
[def insert]
[def set]
[def unset]
[def update]
[list_end]

The [cmd "trace remove variable"] subcommand deletes any trace on
[arg relvarName] that matches [arg traceops] and [arg command].

The [cmd "trace info variable"] subcommand returns the trace information
for the relvar whose name is given by [arg relvarName].
The trace information is a list each element of which describes one trace
on the relvar.
Each trace information element is in turn a list consisting of two elements,
the [arg traceops] and [arg command], of the relvar trace.

It is an error to request any trace operation if [arg relvarName] does not
exist.

The [cmd "trace suspend variable"] subcommand suspends tracing on
the relvar given by [arg relvarName] and executes [arg script] restoring
tracing after [arg script] ends.
This subcommand allows scripts to temporarily suspend tracing on a
relvar.

The [cmd "trace add eval"], [cmd "trace remove eval"] and
[cmd "trace info eval"] variations perform the corresponding
operation for tracing transactions via the [cmd "relvar eval"] command.

See the [emph "RELVAR TRACES"] section below for further details.

[call [cmd "::ral::relvar transaction"] [arg option]]
[list_begin definitions]
[def "[cmd {::ral::relvar transaction begin}]"]
[def "[cmd {::ral::relvar transaction end}]"]
[def "[cmd {::ral::relvar transaction rollback}]"]
[list_end]
The [cmd transaction] subcommand controls the evaluation of relvar
constraints.
Like [cmd "relvar eval"], [cmd "relvar transaction"] demarcates the
boundaries of when relvar constraints are evaluated.
[cmd "relvar transaction begin"] signals the beginning of a constraint
transaction.
When the matching [cmd "relvar transaction end"] is executed,
all contraints for all relvars modified since the corresponding
[cmd "relvar transaction begin"] command are evaluated.
Executing the [cmd "relvar transaction rollback"] command, halts the
transaction and discards all the modifications, restoring the relvar
state to that which existed before the execution of
[cmd "relvar transaction begin"].
This command is complementary to [cmd "relvar eval"] in that it provides
the same behavior but for those circumstances where accumulating the
relvar modifications into a single script is not convenient or possible.
The command returns the empty string.
If relvar modifications are such that the constraints are violated,
then [cmd "relvar transaction end"] will throw an error.

[call [cmd "::ral::relvar uinsert"] [arg relvarName]\
    [opt "name-value-list ..."]]

The [cmd uinsert] subcommand inserts the tuples given by [arg name-value-list]
arguments into the value of the relation stored in [arg relvarName].
The value of [arg relvarName] is modified in place.
The return value is a relation value with the same heading as the
value held in [arg relvarName] and whose body contains the tuples as they
were actually inserted.
This gives the caller access to any attributes that might have been added
or modified by a relvar trace.
The [cmd uinsert] command has union-like semantics and so any
attempt to insert a duplicate tuple is silently ignored.
It is an error to attempt to insert a tuple whose
heading does not match the heading of the value held in [arg relvarName].
The values of all attributes of the inserted tuples must be specified
either in the command invocation or as supplied by any relvar insert traces
with a valid value representation of the type associated with the attribute.
However, because of relvar tracing, the tuple values given in
[arg name-value-list] may not be the ones actually inserted into the relvar
and relvar traces may extend the given [arg name-value-list] argument.

[call [cmd "::ral::relvar union"] [arg relvarName]\
    [opt "relationValue1 relationValue2 ..."]]

The [cmd union] subcommand performs the relation union of the
relation value contained in [arg relvarName] with the given
[arg relationValueN] arguments and assigns the result back into
[arg relvarName].
This command is shorthand for:
[para]
relvar set <relvarName> [lb]relation union [lb]relvar set <relvarName>[rb]
<relationValue1> ...[rb]
[para]
The headings of each [arg relationValueN] must match that of the value
contained in [arg relvarName].
Since the union operation is commutative, the order of the
[arg relationValueN] arguments does not affect the result.
The return value of the command is the new relation value stored in
[arg relvarName].

[call [cmd "::ral::relvar unset"] [opt "relvarName ..."]]

The [cmd unset] subcommand deletes all the relation variables whose
names are given as the subcommand arguments.
The corresponding Tcl variables that were created when the relvar
was created are also [cmd unset].
This command operates in a manner analogous to the [cmd unset] command for
ordinary Tcl variables.
It is an error to attempt to [cmd unset] a relvar that is associated
with any constraints.

[call [cmd "::ral::relvar update"] [arg relvarName] [arg tupleVarName]\
    [arg expression] [arg script]]

The [cmd update] subcommand modifies the values of tuples in the relation
value contained in the relation variable given by [arg relvarName].
Each tuple in [arg relvarName] is successively assigned to the tuple
variable given by [arg tupleVarName] and [arg expression] is evaluated.
If the result of the evaluation is true,
then [arg script] is evaluated.
The return value of [arg script] is then used to update the corresponding
tuple in [arg relvarName].
The return value of the [cmd "relvar update"] subcommand
is a relation value with the same heading as the
value held in [arg relvarName] and whose body contains the tuples as they
were actually updated into the relvar.
This gives the caller access to any attributes that might have been modified
by a relvar trace.
If the [arg script] throws an error or the modified value of a tuple cannot
be updated into [arg relvarName], all changes to [arg relvarName] are
discarded and the value of [arg relvarName] is restored to its previous value.
Should [arg script] execute a [cmd break] or otherwise arrange to
return a break result code,
then the tuple is updated, but no further tuples in the relvar are considered.

[example {
% puts [relformat $::OWNER]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
% relvar update OWNER o {[tuple extract $o OwnerName] eq "Sue"} {
>       tuple update $o Age [expr {[tuple extract $o Age] + 1}]
>     }
% puts [relformat $::OWNER]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |25 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
% relvar update OWNER o {[tuple extract $o OwnerName] eq "George"} {
>       tuple update $o OwnerName Alfonse
>     }
for association ::A1(::OWNERSHIP [+] ==> [1] ::OWNER), in relvar ::OWNER
tuple {OwnerName Alfonse Age 35 City Sunnyvale} is not referenced by any tuple
for association ::A1(::OWNERSHIP [+] ==> [1] ::OWNER), in relvar ::OWNERSHIP
tuple {OwnerName George DogName Fido Acquired 2001} references no tuple
tuple {OwnerName George DogName Sam Acquired 2000} references no tuple

% puts [relformat $::OWNER]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |25 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
}]

[call [cmd "::ral::relvar updateone"] [arg relvarName] [arg tupleVarName]\
    [arg id-name-value-list] [arg script]]

The [cmd updateone] subcommand modifies at most one tuple from the
relation variable given by the [arg relvarName] argument.
The [arg id-name-value-list] must be list containing an even number of elements
which are attribute names alternating with attribute values.
The set of attribute names in [arg id-name-value-list] must form one of the
identifiers of [arg relvarName].
If [arg relvarName] contains a tuple whose attribute values match those
in [arg name-value-list] then
that tuple is assigned to [arg tupleVarName] and [arg script] is executed.
The tuple value returned by [arg script]
replaces the original tuple, modifying [arg relvarName] in place.
If no matching tuple is found, then [arg relvarName] is unchanged.
This subcommand is useful in those contexts where the attribute values of an
identifier are known and evaluating an expression over all the tuples
in [arg relvarName] is superfluous.
The return value is a relation value with the same heading as the
value held in [arg relvarName] and whose body contains either the single
tuple that was updated or is empty if no matching tuple was found.
This gives the caller access to any attributes that might have been modified
by a relvar trace.

[example {
% puts [relformat $::OWNER]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |35 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
% relvar updateone OWNER o {OwnerName George} {
> tuple update $o Age 37 City {New York}
> }
% puts [relformat $::OWNER]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |24 |Cupertino    |
|George   |37 |New York     |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |42 |San Francisco|
+---------+---+-------------+
}]

[call [cmd "::ral::relvar updateper"] [arg relvarName] [arg relationValue]]
The [cmd updateper] subcommand modifies the relation value held in
[arg relvarName] according to the tuples found in [arg relationValue].
The heading of [arg relationValue] must be a subset (possibly improper)
of the heading of the value held in [arg relvarName].
The heading of [arg relationValue] must also contain all of the attributes
of at least one of the identifiers of [arg relvarName].
The action of [cmd updateper] is to update the non-identifying attributes of
all the tuples in [arg relvarName] whose identifying attributes are equal to
the corresponding attributes in [arg relationValue].
The return value of the command is a relation value with the same heading as
the value held in [arg relvarName] and whose body contains tuples that were
updated.
This gives the caller access to any attributes that might have been modified
by a relvar trace.
This command is most useful when a subset of tuples from a relvar
are modified by a series of relational expressions and the results of those
modification need to be updated back into the original relvar.

[example {
% relvar updateper OWNER [relation table {OwnerName string Age int} {
        George 37} {
        Jim 44}]
% puts [relformat $::OWNER]
+---------+---+-------------+
|OwnerName|Age|City         |
|string   |int|string       |
+---------+---+-------------+
|Sue      |25 |Cupertino    |
|George   |37 |Sunnyvale    |
|Alice    |30 |San Jose     |
|Mike     |50 |San Jose     |
|Jim      |44 |San Francisco|
+---------+---+-------------+
}]

[list_end]

[section "RELVAR TRACES"]

By analogy to tracing ordinary variables in Tcl,
relvars can also be traced.
However, relvar tracing must account for the
operations that are applied to relvars.
There are five operations that may be traced on the relation values
held in relvars and one operation that allows tracing of transactions
on relvars.
For these operations, if a command prefix has been registered using
[cmd "relvar trace add variable"] or [cmd "relvar trace add transaction"]
then that command prefix will have
additional arguments appended and will be invoked as described below.
Relvar traces operate in a very similar fashion as traces on ordinary
Tcl variables.
The trace command is invoked when the corresponding relvar operation
is invoked and is invoked in the context of the caller of the relvar operation,
which, in general, is different from the caller of the [cmd "relvar trace add"]
command.
When executing a variable type trace for a relvar,
all other variable type tracing for that relvar is inhibited.
If multiple traces have been added to a relvar,
the traces are invoked in the order of most recent first to oldest last.

[list_begin definitions]
[def delete]
A delete trace is invoked for each tuple in a relvar that is deleted.
Tuples are deleted from relvars by invocations of [cmd "relvar delete"]
or [cmd "relvar deleteone"].
The delete trace command is invoked as:
[para]
[cmd cmdPrefix] [cmd delete] [arg relvarName] [arg tupleValue]
[para]
where [cmd cmdPrefix] is the command prefix of the trace as given
in [cmd "relvar trace add variable"],
the string [cmd delete] indicates that a tuple is to be deleted from the relvar,
[arg relvarName] is the fully qualified name of the relvar
and [arg tupleValue] is the value of the tuple about to be deleted.
The trace command is invoked before the tuple is actually removed from
the relation value held in the relvar.
If [cmd cmdPrefix] throws an error, then no other delete traces in the chain of
delete traces for the relvar are invoked and [arg tupleValue] is not removed
from the relvar value.

[def insert]
An insert trace is invoked for each tuple in a relvar that is inserted.
Tuples are inserted into relvars by invocations of [cmd "relvar insert"].
The insert trace command is invoked as:
[para]
[cmd cmdPrefix] [cmd insert] [arg relvarName] [arg tupleValue]
[para]
where [cmd cmdPrefix] is the command prefix of the trace as given
in [cmd "relvar trace add variable"],
the string [cmd insert] indicates that a tuple
is to be inserted into the relvar,
[arg relvarName] is the fully qualified name of the relvar
and [arg tupleValue] is the value of the tuple about to be inserted
The trace command is invoked before the tuple is actually inserted into
the relation value held in the relvar.
The [cmd cmdPrefix] command must return a Tuple value.
The returned Tuple value is passed along to the next insert trace command.
The returned Tuple from the last trace command in the chain is then
inserted into the relvar.
Insert trace commands may choose to modify the value of [arg tupleValue] but
are required to return a value that can be coerced into a Tuple type and the
heading of the returned Tuple of the last trace command in the chain
must match that of the relvar if it is to be successfully inserted.
If [cmd cmdPrefix] throws an error, then no other insert traces in the chain of
insert traces for the relvar are invoked and [arg tupleValue] is not inserted
into the relvar value.

[def set]
A set trace is invoked each time that the relation value held in a relvar
is assigned.
Relations values are assigned to relvars by invocation of the
[cmd "relvar set"],
[cmd "relvar union"],
[cmd "relvar intersect"]
or
[cmd "relvar minus"]
command.
The set trace command is invoked as:
[para]
[cmd cmdPrefix] [cmd set] [arg relvarName] [arg relationValue]
[para]
where [cmd cmdPrefix] is the command prefix of the trace as given
in [cmd "relvar trace add variable"],
the string [cmd set] indicates that a relation value
is to be assigned to the relvar,
[arg relvarName] is the fully qualified
name of the relvar
and [arg relationValue] is the relation value that is about to be assigned
to the relvar.
The trace command is invoked before the relation is actually assigned to the
relvar.
The [cmd cmdPrefix] command must return a Relation value.
The returned Relation value is passed along to the next set trace command.
The returned Relation from the last trace command in the chain is then
assigned to the relvar.
Set trace commands may choose to modify the value of [arg relationValue] but
are required to return a value that can be coerced into a Relation type and the
heading of the returned Relation of the last trace command in the chain
must match that of the relvar if it is to be successfully assigned
If [cmd cmdPrefix] throws an error, then no other set traces in the chain of
set traces for the relvar are invoked and [arg relationValue] is not assigned
to the relvar.

[def unset]
An unset trace is invoked each time a relvar is destroyed.
Relvars are destroyed by the invocation of the [cmd "relvar unset"] command.
Note that due to a limitation of Tcl,
relvar unset traces are not invoked when an interpreter is deleted.
The unset trace command is invoked as:
[para]
[cmd cmdPrefix] [cmd unset] [arg relvarName]
[para]
where [cmd cmdPrefix] is the command prefix of the trace as given
in [cmd "relvar trace add variable"],
the string [cmd unset] indicates that the relvar
is being destroyed,
and [arg relvarName] is the fully qualified
name of the relvar.
The trace command is invoked before the relvar is actually destroyed so
the relation value stored in the relvar is still intact.
Any errors thown by [cmd cmdPrefix] are ignored.

[def update]
An update trace is invoked for each tuple in a relvar that is updated.
Tuples are updated into relvars by invocations of [cmd "relvar update"],
[cmd "relvar updateone"] or [cmd "relvar updateper"] commands.
The update trace command is invoked as:
[para]
[cmd cmdPrefix] [cmd update] [arg relvarName] [arg oldTupleValue]
    [arg newTupleValue]
[para]
where [cmd cmdPrefix] is the command prefix of the trace as given
in [cmd "relvar trace add variable"],
the string [cmd update] indicates that a tuple
is to be updated in the relvar,
[arg relvarName] is the fully qualified name of the relvar,
[arg oldTupleValue] is the current value of the tuple in the relvar
and [arg newTupleValue] is the value that the tuple will have when
the update is performed.
The trace command is invoked before the tuple is actually updated in
the relation value held in the relvar.
The [cmd cmdPrefix] command must return a Tuple value.
The returned Tuple value is passed along to the next update trace command.
The returned Tuple from the last trace command in the chain is then
updated into the relvar.
Update trace commands may choose to modify the value of [arg newTupleValue] but
are required to return a value that can be coerced into a Tuple type and the
heading of the returned Tuple of the last trace command in the chain
must match that of the relvar if it is to be successfully updated.
If [cmd cmdPrefix] throws an error, then no other update traces in the chain of
update traces for the relvar are invoked and [arg tupleValue] is not updated
into the relvar value.

[def transaction]
A transaction trace is invoked each time that a [cmd "relvar eval"] begins and
ends or when [cmd "relvar transaction"] commands are invoked.
Eval traces are different from other relvar traces in that they provide a
means of determining the boundaries of a transaction rather than specific
operations on a relvar.
The transaction trace command is invoked as:
[para]
[cmd cmdPrefix] [cmd transaction] [arg op] [arg level]
[para]
where [cmd cmdPrefix] is the command prefix of the trace as given
in [cmd "relvar trace add transaction"], 
the string [cmd transaction] indicates that the relvar transaction is in
progress, the [arg op] is either the string "begin" or "end" indicating whether
or not the [cmd "relvar transaction"] is just starting or is ending and
[arg level] is an integer greater than zero indicating the nesting depth
of the transaction.
The transaction trace is invoked with the "begin" op before the script
associated with a [cmd "relvar eval"] or [cmd "relvar transaction"]
is executed and
is invoked with the "end" op after the script is executed but before the relvar
constraints are evaluated for the transaction.
Any errors thown by [cmd cmdPrefix] are ignored.

[list_end]

[see_also tuple relation]

[keywords relation variable]

[manpage_end]
